using System;
using System.IO;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

using ircam.jmax.fts;
using ircam.jmax.dialogs;
using ircam.jmax.editors.patcher;

namespace ircam.jmax.editors.project
{
	public partial class ProjectEditor: Form
	{
        private FtsPackage ftsPkg;

        public ProjectEditor(FtsPackage pkg)
		{
			InitializeComponent();

            ftsPkg = pkg;
            pkg.Listener = this;
            InitDataModels(pkg);

            this.Text += " - " + pkg.FileName;

            if ((pkg is FtsProject) && ((FtsProject)pkg).Config != null)
                this.configPanelTextBox.Text = ((FtsProject)pkg).Config;
		}

        private void InitDataModels(FtsPackage pkg)
        {
            if (pkg == null) return;

            pkg.Requires.ForEach(
                delegate(string s)
                {
                    this.requiresCheckedListBox.Items.Add(s, true);
                });

            pkg.PackagePaths.ForEach(
                delegate(string s)
                {
                    this.requiresCheckedListBox.Items.Add(s, true);
                });

            pkg.DataPaths.ForEach(
                delegate(string s)
                {
                    this.requiresCheckedListBox.Items.Add(s, true);
                });

            //FtsPackage.TwoNames tn;
            templatesListView.View = View.Details;
            templatesListView.LabelEdit = false;
            templatesListView.FullRowSelect = true;
            templatesListView.Columns.Add("template name", -2, HorizontalAlignment.Left);
            templatesListView.Columns.Add("file name", -2, HorizontalAlignment.Left);

            pkg.Templates.ForEach(
                delegate(FtsPackage.TwoNames tn)
                {
                    ListViewItem item = new ListViewItem(tn.name);
                    item.SubItems.Add(tn.fileName);

                    templatesListView.Items.AddRange(new ListViewItem[] { item });
                });

            pkg.TemplatePaths.ForEach(
                delegate(string s)
                {
                    this.requiresCheckedListBox.Items.Add(s, true);
                });

            helpsListView.View = View.Details;
            helpsListView.LabelEdit = false;
            helpsListView.FullRowSelect = true;
            helpsListView.Columns.Add("class name", -2, HorizontalAlignment.Left);
            helpsListView.Columns.Add("file name", -2, HorizontalAlignment.Left);

            pkg.Helps.ForEach(
                delegate(FtsPackage.TwoNames tn)
                {
                    ListViewItem item = new ListViewItem(tn.name);
                    item.SubItems.Add(tn.fileName);

                    helpsListView.Items.AddRange(new ListViewItem[] { item });
                });
        }

        public FtsPackage Package
        {
            set
            {
                if (ftsPkg == value)
                    return;

                ftsPkg = value;
                ftsPkg.Listener = this;
            }
        }

        //        The keyword new is required on 'ircam.jmax.editors.project.ProjectEditor.Update()' 
        //        because it hides inherited member 'System.Windows.Forms.Control.Update()'
        //        public void Update()
        //        {
        //            InitDataModels(ftsPkg);
        //        }

        public void UpdateDone()
        {
            InitDataModels(ftsPkg);
        }

        public void PackagePathChanged()
        {
            ftsPkg.PackagePaths.ForEach(
                delegate(string s)
                {
                    this.packagePathsListBox.Items.Add(s);
                });
        }

        public void DataPathChanged()
        {
            ftsPkg.DataPaths.ForEach(
                delegate(string s)
                {
                    this.dataPathsListBox.Items.Add(s);
                });
        }

        public void TemplatePathChanged()
        {
            ftsPkg.TemplatePaths.ForEach(
                delegate(string s)
                {
                    this.templatePathsListBox.Items.Add(s);
                });
        }

        public void TemplateChanged()
        {
            ftsPkg.Templates.ForEach(
                delegate(FtsPackage.TwoNames tn)
                {
                    ListViewItem item = new ListViewItem(tn.name);
                    item.SubItems.Add(tn.fileName);

                    templatesListView.Items.AddRange(new ListViewItem[] { item });
                });
        }

        public void HelpChanged()
        {
            ftsPkg.Helps.ForEach(
                delegate(FtsPackage.TwoNames tn)
                {
                    ListViewItem item = new ListViewItem(tn.name);
                    item.SubItems.Add(tn.fileName);

                    helpsListView.Items.AddRange(new ListViewItem[] { item });
                });
        }

        public void SetErrorAt(int index)
        {
            if (index > this.requiresCheckedListBox.Items.Count)
                return;
            this.requiresCheckedListBox.SetItemChecked(index, false);
        }

        public void ConfigChanged(string config)
        {
            if (config != null)
            {
                this.configPanelTextBox.Text = config;
            }
            else
            {
                this.configPanelTextBox.Text = "";
            }
        }

        public static void NewProject()
        {
            NewProjectDialog npd = new NewProjectDialog();
            npd.ShowDialog();
        }

        public static void OpenProject()
        {
            OpenFileDialog ofd = new OpenFileDialog();
            ofd.Filter = "Project files (*.jprj)|*.jprj|All files (*.*)|*.*";

            if (ofd.ShowDialog() == DialogResult.OK)
            {
                string fileName = ofd.FileName;

                if (fileName != null)
                {
                    CloseWindowsAndSave();

                    try
                    {
                        JMaxLibrary.LoadProject(fileName);
                    }
                    catch (IOException e)
                    {
                        Console.Error.WriteLine("[ProjectEditor]: I/O error loading project " + fileName);
                    }
                }
            }
        }

        public static void CloseWindowsAndSave()
        {
            Form win;
            FtsPatcherObject patcher;
            List<FtsPatcherObject> patchs = new List<FtsPatcherObject>();
            object[] windows = MaxWindowManager.WindowManager.WindowListArrayCopy;
            bool someOneNeedSave = false;
            bool somePatcherIsOpen = false;

            // First, search if there is anything to save
        // Loop in all the documents in all the  types.

            search: for (int i = 0; i < windows.Length; i++)
            {
                win = (Form)windows[i];
                if (win is ErmesSketchWindow)
                {
                    patcher = ((ErmesSketchWindow)win).SketchPad.FtsPatcher;
                    somePatcherIsOpen = true;

                    if (patcher.IsDirty)
                    {
                        someOneNeedSave = true;
                        goto search;
                    }
                }
            }

            DialogResult result = DialogResult.No;
            if (someOneNeedSave)
            {
                /* ask if has to save windows */
                result = MessageBox.Show("Closing Project!\nAll open patchers will be closed. \nDo you want save the unsaved patchers?",
                    "Closing Project",
                    MessageBoxButtons.YesNo,
                    MessageBoxIcon.Question,
                    MessageBoxDefaultButton.Button1);
            }
            if (somePatcherIsOpen)
            {
                /* close all windows and save if needed */
                for (int i = 0; i < windows.Length; i++)
                {
                    win = (Form)windows[i];
                    if (win is ErmesSketchWindow)
                    {
                        patcher = ((ErmesSketchWindow)win).SketchPad.FtsPatcher;

                        if (patcher.IsARootPatcher)
                        {
                            if ((result == DialogResult.Yes) && patcher.IsDirty)
                            {
                                if (patcher.CanSave)
                                    patcher.Save();
                                else
                                {
                                    FileInfo file = new FileInfo(MaxFileChooser.ChooseFileToSave(null, null, "Save As", MaxFileChooser.JMAX_FILE_TYPE));

                                    if (file != null)
                                    {
                                        if (file.Exists)
                                        {
                                            DialogResult res = MessageBox.Show("File \"" + file.Name + "\" exists.\nOK to overwrite ?",
                                                "Warning",
                                                MessageBoxButtons.OKCancel,
                                                MessageBoxIcon.Warning,
                                                MessageBoxDefaultButton.Button2);

                                            if (res == DialogResult.Yes)
                                                patcher.Save(MaxFileChooser.SaveType, file.FullName);
                                        }
                                        else
                                            patcher.Save(MaxFileChooser.SaveType, file.FullName);
                                    }
                                }
                            }

                            if (patcher.Name != null)
                                patchs.Add(patcher);

                            patcher.StopUpdates();
                            patcher.RequestDestroyEditor();
                            JMaxLibrary.RootPatcher.RequestDeleteObject(patcher);
                            ((ErmesSketchWindow)win).Dispose();
                        }
                    }
                }
            }
            /* save the open windows in the project */
            JMaxLibrary.Project.SaveWindows(patchs.GetEnumerator());

            JMaxLibrary.Project.Save(null);
        }

        public static void SaveWindows()
        {
            List<Form> windows = MaxWindowManager.WindowManager.WindowList;
            List<FtsPatcherObject> patchs = new List<FtsPatcherObject>();
            Form win;
            FtsPatcherObject patcher;

            for (int i = 0; i < windows.Count; i++)
            {
                win = (Form)windows[i];
                if (win is ErmesSketchWindow)
                {
                    patcher = ((ErmesSketchWindow)win).SketchPad.FtsPatcher;
                    if (patcher.Name != null)
                        patchs.Add(patcher);
                }
            }
            JMaxLibrary.Project.SaveWindows(patchs.GetEnumerator());
        }

        private void requiresNewButton_Click(object sender, System.EventArgs e)
        {
            //            this.requiresCheckedListBox.Items.Add("test", false);
            string[] s = { "test" };
            //            s[0] = "test";
            ftsPkg.SetMessage("require", s);
            //            string s[0] = new string("test");
            //            s[0] = "test";

            //            if (ftsPkg != null)
            //            {
            //                if (data[row][0] != null)
            //                    ftsPkg.SetMessage("require", s);
            //            }
            //            this.UpdateDone();
        }
	}
}