//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;

using ircam.jmax;
using ircam.fts.client;
using ircam.jmax.editors.patcher.objects;

namespace ircam.jmax.fts
{
    /// <summary> The proxy of an Fts connection.
    /// Deals with the connection creation/deletion and saving/loading.
    /// </summary>
    public class FtsConnection : FtsObject
    {
        private event FtsConnectionListener listener;

        /* numbering doubled on server side (connection.h) */
        public const int fts_connection_invalid = -1;        /* from error object or type missmatch */
        public const int fts_connection_anything = 0;        /* message which is not one of the following */
        public const int fts_connection_message = 1;        /* message */
        public const int fts_connection_value = 2;            /* varargs (unpacked tuple) or single value */
        public const int fts_connection_audio = 3;            /* signal connection */
        public const int fts_connection_audio_active = 4;    /* active signal connection */

        [NonSerialized()]
        private int id;

        [NonSerialized()]
        private bool deleted = false;

        private FtsGraphicObject from;
        private int outlet;

        private FtsGraphicObject to;
        private int inlet;

        private GraphicConnection graphicConnection;

        private int type;

        /// <summary>Create a FTS connection instance.
        /// The FTS side of the connection is created in the Fts class.
        /// </summary>
        /// <seealso cref="ircam.jmax.fts.Fts#makeFtsConnection">
        /// </seealso>
        internal FtsConnection(FtsServer server, FtsPatcherObject patcher, int id, FtsGraphicObject from, int outlet, FtsGraphicObject to, int inlet, int type)
            : base(server, patcher, id)
        {
            this.from = from;
            this.outlet = outlet;
            this.to = to;
            this.inlet = inlet;
            this.type = type;
        }

        internal FtsConnection(FtsServer server, FtsPatcherObject patcher, int id, FtsGraphicObject from, int outlet, FtsGraphicObject to, int inlet)
            : this(server, patcher, id, from, outlet, to, inlet, fts_connection_anything)
        {
        }

        static FtsConnection()
        {
            FtsObject.RegisterMessageHandler(typeof(FtsConnection), FtsSymbol.Get("setType"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsConnection)obj).Type = (args.GetInt(0));
                });
        }

        /// <summary>Get the current object listener </summary>
        /// <summary>Set the unique object listener </summary>
        public GraphicConnection ConnectionListener
        {
            get
            {
                return graphicConnection;
            }
            set
            {
                graphicConnection = value;

                listener += new FtsConnectionListener(value.TypeChanged);
            }
        }

        public int Type
        {
            get
            {
                return type;
            }
            set
            {
                if (value != this.type)
                {
                    this.type = value;

                    if (listener != null)
                        listener(graphicConnection, new FtsConnectionListenerArgs(value));
                }
            }
        }

        /// <summary> Get the fts connection id. <p></summary>
        /// <summary>Set the objid. Private, used only by the server. </summary>
        internal int ConnectionId
        {
            get
            {
                return id;
            }
            set
            {
                this.id = value;
            }
        }

        /// <summary>Ask FTS to Undo the connection. </summary>
        public void Delete()
        {
            if (deleted)
                return;

            deleted = true;

            ircam.jmax.editors.patcher.ErmesSelection.FtsSelection.Remove(this);

            ((FtsPatcherObject)from.Parent).RequestDeleteConnection(this);
        }

        /// <summary>Undo the connection, only the client part.
        /// indirectly called by FTS.
        /// </summary>
        public void Release()
        {
            deleted = true;

            if ((from.Parent != null) && (from.Parent is FtsPatcherObject))
                ((FtsPatcherObject)from.Parent).RemoveConnection(this);

            // Clean up

            from = null;
            to = null;
        }

        /// <summary>Access the From. The From is the FtsObject origin of the connection. </summary>
        public FtsGraphicObject From
        {
            get
            {
                return from;
            }
        }

        /// <summary>Access the To. The To is the FtsObject destination of the connection. </summary>
        public FtsGraphicObject To
        {
            get
            {
                return to;
            }
        }

        /// <summary>Access the From outlet. </summary>
        public int FromOutlet
        {
            get
            {
                return outlet;
            }
        }

        /// <summary>Access the To inlet. </summary>
        public int ToInlet
        {
            get
            {
                return inlet;
            }
        }

        /// <summary>Get a string debug representation for the connection </summary>
        public override string ToString()
        {
            return "FtsConnection(" + from + "," + outlet + "," + to + "," + inlet + ", #" + id + ", <" + type + ">)";
        }
    }
}