//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.IO;

using ircam.fts.client;
using ircam.jmax;

namespace ircam.jmax.fts
{
    /// <summary>Java class for the dsp control remote data class.
    /// 
    /// It offer visibility and control over a number of DSP related
    /// server parameter and status.
    /// It is implemented as a Java Beans, with Java Beans Properties 
    /// corresponding to DSP server properties.
    /// </summary>

    public class FtsDspControl : FtsObject
    {
        private int dacSlip;
        private int invalidFpe;
        private int divideByZeroFpe;
        private int overflowFpe;
        private int denormalizedFpe;
        private int samplingRate;
        private int fifoSize;
        private bool dspOn;

        internal bool atomic = false;

        protected internal FtsArgs args = new FtsArgs();

        static FtsDspControl()
        {
            FtsObject.RegisterMessageHandler(typeof(FtsDspControl), FtsSymbol.Get("setDspOnState"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsDspControl)obj).DspOn = (args.GetInt(0) == 0) ? false : true;
                });
            FtsObject.RegisterMessageHandler(typeof(FtsDspControl), FtsSymbol.Get("setFifoSize"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsDspControl)obj).FifoSize = args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsDspControl), FtsSymbol.Get("setSamplingRate"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsDspControl)obj).SamplingRate = args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsDspControl), FtsSymbol.Get("setDACSlipState"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsDspControl)obj).DacSlip = args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsDspControl), FtsSymbol.Get("setInvalidFpeState"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsDspControl)obj).InvalidFpe = args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsDspControl), FtsSymbol.Get("setDivideByZeroFpeState"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsDspControl)obj).DivideByZeroFpe = args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsDspControl), FtsSymbol.Get("setOverflowFpeState"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsDspControl)obj).OverflowFpe = args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsDspControl), FtsSymbol.Get("setDenormalizeFpeState"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsDspControl)obj).DenormalizedFpe = args.GetInt(0);
                });
        }

        private class FtsDspControlListener : IFtsEditListener
        {
            public FtsDspControlListener(FtsDspControl dspControl)
            {
                this.dspControl = dspControl;
            }

            public void ObjectAdded(FtsObject ftsObject)
            {
                if (!dspControl.atomic && dspControl.dspOn)
                    dspControl.Restart();
            }

            public void ObjectRemoved(FtsObject ftsObject)
            {
            }

            public void ConnectionAdded(FtsConnection connection)
            {
                if (!dspControl.atomic && dspControl.dspOn)
                    dspControl.Restart();
            }

            public void ConnectionRemoved(FtsConnection connection)
            {
            }

            public void AtomicAction(bool active)
            {
                dspControl.atomic = active;
                if (!dspControl.atomic && dspControl.dspOn)
                    dspControl.Restart();
            }

            private FtsDspControl dspControl;
        }

        public FtsDspControl()
            : base(JMaxLibrary.FtsServer, JMaxLibrary.RootPatcher, FtsSymbol.Get("__dspcontrol"))
        {
            dacSlip = 0;
            invalidFpe = 0;
            divideByZeroFpe = 0;
            overflowFpe = 0;
            denormalizedFpe = 0;
            samplingRate = 0;
            fifoSize = 0;
            dspOn = false;

            FtsPatcherObject.AddGlobalEditListener(new FtsDspControlListener(this));
        }

        /* Accessors for fields */
        /* Modifiers, for values that can be modified */

        public int DacSlip
        {
            get
            {
                return dacSlip;
            }
            set
            {
                dacSlip = value;
            }
        }

        public int InvalidFpe
        {
            get
            {
                return invalidFpe;
            }
            set
            {
                invalidFpe = value;
            }
        }

        public int DivideByZeroFpe
        {
            get
            {
                return divideByZeroFpe;
            }
            set
            {
                divideByZeroFpe = value;
            }
        }

        public int OverflowFpe
        {
            get
            {
                return overflowFpe;
            }
            set
            {
                overflowFpe = value;
            }
        }

        public int DenormalizedFpe
        {
            get
            {
                return denormalizedFpe;
            }
            set
            {
                denormalizedFpe = value;
            }
        }

        public int SamplingRate
        {
            get
            {
                return samplingRate;
            }
            set
            {
                samplingRate = value;
            }
        }

        public int FifoSize
        {
            get
            {
                return fifoSize;
            }
            set
            {
                fifoSize = value;
            }
        }

        public bool DspOn
        {
            get
            {
                return dspOn;
            }
            set
            {
                dspOn = value;

                JMaxLibrary.ConsoleWindow.DspOn = value;
            }
        }

        public void RequestSetDspOn(bool val)
        {
            args.Clear();
            args.AddBoolean(val);

            try
            {
                Send(FtsSymbol.Get("dsp_on"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsDspControl: I/O Error sending dspOn Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void DspPrint()
        {
            try
            {
                Send(FtsSymbol.Get("dsp_print"));
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsDspControl: I/O Error sending dsp_print Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void Restart()
        {
            try
            {
                Send(FtsSymbol.Get("dsp_restart"));
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsDspControl: I/O Error sending dsp_restart Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        /* Fpe support */

        public void StartFpeCollecting(FtsObjectSet ftsObjectSet)
        {
            args.Clear();
            args.AddObject(ftsObjectSet);

            try
            {
                Send(FtsSymbol.Get("fpe_start_collect"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsDspControl: I/O Error sending fpe_start_collect Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void StopFpeCollecting()
        {
            try
            {
                Send(FtsSymbol.Get("fpe_stop_collecting"));
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsDspControl: I/O Error sending fpe_stop_collect Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void ClearFpeCollecting()
        {
            try
            {
                Send(FtsSymbol.Get("fpe_clear_collecting"));
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsDspControl: I/O Error sending fpe_clear_collecting Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public bool CheckNan
        {
            set
            {
                /*sendArgs[0].SetInt(checkNan ? 1 : 0);
                sendMessage(FtsObject.systemInlet, "set_check_nan", 1, sendArgs);*/
            }
        }
    }
}