//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.IO;
using System.Collections;

using ircam.fts.client;
using ircam.jmax;
using ircam.jmax.editors.project;

namespace ircam.jmax.fts
{
    public class FtsMidiConfig : FtsObject
    {
        protected internal FtsArgs args = new FtsArgs();

        static FtsMidiConfig()
        {
            FtsObject.RegisterMessageHandler(typeof(FtsMidiConfig), FtsSymbol.Get("inputs"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsMidiConfig)obj).SetSources(args.Length, args.Atoms);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsMidiConfig), FtsSymbol.Get("outputs"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsMidiConfig)obj).SetDestinations(args.Length, args.Atoms);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsMidiConfig), FtsSymbol.Get("insert"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsMidiConfig)obj).InsertLabel(args.GetInt(0), args.GetSymbol(1).ToString());

                    if (args.Length > 2)
                        ((FtsMidiConfig)obj).SetInput(args.GetInt(0), args.GetSymbol(2).ToString());

                    if (args.Length > 3)
                        ((FtsMidiConfig)obj).SetOutput(args.GetInt(0), args.GetSymbol(3).ToString());
                });
            FtsObject.RegisterMessageHandler(typeof(FtsMidiConfig), FtsSymbol.Get("set"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsMidiConfig)obj).SetLabel(args.GetInt(0), args.GetSymbol(1).ToString(),
                        args.GetSymbol(2).ToString(), args.GetSymbol(3).ToString());
                });
            FtsObject.RegisterMessageHandler(typeof(FtsMidiConfig), FtsSymbol.Get("remove"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsMidiConfig)obj).RemoveLabel(args.GetInt(0));
                });
            FtsObject.RegisterMessageHandler(typeof(FtsMidiConfig), FtsSymbol.Get("input"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsMidiConfig)obj).SetInput(args.GetInt(0), args.GetSymbol(1).ToString());
                });
            FtsObject.RegisterMessageHandler(typeof(FtsMidiConfig), FtsSymbol.Get("output"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsMidiConfig)obj).SetOutput(args.GetInt(0), args.GetSymbol(1).ToString());
                });
            FtsObject.RegisterMessageHandler(typeof(FtsMidiConfig), FtsSymbol.Get("clear"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsMidiConfig)obj).Clear();
                });
        }

        public FtsMidiConfig(FtsServer server, FtsObject parent, int id)
            : base(server, parent, id)
        {
            labels = new ArrayList();
        }

        public FtsMidiConfig()
            : base(JMaxLibrary.FtsServer, JMaxLibrary.RootPatcher, FtsSymbol.Get("slider"))
        {
        }

        public ircam.jmax.editors.configuration.MidiConfigPanel Listener
        {
            set
            {
                this.listener = value;
            }
        }

        public void RequestInsertLabel(int index, string label)
        {
            labels.Insert(index, new MidiLabel(label, null, null));

            args.Clear();

            args.AddInt(index);
            args.AddSymbol(FtsSymbol.Get(label));

            try
            {
                Send(FtsSymbol.Get("insert"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsMidiConfig: I/O Error sending insert Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void RequestRemoveLabel(int index)
        {
            args.Clear();

            args.AddInt(index);

            try
            {
                Send(FtsSymbol.Get("remove"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsMidiConfig: I/O Error sending remove Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void RequestSetInput(int index, string input)
        {
            args.Clear();

            args.AddInt(index);
            args.AddSymbol(FtsSymbol.Get(input));

            try
            {
                Send(FtsSymbol.Get("input"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsMidiConfig: I/O Error sending input Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void RequestSetOutput(int index, string output)
        {
            args.Clear();

            args.AddInt(index);
            args.AddSymbol(FtsSymbol.Get(output));

            try
            {
                Send(FtsSymbol.Get("output"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsMidiConfig: I/O Error sending output Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        internal void SetSources(int nArgs, FtsAtom[] args)
        {
            sources = new string[nArgs];
            for (int i = 0; i < nArgs; i++)
                sources[i] = args[i].symbolValue.ToString();

            if (listener != null)
                listener.SourcesChanged();
        }

        public string[] GetSources()
        {
            return sources;
        }

        internal void SetDestinations(int nArgs, FtsAtom[] args)
        {
            destinations = new string[nArgs];
            for (int i = 0; i < nArgs; i++)
                destinations[i] = args[i].symbolValue.ToString();

            if (listener != null)
                listener.DestinationsChanged();
        }

        public string[] getDestinations()
        {
            return destinations;
        }

        internal void InsertLabel(int index, string label)
        {
            labels.Insert(index, new MidiLabel(label, null, null));
        }

        internal void RemoveLabel(int index)
        {
            if (index >= 0 && index < labels.Count)
            {
                labels.RemoveAt(index);
            }
        }

        public IEnumerator Labels
        {
            get
            {
                return labels.GetEnumerator();
            }
        }

        internal void SetLabel(int index, string name, string input, string output)
        {
            ((MidiLabel)labels[index]).SetLabel(name, input, output);
            if (listener != null)
                listener.LabelChanged(index, name, input, output);
        }

        internal void SetInput(int index, string input)
        {
            ((MidiLabel)labels[index]).Input = input;
            if (listener != null)
                listener.LabelChanged(index, null, input, null);
        }

        internal void SetOutput(int index, string output)
        {
            ((MidiLabel)labels[index]).Output = output;
            if (listener != null)
                listener.LabelChanged(index, null, null, output);
        }

        public void Clear()
        {
            labels.Clear();
        }

        public class MidiLabel
        {
            public MidiLabel(string label, string input, string output)
            {
                this.label = label;
                this.input = input;
                this.output = output;
            }

            public void SetLabel(string name, string input, string output)
            {
                this.label = name;
                this.input = input;
                this.output = output;
            }

            public string Input
            {
                set
                {
                    input = value;
                }
            }

            public string Output
            {
                set
                {
                    output = value;
                }
            }

            public string label, input, output;
        }

        private string[] sources;
        private string[] destinations;
        private ArrayList labels;
        private ircam.jmax.editors.configuration.MidiConfigPanel listener;
    }
}