//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.IO;
using System.Collections.Generic;

using ircam.fts.client;
using ircam.jmax;
using ircam.jmax.editors.project;

namespace ircam.jmax.fts
{
    public class FtsPackage : FtsObjectWithEditor
    {
        new protected internal FtsArgs args = new FtsArgs();

        static FtsPackage()
        {
            FtsObject.RegisterMessageHandler(typeof(FtsPackage), FtsSymbol.Get("require"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPackage)obj).AddRequires(args.Length, args.Atoms);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPackage), FtsSymbol.Get("package_path"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPackage)obj).AddPackagePath(args.Length, args.Atoms);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPackage), FtsSymbol.Get("template_path"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPackage)obj).AddTemplatePath(args.Length, args.Atoms);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPackage), FtsSymbol.Get("data_path"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPackage)obj).AddDataPath(args.Length, args.Atoms);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPackage), FtsSymbol.Get("template"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPackage)obj).AddTemplates(args.Length, args.Atoms);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPackage), FtsSymbol.Get("help"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPackage)obj).AddHelps(args.Length, args.Atoms);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPackage), FtsSymbol.Get("setNames"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPackage)obj).Name = args.GetSymbol(0).ToString();
                    ((FtsPackage)obj).Dir = args.GetSymbol(1).ToString();
                    ((FtsPackage)obj).FileName = args.GetSymbol(2).ToString();
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPackage), FtsSymbol.Get("hasSummaryHelp"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPackage)obj).hasSummaryHelp = (args.GetInt(0) == 1);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPackage), FtsSymbol.Get("updateDone"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    if (((FtsPackage)obj).listener != null)
                        ((FtsPackage)obj).listener.UpdateDone();
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPackage), FtsSymbol.Get("uploadDone"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPackage)obj).UploadDone();
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPackage), FtsSymbol.Get("setDirty"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPackage)obj).IsDirty = args.GetInt(0) == 1;
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPackage), FtsSymbol.Get("requireError"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPackage)obj).RequireError(args.GetInt(0));
                });
        }

        public FtsPackage()
            : base(JMaxLibrary.FtsServer, JMaxLibrary.RootPatcher, FtsSymbol.Get("package"))
        {
            requires = new List<string>();
            packagePaths = new List<string>();
            templatePaths = new List<string>();
            absPaths = new List<string>();
            dataPaths = new List<string>();
            templates = new List<FtsPackage.TwoNames>();
            helpPatches = new List<FtsPackage.TwoNames>();
        }

        public FtsPackage(FtsServer server, FtsObject parent, int id) : base(server, parent, id)
        {
            requires = new List<string>();
            packagePaths = new List<string>();
            templatePaths = new List<string>();
            absPaths = new List<string>();
            dataPaths = new List<string>();
            templates = new List<FtsPackage.TwoNames>();
            helpPatches = new List<FtsPackage.TwoNames>();
        }

        public ircam.jmax.editors.project.ProjectEditor Listener
        {
            get
            {
                return this.listener;
            }
            set
            {
                this.listener = value;
            }
        }

        public void SetMessage(string message, List<string> values)
        {
            args.Clear();

            values.ForEach(
                delegate(string str)
                {
                    if (str != null)
                        args.AddSymbol(FtsSymbol.Get(str));
                });

            try
            {
                Send(FtsSymbol.Get(message), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPackage: I/O Error sending " + message + " Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void SetMessage(string message, string[] values)
        {
            args.Clear();
            for (int i = 0; i < values.Length; i++)
            {
                if (values[i] != null)
                    args.AddSymbol(FtsSymbol.Get(values[i]));
            }
            try
            {
                Send(FtsSymbol.Get(message), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPackage: I/O Error sending " + message + " Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void RequestAddTemplate(string name, string fileName, int index)
        {
            args.Clear();
            args.AddSymbol(FtsSymbol.Get(name));
            args.AddSymbol(FtsSymbol.Get(fileName));
            args.AddInt(index);

            try
            {
                Send(FtsSymbol.Get("insert_template"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPackage: I/O Error sending template Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void RequestRemoveTemplate(string name, string fileName)
        {
            args.Clear();
            args.AddSymbol(FtsSymbol.Get(name));
            args.AddSymbol(FtsSymbol.Get(fileName));

            try
            {
                Send(FtsSymbol.Get("remove_template"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPackage: I/O Error sending remove_template Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void Save(string name)
        {
            args.Clear();
            args.AddSymbol(FtsSymbol.Get((name != null) ? name : fileName));

            try
            {
                Send(FtsSymbol.Get("save"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPackage: I/O Error sending save Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void Update()
        {
            requires.Clear();
            templatePaths.Clear();
            dataPaths.Clear();
            templates.Clear();
            helpPatches.Clear();
            helpPatches.Clear();

            try
            {
                Send(FtsSymbol.Get("update"));
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPackage: I/O Error sending update Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void Upload()
        {
            try
            {
                Send(FtsSymbol.Get("upload"));
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPackage: I/O Error sending upload Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        internal void AddRequires(int nArgs, FtsAtom[] args)
        {
            requires.Clear();

            for (int i = 0; i < nArgs; i += 2)
                AddPackage(args[i].symbolValue.ToString(), args[i + 1].intValue);
        }

        public virtual void AddPackage(string pkgName, int id)
        {
            requires.Add(pkgName);
        }

        internal void AddPackagePath(int nArgs, FtsAtom[] args)
        {
            packagePaths.Clear();

            for (int i = 0; i < nArgs; i++)
                packagePaths.Add(args[i].symbolValue.ToString());

            if (listener != null)
                listener.PackagePathChanged();
        }

        internal void AddTemplatePath(int nArgs, FtsAtom[] args)
        {
            templatePaths.Clear();

            for (int i = 0; i < nArgs; i++)
                templatePaths.Add(args[i].symbolValue.ToString());

            if (listener != null)
                listener.TemplatePathChanged();
        }

        internal void AddDataPath(int nArgs, FtsAtom[] args)
        {
            dataPaths.Clear();

            for (int i = 0; i < nArgs; i++)
                dataPaths.Add(args[i].symbolValue.ToString());

            if (listener != null)
                listener.DataPathChanged();
        }

        internal void AddTemplates(int nArgs, FtsAtom[] args)
        {
            templates.Clear();

            for (int i = 0; i < nArgs; i += 2)
                templates.Add(new TwoNames(args[i].symbolValue.ToString(),
                              args[i + 1].symbolValue.ToString()));

            if (listener != null)
                listener.TemplateChanged();
        }

        internal void AddHelps(int nArgs, FtsAtom[] args)
        {
            helpPatches.Clear();

            for (int i = 0; i < nArgs; i += 2)
                helpPatches.Add(new TwoNames(args[i].symbolValue.ToString(),
                                args[i + 1].symbolValue.ToString()));

            /*if( listener != null)
            listener.HelpChanged();*/
        }

        public string Name
        {
            get
            {
                return name;
            }
            set
            {
                this.name = value;
            }
        }

        public string FileName
        {
            get
            {
                return fileName;
            }
            set
            {
                this.fileName = value;
            }
        }

        public string Dir
        {
            get
            {
                return dir;
            }
            set
            {
                this.dir = value;
            }

        }

        internal void UploadDone()
        {
            if (hasSummaryHelp)
                FtsHelpPatchTable.AddSummary(name.ToString(), dir + "\\help\\" + name + ".summary.jmax");
        }

        public override void OpenEditor(int nArgs, FtsAtom[] args)
        {
            if (EditorForm == null)
                EditorForm = new ProjectEditor(this);

            ShowEditor();
        }

        public override void DestroyEditor()
        {
            DisposeEditor();
        }

        public bool IsDirty
        {
            get
            {
                return isDirty;
            }
            set
            {
                this.isDirty = isDirty;
            }
        }

        public void RequireError(int index)
        {
            if (listener != null)
                listener.SetErrorAt(index);
        }
        /*************************************/

        public List<string> Requires
        {
            get
            {
                return requires;
            }
        }

        public List<string> PackagePaths
        {
            get
            {
                return packagePaths;
            }
        }

        public List<string> TemplatePaths
        {
            get
            {
                return templatePaths;
            }
        }

        public List<string> DataPaths
        {
            get
            {
                return dataPaths;
            }
        }

        public List<FtsPackage.TwoNames> Templates
        {
            get
            {
                return templates;
            }
        }

        public List<FtsPackage.TwoNames> Helps
        {
            get
            {
                return helpPatches;
            }
        }

        public class TwoNames
        {
            public TwoNames(string name, string fileName)
            {
                this.name = name;
                this.fileName = fileName;
            }

            public void SetName(string n, string fn)
            {
                name = n;
                fileName = fn;
            }

            public string name, fileName;
        }

        private List<string> requires, packagePaths, templatePaths, absPaths, dataPaths;
        private List<FtsPackage.TwoNames> templates, helpPatches;
        private string name;
        private string fileName;
        private string dir;
        internal ircam.jmax.editors.project.ProjectEditor listener;
        private bool hasSummaryHelp = true;
        private bool isDirty = false;
    }
}