//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.IO;
using System.Threading;
using System.Collections;
using System.Windows.Forms;

using ircam.fts.client;
using ircam.jmax.editors.patcher;
using ircam.jmax.editors.patcher.objects;

namespace ircam.jmax.fts
{
    /// <summary> Proxy of an FTS patcher.</summary>
    public class FtsPatcherObject : FtsObjectWithEditor
    {
        /// <summary>Patcher content: the window size and position </summary>
        private int windowX = 0;
        private int windowY = 0;
        private int windowHeight = 0;
        private int windowWidth = 0;

        [NonSerialized()]
        private FtsArgs args = new FtsArgs();

        /*
            Patcher content: the edit mode;
            temporary hack, local, not propagated to fts,
            should disappear with the toolbar instead of the mode
            in the patcher editor.
            */

        public const int UNKNOWN_MODE = 0;
        public const int EDIT_MODE = 1;
        public const int RUN_MODE = 2;

        public const int JMAX_FILE_TYPE = 1;
        public const int PAT_FILE_TYPE = 0;

        //int editMode = UNKNOWN_MODE;
        internal int editMode = EDIT_MODE;

        /// <summary>Patcher content: objects </summary>
        private MaxVector objects = new MaxVector();

        /// <summary>Patcher content: connections </summary>
        private MaxVector connections = new MaxVector();

        /// <summary>List of subPatchers </summary>
        private MaxVector subPatchers = new MaxVector();

        /// <summary>true if patcher need to be saved </summary>
        private bool dirty = false;

        /// <summary>true if is already saved a first time </summary>
        private bool canSave = false;

        private string name = null;
        private int type = JMAX_FILE_TYPE;

        private bool pasting = false;

        /******************************************************************************/
        /*                                                                            */
        /*              STATIC FUNCTION                                               */
        /*                                                                            */
        /******************************************************************************/

        /// <summary>This function create an application layer object for an already existing
        /// object in FTS
        /// </summary>

        internal static GraphicObject makeGraphicObjectFromServer(FtsServer server, FtsObject parent, int objId, string className,
                                                                  FtsAtom[] args, int offset, int nArgs)
        {
            JMaxObjectCreator creator = null;

            if (className != null)
            {
                creator = JMaxClassMap.GetCreator(className);

                if (creator != null)
                    return (GraphicObject)creator(server, parent, objId, className, args, offset, nArgs);
            }
            creator = JMaxClassMap.GetCreator("standard");
            if (creator != null)
                return (GraphicObject)creator(server, parent, objId, className, args, offset, nArgs);

            return null;
        }

        /*****************************************************************************/
        /*                                                                           */
        /*                               MessageHandlers                             */
        /*                                                                           */
        /*****************************************************************************/

        static FtsPatcherObject()
        {
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("addObject"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).AddObject(args.Length, args.Atoms);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("removeObject"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).ReleaseObject((FtsObject)args.GetObject(0));
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("addConnection"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).AddConnection(args.Length, args.Atoms);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("objectRedefined"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).ObjectRedefined((FtsGraphicObject)args.GetObject(0));
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("stopWaiting"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).StopWaiting();
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("showObject"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).ShowObject((FtsGraphicObject)args.GetObject(0));
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("redefineStart"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).RedefineStart();
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("setRedefined"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    //((FtsPatcherObject)obj).FirePatcherChanged();
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("startUpload"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).StartUpload();
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("endUpload"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).EndUpload();
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("setDescription"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).Description = args.GetString(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("setWX"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).WX = args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("setWY"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).WY = args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("setWW"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).WW = args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("setWH"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).WH = args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("setPatcherBounds"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).SetPatcherBounds(args.GetInt(0), args.GetInt(1), args.GetInt(2), args.GetInt(3));
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("setMessage"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).Message = args.GetString(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("setDirty"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).IsDirty = (args.GetInt(0) == 1) ? true : false;
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("setSaved"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).SetSaved(args.GetInt(0), args.GetSymbol(1).ToString());
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("startPaste"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).StartPaste();
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("endPaste"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsPatcherObject)obj).EndPaste();
                });
            FtsObject.RegisterMessageHandler(typeof(FtsPatcherObject), FtsSymbol.Get("noHelp"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    FtsGraphicObject ftsGraphicObject = (FtsGraphicObject)args.GetObject(0);
                    MessageBox.Show("Sorry, no help for object " + ftsGraphicObject.ClassName,
                                    "Warning",
                                    MessageBoxButtons.OK,
                                    MessageBoxIcon.Warning);
                });
        }

        /*****************************************************************************************/
        /*                                 Global Edit Listening                                 */
        /*****************************************************************************************/

        [NonSerialized()]
        private static MaxVector editListeners = new MaxVector();

        static public void AddGlobalEditListener(IFtsEditListener listener)
        {
            editListeners.AddElement(listener);
        }

        static public void RemoveGlobalEditListener(IFtsEditListener listener)
        {
            editListeners.RemoveElement(listener);
        }

        static public void FireGlobalObjectRemoved(FtsGraphicObject ftsGraphicObject)
        {
            for (int i = 0; i < editListeners.Size; i++)
                ((IFtsEditListener)editListeners[i]).ObjectRemoved(ftsGraphicObject);
        }

        static public void FireGlobalObjectAdded(FtsGraphicObject ftsGraphicObject)
        {
            for (int i = 0; i < editListeners.Size; i++)
                ((IFtsEditListener)editListeners[i]).ObjectAdded(ftsGraphicObject);
        }

        static public void FireGlobalConnectionAdded(FtsConnection connection)
        {
            for (int i = 0; i < editListeners.Size; i++)
                ((IFtsEditListener)editListeners[i]).ConnectionAdded(connection);
        }

        static public void FireGlobalConnectionRemoved(FtsConnection connection)
        {
            for (int i = 0; i < editListeners.Size; i++)
                ((IFtsEditListener)editListeners[i]).ConnectionRemoved(connection);
        }

        static public void FireGlobalAtomicAction(bool active)
        {
            for (int i = 0; i < editListeners.Size; i++)
                ((IFtsEditListener)editListeners[i]).AtomicAction(active);
        }

        /*****************************************************************************/
        /*                                                                           */
        /*                               CONSTRUCTORS                                */
        /*                                                                           */
        /*****************************************************************************/

        /// <summary> Create a FtsPatcherObject object</summary>
        public FtsPatcherObject(FtsServer server, FtsObject parent, int id, string className, FtsAtom[] args, int offset, int length)
            : base(server, parent, id, className, (length > 0) ? args[offset].stringValue : null) { }

        public FtsPatcherObject()
            : base(JMaxLibrary.FtsServer, JMaxLibrary.RootPatcher, FtsSymbol.Get("jpatcher")) { }

        /// <summary>Get all the objects in this patcherdata </summary>
        public MaxVector Objects
        {
            get
            {
                return objects;
            }
        }

        /// <summary>Get all the connections in this patcherdata </summary>
        public MaxVector Connections
        {
            get
            {
                return connections;
            }
        }

        private void StartUpload()
        {
            FireGlobalAtomicAction(true);
        }

        private void EndUpload()
        {
            this.canSave = true;
            this.dirty = false;

            FireGlobalAtomicAction(false);

            if (EditorForm != null)
            {
                ErmesSketchPad sketch = ((ErmesSketchWindow)EditorForm).SketchPad;
                sketch.DisplayList.SortDisplayList();

                if (IsARootPatcher)
                    sketch.IsLocked = true;
                else
                    sketch.IsLocked = (((FtsPatcherObject)Parent).IsLocked);
                
                ErmesSelection.patcherSelection.DeselectAll();
            }
        }

        public bool IsDirty
        {
            get
            {
                return dirty;
            }
            set
            {
                this.dirty = value;
                if (EditorForm != null)
                    ((ErmesSketchWindow)EditorForm).SketchPad.IsDirty = this.dirty;

                for (IEnumerator e = subPatchers.Elements(); e.MoveNext(); )
                {
                    ((FtsPatcherObject)e.Current).IsDirty = this.dirty;
                }
            }
        }

        private void SetSaved(int type, string name)
        {
            this.canSave = true;
            this.name = name;
            this.type = type;

            if (EditorForm != null)
                EditorForm.Text = name;

            if (IsDirty)
                IsDirty = false;
        }

        public string Name
        {
            get
            {
                return name;
            }
            set
            {
                this.name = value;
            }
        }

        public int Type
        {
            get
            {
                return type;
            }
            set
            {
                this.type = value;
            }
        }

        public bool CanSave
        {
            get
            {
                return canSave;
            }
        }

        public void Save()
        {
            Save(this.type, this.name);
        }

        public void Save(int type, string name)
        {
            args.Clear();
            args.AddInt(type);
            args.AddSymbol(FtsSymbol.Get(name));

            try
            {
                Send(FtsSymbol.Get("save"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPatcherObject: I/O Error sending save Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
            this.name = name;
            this.type = type;
        }

        public int WindowX
        {
            get
            {
                return windowX;
            }
            set
            {
                if (windowX != value)
                {
                    windowX = value;
                    args.Clear();
                    args.AddInt(windowX);

                    try
                    {
                        Send(FtsSymbol.Get("set_wx"), args);
                    }
                    catch (IOException e)
                    {
                        Console.Error.WriteLine("FtsPatcherObject: I/O Error sending set_wx Message!");
                        Console.Error.WriteLine(e.StackTrace);
                    }
                }
            }
        }

        public int WindowY
        {
            get
            {
                return windowY;
            }
            set
            {
                if (windowY != value)
                {
                    windowY = value;
                    args.Clear();
                    args.AddInt(windowY);

                    try
                    {
                        Send(FtsSymbol.Get("set_wy"), args);
                    }
                    catch (IOException e)
                    {
                        Console.Error.WriteLine("FtsPatcherObject: I/O Error sending set_wy Message!");
                        Console.Error.WriteLine(e.StackTrace);
                    }
                }
            }
        }

        public int WindowHeight
        {
            get
            {
                return windowHeight;
            }
            set
            {
                if (windowHeight != value)
                {
                    windowHeight = value;

                    args.Clear();
                    args.AddInt(windowHeight);

                    try
                    {
                        Send(FtsSymbol.Get("set_wh"), args);
                    }
                    catch (IOException e)
                    {
                        Console.Error.WriteLine("FtsPatcherObject: I/O Error sending set_wh Message!");
                        Console.Error.WriteLine(e.StackTrace);
                    }
                }
            }
        }

        public int WindowWidth
        {
            get
            {
                return windowWidth;
            }
            set
            {
                if (windowWidth != value)
                {
                    windowWidth = value;

                    args.Clear();
                    args.AddInt(windowWidth);

                    try
                    {
                        Send(FtsSymbol.Get("set_ww"), args);
                    }
                    catch (IOException e)
                    {
                        Console.Error.WriteLine("FtsPatcherObject: I/O Error sending set_ww Message!");
                        Console.Error.WriteLine(e.StackTrace);
                    }
                }
            }
        }

        /// <summary>Get the edit mode of this patcher data.
        /// The edit mode is stored in the patcher data 
        /// because we need to know it in order to decide
        /// the initial mode of a new patcher editor.
        /// </summary>
        /// <summary>Set the edit mode of this patcher data.
        /// The edit mode is stored in the patcher data 
        /// because we need to know it in order to decide
        /// the initial mode of a new patcher editor.
        /// </summary>
        public int EditMode
        {
            get
            {
                return editMode;
            }
            set
            {
                editMode = value;
            }
        }

        /// <summary>Get the edit mode of this patcher data,
        /// or, if unknown, the first known edit mode in
        /// its patcher ancestor chain.
        /// The edit mode is stored in the patcher data 
        /// because we need to know it in order to decide
        /// the initial mode of a new patcher editor.
        /// </summary>
        public int RecursiveEditMode
        {
            get
            {
                if (editMode == UNKNOWN_MODE)
                {
                    if (Parent != null)
                        if (Parent is FtsPatcherObject)
                            return ((FtsPatcherObject)Parent).RecursiveEditMode;

                    return UNKNOWN_MODE;
                }
                else
                    return editMode;
            }
        }

        /// <summary>Add an object to this patcher data </summary>
        internal void AddObject(FtsGraphicObject obj)
        {
            objects.AddElement(obj);

            if (obj is FtsPatcherObject)
            {
                ((FtsPatcherObject)obj).IsDirty = IsDirty;
                subPatchers.AddElement(obj);
            }
            FireGlobalObjectAdded(obj);
        }

        /// <summary>Add a connection to this patcher data </summary>
        internal void AddConnection(FtsConnection c)
        {
            connections.AddElement(c);
            FireGlobalConnectionAdded(c);
        }

        /// <summary>Remove an object to this patcher data </summary>
        public void RemoveObject(FtsGraphicObject obj)
        {
            objects.RemoveElement(obj);

            if (obj is FtsPatcherObject)
                subPatchers.RemoveElement(obj);

            FireGlobalObjectRemoved(obj);
        }

        /// <summary>Remove a connection to this patcher data </summary>
        internal void RemoveConnection(FtsConnection c)
        {
            connections.RemoveElement(c);
            FireGlobalConnectionRemoved(c);
        }

        /// <summary> Redefine a patcher without looosing its content.</summary>
        public void RedefinePatcher(string description)
        {
            args.Clear();
            args.AddRawString(description);

            try
            {
                Send(FtsSymbol.Get("set_arguments"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPatcherObject: I/O Error sending redefine_patcher Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void RequestShowObject(FtsGraphicObject obj)
        {
            args.Clear();
            args.AddObject(obj);

            try
            {
                Send(FtsSymbol.Get("show_object"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPatcherObject: I/O Error sending show_object Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        [NonSerialized()]
        internal IFtsActionListener waitingListener;

        public void RequestStopWaiting()
        {
            try
            {
                Send(FtsSymbol.Get("stop_waiting"));
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPatcherObject: I/O Error sending stop_waiting Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        /// <summary>Ask fts to update the patcher; usefull after a paste </summary>
        public void Update()
        {
            try
            {
                Send(FtsSymbol.Get("patcher_update"));
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPatcherObject: I/O Error sending patcher_update Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void RequestPaste(FtsClipboard clipboard, int dx, int dy)
        {
            args.Clear();
            args.AddObject(clipboard);
            args.AddInt(dx);
            args.AddInt(dy);

            try
            {
                Send(FtsSymbol.Get("paste"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPatcherObject: I/O Error sending paste Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        //
        // Updates Management
        //

        /// <summary>Tell FTS that this patcher is  "alive". 
        /// Fts will Send updates for this patcher.
        /// </summary>
        public void StartUpdates()
        {
            try
            {
                Send(FtsSymbol.Get("start_updates"));
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPatcherObject: I/O Error sending start_updates Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void RequestSubPatcherUpload()
        {
            if (EditorForm == null)
            {
                EditorForm = new ErmesSketchWindow(this);

                try
                {
                    Send(FtsSymbol.Get("upload"));
                }
                catch (IOException e)
                {
                    Console.Error.WriteLine("FtsPatcherObject: I/O Error sending upload Message!");
                    Console.Error.WriteLine(e.StackTrace);
                }
            }
            RequestOpenEditor();
        }

        public bool IsLocked
        {
            get
            {
                if (EditorForm != null)
                    return ((ErmesSketchWindow)EditorForm).SketchPad.IsLocked;
                else
                    return false;
            }
        }

        public void RequestOpenHelpPatch(FtsObject obj)
        {
            args.Clear();
            args.AddObject(obj);
            try
            {
                Send(FtsSymbol.Get("open_help_patch"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPatcherObject: I/O Error sending open_help_patch Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        /// <summary>Tell FTS that this patcher is not "alive".
        /// Fts will stop sending updates for this patcher.
        /// </summary>
        public void StopUpdates()
        {
            try
            {
                Send(FtsSymbol.Get("stop_updates"));
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPatcherObject: I/O Error sending stop_updates Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void RequestAddObject(string description, int x, int y, bool doedit)
        {
            args.Clear();
            args.AddInt(x);
            args.AddInt(y);
            args.AddRawString(description);

            try
            {
                Send(FtsSymbol.Get("add_object"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPatcherObject: I/O Error sending add_object Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void RequestRedefineObject(FtsGraphicObject oldObject, string description)
        {
            args.Clear();
            args.AddObject(oldObject);
            args.AddRawString(description);

            try
            {
                Send(FtsSymbol.Get("redefine_object"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPatcherObject: I/O Error sending redefine_object Message!");
                Console.Error.WriteLine(e.StackTrace);
            }

            //removeObject(oldObject);
        }

        public void RequestAddConnection(FtsGraphicObject from, int outlet, FtsGraphicObject to, int inlet)
        {
            args.Clear();
            args.AddObject(from);
            args.AddInt(outlet);
            args.AddObject(to);
            args.AddInt(inlet);

            try
            {
                Send(FtsSymbol.Get("add_connection"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPatcherObject: I/O Error sending add_connection Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void RequestDeleteConnection(FtsConnection connection)
        {
            args.Clear();
            args.AddObject(connection);

            try
            {
                Send(FtsSymbol.Get("delete_connection"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPatcherObject: I/O Error sending delete_connection Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void RequestDeleteObjects(IEnumerator objects)
        {
            args.Clear();

            for (; objects.MoveNext(); )
            {
                args.AddObject(((GraphicObject)objects.Current).FtsObject);
            }

            try
            {
                Send(FtsSymbol.Get("delete_objects"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPatcherObject: I/O Error sending delete_objects Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void RequestDeleteObject(FtsObject obj)
        {
            args.Clear();
            args.AddObject(obj);

            try
            {
                Send(FtsSymbol.Get("delete_objects"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsPatcherObject: I/O Error sending delete_objects Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        //used in AddObject method to start editing in added object if needed 
        public void AddObject(int nArgs, FtsAtom[] args)
        {
            int objId = args[0].intValue;
            int x = args[1].intValue;
            int y = args[2].intValue;
            int width = args[3].intValue;
            int height = args[4].intValue;
            int numIns = args[5].intValue;
            int numOuts = args[6].intValue;
            int layer = args[7].intValue;
            string errorDescription = args[8].symbolValue.ToString();
            string className = args[9].symbolValue.ToString();

            bool isTemplate = (args[10].intValue == 1);

            GraphicObject newObj;

            if (isTemplate)
                newObj = new Standard(new FtsTemplateObject(this.Server, this, objId, className, args, 11, nArgs - 11));
            else
                newObj = makeGraphicObjectFromServer(this.Server, this, objId, className, args, 11, nArgs - 11);

            newObj.FtsObject.CurrentLayer = layer;

            newObj.setCurrentBounds(x, y, width, height);

            newObj.FtsObject.NumberOfInlets = numIns;
            newObj.FtsObject.NumberOfOutlets = numOuts;

            if (!errorDescription.Equals("no_error"))
            {
                newObj.FtsObject.IsError = true;
                newObj.FtsObject.ErrorDescription = errorDescription;
            }
            else
                newObj.FtsObject.IsError = false;

            AddObject(newObj.FtsObject);

            bool doEdit = false;
            if (((FtsGraphicObject)newObj.FtsObject).ClassName == null || newObj.instantEdit())
                doEdit = true;
            else
            {
                string descr = ((FtsGraphicObject)newObj.FtsObject).Description;
                if (descr != null)
                    descr.Trim();
                if ((descr != null) && descr.Equals(""))
                    doEdit = true;
            }
            newObj.SketchPad.AddNewObject(newObj, doEdit);

            if (pasting)
                ((ErmesSketchWindow)EditorForm).SketchPad.AddPastedObject(newObj);
        }

        public void AddConnection(int nArgs, FtsAtom[] args)
        {
            if (nArgs == 6)
            {
                FtsConnection connection = new FtsConnection(this.Server, this, args[0].intValue,
                                                             (FtsGraphicObject)args[1].objectValue,
                                                             args[2].intValue, (FtsGraphicObject)args[3].objectValue,
                                                             args[4].intValue, args[5].intValue);
                AddConnection(connection);
                GraphicConnection gc = ((ErmesSketchWindow)EditorForm).SketchPad.AddNewConnection(connection);

                if (pasting)
                    ((ErmesSketchWindow)EditorForm).SketchPad.AddPastedConnection(gc);
            }
        }

        public void ReleaseConnection(FtsConnection c)
        {
            if (EditorForm != null)
                ((ErmesSketchWindow)EditorForm).SketchPad.DisplayList.Remove(c);
        }

        public void ObjectRedefined(FtsGraphicObject obj)
        {
            ((ErmesSketchWindow)EditorForm).SketchPad.objectRedefined(obj);
        }

        public virtual void ReleaseObject(FtsObject obj)
        {
            if (obj is FtsConnection)
            {
                RemoveConnection((FtsConnection)obj);

                if (EditorForm != null)
                    ((ErmesSketchWindow)EditorForm).SketchPad.DisplayList.Remove((FtsConnection)obj);
            }
            else
            {
                RemoveObject((FtsGraphicObject)obj);

                if (EditorForm != null)
                    ((ErmesSketchWindow)EditorForm).SketchPad.DisplayList.Remove((FtsGraphicObject)obj);

                obj.Dispose();
            }
        }

        public override void Dispose()
        {
            for (IEnumerator o = objects.Elements(); o.MoveNext(); )
                ((FtsObject)o.Current).Dispose();
            for (IEnumerator c = connections.Elements(); c.MoveNext(); )
                ((FtsObject)c.Current).Dispose();

            base.Dispose();
        }

        /**************************************************************/
        /**************************************************************/

        /// <summary> Fts callback: open the editor associated with this FtsSequenceObject.
        /// If not exist create them else show them.
        /// </summary>
        public override void OpenEditor(int nArgs, FtsAtom[] args)
        {
            if (EditorForm == null)
            {
                Thread thread = new Thread(
                    delegate()
                    {
                        EditorForm = new ErmesSketchWindow(this);
                        ShowEditor();
                    });
                thread.Start();
            }
            else
            {
                Thread thread = new Thread(
                    delegate()
                    {
                        ShowEditor();
                    });
                thread.Start();
            }
        }

        public void StopWaiting()
        {
            if (waitingListener != null)
            {
                waitingListener.FtsActionDone();
                waitingListener = null;
            }
            else if (EditorForm != null)
                ((ErmesSketchWindow)EditorForm).SketchPad.StopWaiting();
        }

        /// <summary> Fts callback: destroy the editor associated with this FtsSequenceObject.</summary>
        public override void DestroyEditor()
        {
            DisposeEditor();
            FireGlobalAtomicAction(false);
        }

        public void ShowObject(FtsGraphicObject obj)
        {
            if (EditorForm != null)
                ((ErmesSketchPad)((ErmesSketchWindow)EditorForm).IEditor).ShowObject(obj);
        }

        public void RedefineStart()
        {
            objects.RemoveAllElements();
            connections.RemoveAllElements();
        }

        public override string Description
        {
            get
            {
                return this.Description;
            }
            set
            {
                this.Description = value;
                if (GraphicListener != null)
                    GraphicListener.Redefined(this);
                if (EditorForm != null)
                    ((ErmesSketchWindow)EditorForm).UpdateTitle();
            }
        }

        public int WX
        {
            set
            {
                windowX = value;
            }
        }

        public int WY
        {
            set
            {
                windowY = value;
            }
        }

        public int WW
        {
            set
            {
                windowWidth = value;
            }
        }

        public int WH
        {
            set
            {
                windowHeight = value;
            }
        }

        public void SetPatcherBounds(int x, int y, int w, int h)
        {
            windowX = x;
            windowY = y;
            windowWidth = w;
            windowHeight = h;

            if (EditorForm != null)
                ((ErmesSketchWindow)EditorForm).PatcherBounds = this;
        }

        public string Message
        {
            set
            {
                FirePatcherHaveMessage(value);
            }
        }

        /////////////////////////////////////////
        public void UpdateData()
        {
            Update();
        }

        // Paste
        internal void StartPaste()
        {
            pasting = true;
        }

        internal void EndPaste()
        {
            pasting = false;
            ((ErmesSketchWindow)EditorForm).SketchPad.EndPaste();
        }

        //
        // Edit And Change Listener
        //
        [NonSerialized()]
        private IFtsPatcherListener listener;

        // Set the patcher data listener 
        public IFtsPatcherListener PatcherListener
        {
            set
            {
                this.listener = value;
            }
        }

        // Reset the patcher data listener 
        public void ResetPatcherListener()
        {
            listener = null;
        }

        public override int NumberOfInlets
        {
            set
            {
                base.NumberOfInlets = value;
                FirePatcherChangedNumberOfInlets(value);
            }
        }

        internal void FirePatcherChangedNumberOfInlets(int ins)
        {
            if (listener != null)
                listener.PatcherChangedNumberOfInlets(this, ins);
        }

        public override int NumberOfOutlets
        {
            set
            {
                base.NumberOfOutlets = value;
                FirePatcherChangedNumberOfOutlets(value);
            }
        }

        internal void FirePatcherChangedNumberOfOutlets(int outs)
        {
            if (listener != null)
                listener.PatcherChangedNumberOfOutlets(this, outs);
        }

        private void FirePatcherChanged()
        {
            if (listener != null)
                listener.PatcherChanged(this);
        }

        private void FirePatcherHaveMessage(string msg)
        {
            if (listener != null)
                listener.PatcherHaveMessage(msg);
        }
    }
}