//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.Collections;

namespace ircam.jmax.toolkit
{
    /// <summary> A class that handles the scroll and zoom values for a group of listeners.</summary>
    public class Geometry
    {
        /// <summary> Constructs a default Geometry object with standard values for
        /// zoom (100%), transposition(0) and inversion fields(false) 
        /// </summary>
        public Geometry()
        {
            xZoomFactor = 1;
            yZoomFactor = 1;
            xTranspose = 0;
            yTranspose = 0;
            xInvertion = false;
            yInvertion = false;
        }

        /// <summary> set the zoom factor for the x coordinates (percentage)</summary>
        public void SetXZoom(int factor)
        {
            float old = xZoomFactor;
            xZoomFactor = factor / (float)100;
            NotifyZoom(xZoomFactor, old);
        }

        public void SetXZoomFactor(float factor)
        {
            xZoomFactor = factor;
        }

        public void IncrXZoom(int delta)
        {
            float old = xZoomFactor;
            float dlt = (float)delta;
            if (xZoomFactor < 1)
                dlt = delta * xZoomFactor;

            xZoomFactor = (xZoomFactor * 100 + dlt) / (float)100;
            if (xZoomFactor < 0.01)
                xZoomFactor = (float)0.01;
            if (xZoomFactor > 10)
                xZoomFactor = (float)10.0;
            NotifyZoom(xZoomFactor, old);
        }

        /// <summary> get the zoom factor for the x coordinates (percentage)</summary>
        public float getXZoom()
        {
            return xZoomFactor;
        }

        /// <summary> get the zoom factor for the y coordinates (percentage)</summary>
        public float getYZoom()
        {
            return yZoomFactor;
        }

        /// <summary> set the zoom factor for the y coordinates (percentage)</summary>
        public void setYZoom(int factor)
        {
            yZoomFactor = factor / 100;
        }

        /// <summary> the x transposition </summary>
        /// <summary> set the transposition amount for the x coordinates (pixels)</summary>
        public int XTransposition
        {
            get
            {
                return xTranspose;
            }
            set
            {
                xTranspose = value;
                NotifyTransposition(value);
            }
        }

        /// <summary> the y transposition </summary>
        /// <summary> set the transposition amount for the y coordinates (pixels)</summary>
        public int YTransposition
        {
            get
            {
                return yTranspose;
            }
            set
            {
                yTranspose = value;
            }
        }

        /// <summary> get the inversion flag for the x coordinates</summary>
        /// <summary> set the inversion flag for the x coordinates</summary>
        public bool XInvertion
        {
            get
            {
                return xInvertion;
            }
            set
            {
                xInvertion = value;
            }
        }

        /// <summary> get the inversion flag for the y coordinates</summary>
        /// <summary> set the inversion flag for the y coordinates</summary>
        public bool YInvertion
        {
            get
            {
                return yInvertion;
            }
            set
            {
                yInvertion = value;
            }
        }

        /// <summary> called by objects that will be informed when the zoom factor changes</summary>
        public void AddZoomListener(IZoomListener listener)
        {
            zoomListeners.Add(listener);
        }

        /// <summary> called by objects that will be informed when the zoom factor changes</summary>
        public void RemoveZoomListener(IZoomListener listener)
        {
            zoomListeners.Remove(listener);
        }

        private void NotifyZoom(float newZoom, float oldZoom)
        {
            IZoomListener aListener;

            for (IEnumerator e = zoomListeners.GetEnumerator(); e.MoveNext(); )
            {
                aListener = (IZoomListener)e.Current;

                aListener.ZoomChanged(newZoom, oldZoom);
            }
        }

        /// <summary> called by objects that will be informed when the transposition factor changes</summary>
        public void AddTranspositionListener(ITranspositionListener listener)
        {
            transpositionListeners.Add(listener);
        }

        /// <summary> remove the listener</summary>
        public void RemoveTranspositionListener(ITranspositionListener listener)
        {
            transpositionListeners.Remove(listener);
        }

        private void NotifyTransposition(int newTransposition)
        {
            ITranspositionListener aListener;

            for (IEnumerator e = transpositionListeners.GetEnumerator(); e.MoveNext(); )
            {
                aListener = (ITranspositionListener)e.Current;

                aListener.TranspositionChanged(newTransposition);
            }
        }

        /*static*/
        public int SizeToMsec(Geometry geometry, int size)
        {
            if (geometry.XInvertion)
            {
                return (int)(geometry.XTransposition - (size) / geometry.getXZoom());
            }
            else
            {
                return (int)((size) / geometry.getXZoom() - geometry.XTransposition);
            }
        }

        //-------  Geometry fields
        internal float xZoomFactor;
        internal float yZoomFactor;
        internal int xTranspose;
        internal int yTranspose;
        internal bool xInvertion;
        internal bool yInvertion;

        internal ArrayList zoomListeners = new ArrayList();
        internal ArrayList transpositionListeners = new ArrayList();
    }
}