//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.Drawing;
using System.Windows.Forms;

namespace ircam.jmax.toolkit
{
    /// <summary> A rubber-banding interaction module.
    /// It communicates the point of selection (eventually, the double-click action), and
    /// takes care of drawing a "selection rectangle" in a 
    /// component, and communicate the result to a GraphicSelectionListener
    /// </summary>
    public class Selecter : InteractionModule, IXORPainter
    {
        /// <summary> Constructor. This class uses a XORHandler to draw the shape
        /// of the rubber banding rectangle
        /// </summary>
        public Selecter(IGraphicSelectionListener theListener) : base()
        {
            itsXORHandler = new XORHandler(this);
            itsListener = theListener;
        }


        /// <summary> overrides InteractionModule.mousePressed()</summary>
        public override void MousePressed(object sender, MouseEventArgs e)
        {
            int x = e.X;
            int y = e.Y;

            if (e.Clicks > 1)
                itsListener.SelectionPointDoubleClicked(x, y);
            InteractionBeginAt(x, y);
            itsListener.SelectionPointChoosen(x, y);
            /*active = true;*/
            InteractionSemaphore.Lock();
        }


        /// <summary> used to set the starting point of the interaction.</summary>
        public override void InteractionBeginAt(int x, int y)
        {
            startSelection = new Point(x, y);
            movingPoint = new Point(x, y);
            itsXORHandler.BeginAt(x, y);
            //UNDONE: Not Implemented Yet.
            //itsRunningG = gc.GraphicDestination;
            active = true;
            InteractionSemaphore.Lock();
        }

        /// <summary> overrides InteractionModule.mouseDragged()</summary>
        public override void MouseDragged(object sender, MouseEventArgs e)
        {
            if (!active)
                return; //!!
            itsXORHandler.MoveTo(e.X, e.Y);
        }

        /// <summary> overrides InteractionModule.mouseReleased()</summary>
        public override void MouseReleased(object sender, MouseEventArgs e)
        {
            if (!active)
            {
                InteractionSemaphore.Unlock();
                return; //abnormal condition
            }
            int x = e.X;
            int y = e.Y;

            itsXORHandler.End();

            //UNDONE: Not Implemented Yet.
            //ReshapeRectangle(ref tempRect, startSelection.X, startSelection.Y, x - startSelection.X, y - startSelection.Y);
            NormalizeRectangle(tempRect);

            itsListener.SelectionChoosen(tempRect.X, tempRect.Y, tempRect.Width, tempRect.Height);
            active = false;
            itsRunningG.Dispose();
            InteractionSemaphore.Unlock();
        }


        protected internal override void UnBindFromProducer()
        {
            base.UnBindFromProducer();
            active = false;
        }

        /// <summary> from the XORPainter interface</summary>
        public void XORErase()
        {
            XORDraw(0, 0);
        }

        /// <summary> from the XOR painter interface. The actual drawing routine</summary>
        public void XORDraw(int dx, int dy)
        {
            //UNDONE: Not implemented Yet.
            //Graphics g = GetGraphics(gc.GraphicDestination);

            //SetColor(g, Color.Gray);
            //g.setXORMode(Color.White); //there's an assumption here on the color of the background.

            movingPoint = new Point(movingPoint.X + dx, movingPoint.Y + dy);

            //ReshapeRectangle(ref tempRect, startSelection.X, startSelection.Y, movingPoint.X - startSelection.X, movingPoint.Y - startSelection.Y);
            NormalizeRectangle(tempRect);

            //if ((tempRect.Width >= 2) || (tempRect.Height >= 2))
            //g.DrawRectangle(GetPen(g), tempRect.X, tempRect.Y, tempRect.Width, tempRect.Height);

            //SetColor(g, Color.Black);
        }

        /// <summary> utility function. Gets rid of the "negative width/lenght" problems
        /// in rectangles.
        /// </summary>
        public static void NormalizeRectangle(Rectangle r)
        {
            /* sets the origin */
            if (r.Width < 0)
                r.X += r.Width;

            if (r.Height < 0)
                r.Y += r.Height;

            /* sets width and height */
            if (r.Width < 0)
                r.Width = -r.Width;
            if (r.Height < 0)
                r.Height = -r.Height;
        }

        //--- Fields
        protected internal IGraphicSelectionListener itsListener;

        protected internal Point startSelection;
        protected internal Point movingPoint;
        internal XORHandler itsXORHandler;

        internal bool active = false;
        internal Rectangle tempRect;
        protected internal Graphics itsRunningG;
    }
}