//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.Drawing;
using System.Windows.Forms;

namespace ircam.jmax.toolkit
{
    /// <summary> An abstract interaction module used to resize a selection of objects.
    /// At the end of a resize operation, it communicates the new position to the listener.
    /// Just redefine the XORDraw method in order to specify what to draw during the resize.
    /// </summary>
    /// <seealso cref="DragListener">
    /// </seealso>
    /// <seealso cref="InteractionModule">
    /// </seealso>
    abstract public class SelectionResizer : InteractionModule, IXORPainter
    {
        /// <summary> constructor. 
        /// It uses a XORHandler to keep track of the drawing positions.
        /// </summary>
        public SelectionResizer(IDragListener theListener)
            : base()
        {
            itsListener = theListener;
            itsXORHandler = new XORHandler(this, XORHandler.ABSOLUTE);
            itsStartingPoint = new Point(0, 0);
        }

        /// <summary> set the direction of movement (constants HORIZONTAL_MOVEMENT  or 
        /// VERTICAL_MOVEMENT, or both)
        /// </summary>
        public int Direction
        {
            set
            {
                movement = value;
            }
        }

        /// <summary> sets the point on which to start the movement</summary>
        public override void InteractionBeginAt(int x, int y)
        {
            itsStartingPoint = new Point(x, y);

            itsXORHandler.BeginAt(x, y);
            InteractionSemaphore.Lock();
        }

        /// <summary> overrides InteractionModule.mouseDragged()</summary>
        public override void MouseDragged(object sender, MouseEventArgs e)
        {

            itsXORHandler.MoveTo(e.X, e.Y);
        }

        /// <summary> overrides InteractionModule.mouseReleased()</summary>
        public override void MouseReleased(object sender, MouseEventArgs e)
        {
            itsXORHandler.End();

            itsListener.DragEnd(e.X, e.Y, e); //this 0 is not generic!
            InteractionSemaphore.Unlock();
        }


        /// <summary> from the XORPainter interface</summary>
        public void XORErase()
        {
            XORDraw(previousX, previousY);
        }


        /// <summary> from the XORPainter interface. The actual drawing function.</summary>
        abstract public void XORDraw(int dx, int dy);


        //---- Fields
        protected internal IDragListener itsListener;
        protected internal XORHandler itsXORHandler;
        protected internal int previousX;
        protected internal int previousY;

        public Point itsStartingPoint;
        public const int HORIZONTAL_MOVEMENT = 1;
        public const int VERTICAL_MOVEMENT = 2;
        protected int movement;
    }
}