# -*- mode: perl; coding: utf-8 -*-
# Keitairc::IrcBuffer
# $Id: IrcBuffer.pm,v 1.18 2008-07-17 15:35:52 morimoto Exp $
# $Source: /home/ishikawa/work/keitairc/tmp/keitairc/lib/Keitairc/IrcBuffer.pm,v $
#
# Copyright (c) 2008 Jun Morimoto <morimoto@mrmt.net>
# This program is covered by the GNU General Public License 2

package Keitairc::IrcBuffer;
use Encode;
use strict;
use warnings;

################################################################
sub new{
	my $proto = shift;
	my $arg = shift;
	my $me = {};

	$me->{history} = $arg->{history};

	# join しているchannelの名称を記録するハッシュ。
	# - cid および name2cid ハッシュに格納されている値は整数。
	# - cid2nameハッシュに格納されている文字列、および name2cid の
	#   ハッシュキーは iso-2022-jp (ないしはそのircチャネルで使われ
	#   ているcharset)のまま。これは歴史的理由でこうするしかない。
	$me->{cid2name} = {};
	$me->{name2cid} = {};

	# join しているtopicの名称を記録するハッシュ
	# charset: perl internal
	$me->{topic} = {};

	$me->{nicks} = {};

	# チャネルの会話内容を記録するハッシュ
	# charset: perl internal
	$me->{buffer} = {};
	$me->{unread} = {};

	# 各チャネルの最終発言時刻
	$me->{mtime} = {};
	# 各チャネルの未読行数
	$me->{unread_lines} = {};

	# chk
	$me->{message_added} = 0;

	# timestamp of last posted message
	$me->{timestamp} = 0;

        bless $me;
}

################################################################
sub add_nick{
	my($me, $cid, $nick, $chop, $realname) = @_;
	$me->{nicks}->{$cid}->{$nick}->{realname} = $realname;
	$me->{nicks}->{$cid}->{$nick}->{chop} = $chop;
}

################################################################
sub list_nick{
	my($me, $cid, $nick, $chop, $realname) = @_;
	keys %{$me->{nicks}->{$cid}};
}

################################################################
sub remove_nick{
	my($me, $cid, $nick) = @_;
	delete $me->{nicks}->{$cid}->{$nick};
}

################################################################
sub get_nick_realname{
	my($me, $cid, $nick) = @_;
	$me->{nicks}->{$cid}->{$nick}->{realname};
}

################################################################
sub op_nick{
	my($me, $cid, $nick) = @_;
	if(defined $me->{nicks}->{$cid}){
		if(defined $me->{nicks}->{$cid}->{$nick}){
			$me->{nicks}->{$cid}->{$nick}->{chop} = 1;
		}
	}
}

################################################################
sub deop_nick{
	my($me, $cid, $nick) = @_;
	if(defined $me->{nicks}->{$cid}){
		if(defined $me->{nicks}->{$cid}->{$nick}){
			$me->{nicks}->{$cid}->{$nick}->{chop} = 0;
		}
	}
}

################################################################
sub get_nick_op{
	my($me, $cid, $nick) = @_;
	if(defined $me->{nicks}->{$cid}){
		if(defined $me->{nicks}->{$cid}->{$nick}){
			return $me->{nicks}->{$cid}->{$nick}->{chop};
		}
	}
}

################################################################
sub channels{
	my $me = shift;
	map {
		$_
	}(sort
	  {
		  $me->mtime($b) <=> $me->mtime($a)
	  } keys %{$me->{cid2name}});
}

################################################################
sub now{
	my ($sec, $min, $hour) = localtime(time);
	sprintf('%02d:%02d', $hour, $min);
}

################################################################
# 引数の $name (チャネル名) は、生の iso-2022-jp のまま、ないしは irc
# チャネルで扱われている charset のまま扱うこと。こういう慣習なので
# しょうがない。
sub name2cid{
	my($me, $name) = @_;
	$name =~ tr/A-Z[\\]^/a-z{|}~/;

	unless(defined $me->{name2cid}->{$name}){
		my $cid = (sort { $b - $a } (keys %{$me->{cid2name}}))[0];
		$cid++;
		$me->{cid2name}->{$cid} = $name;
		$me->{name2cid}->{$name} = $cid;
	}

	$me->{name2cid}->{$name};
}

################################################################
sub cid2name{
	my($me, $cid) = @_;
	$me->{cid2name}->{$cid};
}

################################################################
sub part{
	my($me, $cid) = @_;
	delete $me->{cid2name}->{$cid};
	delete $me->{name2cid}->{$cid};
	delete $me->{topic}->{$cid};
	delete $me->{nicks}->{$cid};
	delete $me->{buffer}->{$cid};
	delete $me->{unread}->{$cid};
}

################################################################
sub join{
	my ($me, $name) = @_;
	my $cid = $me->name2cid($name);
	$me->{cid2name}->{$cid} = $name;
}

################################################################
sub mtime{
	my($me, $cid) = @_;
	$me->{mtime}->{$cid} || 0;
}

################################################################
sub message_added{
	my($me, $v) = @_;
	if(defined $v){
		$me->{message_added} = $v;
	}
	$me->{message_added};
}

################################################################
sub unread_lines{
	my($me, $cid) = @_;
	$me->{unread_lines}->{$cid};
}

################################################################
sub unread{
	my($me, $cid) = @_;
	$me->{unread}->{$cid};
}

################################################################
sub clear_unread{
	my($me, $cid) = @_;
	delete $me->{unread}->{$cid};
	$me->{unread_lines}->{$cid} = 0;
}

################################################################
sub topic{
	my($me, $cid, $topic) = @_;
	if(defined $topic){
		$me->{topic}->{$cid} = $topic;
	}
	$me->{topic}->{$cid};
}

################################################################
sub buffer{
	my($me, $cid) = @_;
	$me->{buffer}->{$cid};
}

################################################################
# 引数の $msg の charset は perl internal
# $channel は iso-2022-jp または irc channel specific
sub add_message{
	my($me, $cid, $message, $who) = @_;

	if(defined($who) && length($who)){
		$message = sprintf('%s %s> %s', now(), $who, $message);
	}else{
		$message = sprintf('%s %s', now(), $message);
	}

	{
		my @tmp;
		if(defined($me->{buffer}->{$cid})){
			@tmp = split("\n", $me->{buffer}->{$cid});
		}
		push @tmp, $message;

		if(@tmp > $me->{history}){
			$me->{buffer}->{$cid} =
				CORE::join("\n", splice(@tmp, -$me->{history}));
		}else{
			$me->{buffer}->{$cid} = CORE::join("\n", @tmp);
		}
	}

	{
		my @tmp;
		if(defined($me->{unread}->{$cid})){
	@tmp = split("\n", $me->{unread}->{$cid});
		}
		push @tmp, $message;

		if(@tmp > $me->{history}){
			$me->{unread}->{$cid} =
				CORE::join("\n", @tmp[1 .. $me->{history}]);
		}else{
			$me->{unread}->{$cid} = CORE::join("\n", @tmp);
		}

		$me->{unread_lines}->{$cid} = scalar(@tmp);
	}

	if($me->{unread_lines}->{$cid} > $me->{history}){
		$me->{unread_lines}->{$cid} = $me->{history};
	}

	if($me->{cid2name}->{$cid} eq '*console*') {
		$me->{mtime}->{$cid} = -1;
	} else {
		$me->{mtime}->{$cid} = time;
	}
}

################################################################
# チャネル名称を短かくする
# 返り値は Perl internal code
sub compact_channel_name{
	my $me = shift;
	my $cid = shift;
	my $name = $me->cid2name($cid);

	# この後の置換処理は、チャネル名の文字列が実際に運用されている
	# charset で行う必要がある。日本語が用いられている従来の irc チャ
	# ネルであれば iso-2022-jp-1 だし、ないしは utf8 のまま。
	$name = decode($::cf->irc_charset(), $name);

	# #name:*.jp を %name に
	if($name =~ s/:\*\.jp$//){
		$name =~ s/^#/%/;
	}

	# 末尾の単独の @ は取る (plumプラグインのmulticast.plm対策)
	# @ の後に空白が入ることもあるようだ。理由はわからない。
	$name =~ s/\@\s*$//;
	$name;
}

################################################################
sub simple_escape{
	my $me = shift;
        local($_) = shift;
	if(defined $_){
		s/&/&amp;/g;
		s/>/&gt;/g;
		s/</&lt;/g;
	}
        $_;
}

################################################################
sub colorize{
	my $me = shift;
        local($_) = shift;

	my %ct = (
		1 => 'Black',
		2 => '#000080', # Navy Blue
		3 => 'Green',
		4 => 'Red',
		5 => 'Maroon',
		6 => 'Purple',
		7 => 'Olive',
		8 => 'Yellow',
		9 => '#32cd32', # Lime Green
		10 => 'Teal',
		11 => 'Aqua',
		12 => '#4169e1', # Royal Blue
		13 => '#ff69b4', # Hot Pink
		14 => '#a9a9a9', # Dark Gray
		15 => '#d3d3d3', # Light Gray
		16 => 'White');
	my $colored = 0;

	do{
		if($colored){
			s|\x03(\d{1,2})|sprintf('</font><font color="%s">', $ct{0+$1})|e;
			if(s|\x03|</font>|){
				$colored = 0;
			}
		}else{
			if(s|\x03(\d{1,2})|sprintf('<font color="%s">', $ct{0+$1})|e){
				$colored = 1;
			}
		}
	}while(m|\x03\d{1,2}|);

	if($colored){
		$_ .= '</font>';
	}

	$_;
}

################################################################
# 同一秒間の連続発言を防ぐためのチェック。
#
# 前回 update_timestamp() が呼ばれた時刻と同じ時刻に
# 再度 update_timestamp() が呼ばれたら 0 を返す。
#
# 前回 update_timestamp() が呼ばれた時刻と異なる時刻に
# 再度 update_timestamp() が呼ばれたら 1 を返す。
#
sub update_timestamp{
	my $me = shift;
	my $time = time;

	if($me->{timestamp} != $time){
		$me->{timestamp} = $time;
		return 1;
	}

	return 0;
}

1;
