/**
 * @file tinyrand.c
 * @author Shinichiro Nakamura
 */

/*
 * ===============================================================
 * "Natural Tiny Basic (NT-Basic)"
 * "A tiny BASIC interpreter"
 * ---------------------------------------------------------------
 * Random number module
 * ===============================================================
 * Copyright (c) 2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "tinyrand.h"

/**
 * @brief Get the bit position.
 * @param N Bit number.
 * @return Bit position.
 */
#define BP(N)   (1 << ((N) - 1))

int tinyrand_init(tinyrand_t *p, const unsigned int seed)
{
    if (seed == 0) {
        return -1;
    }

    p->seed = seed;
    p->reg = seed;
    return 0;
}

static int tinyrand_update(tinyrand_t *p)
{
    const unsigned int mask = BP(1) | BP(3) | BP(4);
    const unsigned int bits = BP(24);
    if ((p->reg & bits) == bits) {
        p->reg = ((p->reg ^ mask) << 1) | 1;
        return 1;
    } else {
        p->reg = p->reg << 1;
        return 0;
    }
}

int tinyrand_pop(tinyrand_t *p, const unsigned char nbits, unsigned int *value)
{
    int i;

    /*
     * Init value.
     */
    *value = 0;

    /*
     * Do nothing if the input parameter is invalid.
     */
    if (nbits == 0) {
        return -1;
    }

    /*
     * Generate nbits random value.
     */
    for (i = 0; i < nbits; i++) {
        *value |= tinyrand_update(p) ? (1 << i) : 0;
    }
    return 0;
}

