//
//  ICSCalendar.m
//  iCal to KS2
//
//  Created by FUJIDANA on 05/05/11.
//  Copyright 2005 FUJIDANA. All rights reserved.
//
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#define ICalVersionNumber10_4 800 

#import "ICSCalendar.h"
#import "ICSEvent.h"
#import "ICSTodo.h"

@interface ICSCalendar (Private)

+ (NSColor *)colorWithString:(NSString *)string;

@end

#pragma mark -

@implementation ICSCalendar

#pragma mark class methods

+ (NSArray *)calendars
{
	NSWorkspace		*workspace	= [NSWorkspace sharedWorkspace];
	NSString		*iCalPath	= [workspace fullPathForApplication:@"iCal"];
	if (iCalPath == nil) {
		return nil;
	}
	
	NSDictionary	*iCalPlist			= [NSDictionary dictionaryWithContentsOfFile:[[iCalPath stringByAppendingPathComponent:@"Contents"]
		stringByAppendingPathComponent:@"Info.plist"]];
	int				iCalBundleVersion	= [[iCalPlist objectForKey:@"CFBundleVersion"] intValue];
	
	if (iCalBundleVersion >= ICalVersionNumber10_4) {
		NSMutableArray	*calendars;
		
		NSString		*iCalNodesPath		= [[[[NSHomeDirectory() 
			stringByAppendingPathComponent:@"Library"]
			stringByAppendingPathComponent:@"Application Support"]
			stringByAppendingPathComponent:@"iCal"]
			stringByAppendingPathComponent:@"nodes.plist"];
		NSDictionary	*nodesDict			= [NSDictionary dictionaryWithContentsOfFile:iCalNodesPath];
		NSArray			*nodesArray			= [nodesDict objectForKey:@"List"];
		NSEnumerator	*nodesEnumerator	= [nodesArray objectEnumerator];
		
						calendars			= [NSMutableArray arrayWithCapacity:[nodesArray count]];
		
		NSDictionary	*dictionaryInNodes;
		
		while (dictionaryInNodes = [nodesEnumerator nextObject]) {
			NSString	*sourceKey		= [dictionaryInNodes objectForKey:@"SourceKey"];
			NSArray		*subnodesArray	= [dictionaryInNodes objectForKey:@"Subnodes"];
			
			if (sourceKey) {
				ICSCalendar	*calendar = [ICSCalendar calendarWithDictionaryInNodes:dictionaryInNodes];
				if (calendar) {
					[calendars addObject:calendar];
				}
			} else if (subnodesArray) {
				NSEnumerator *subnodesEnumerator = [subnodesArray objectEnumerator];
				
				while (dictionaryInNodes = [subnodesEnumerator nextObject]) {
					sourceKey	= [dictionaryInNodes objectForKey:@"SourceKey"];
					if (sourceKey) {
						ICSCalendar *calendar = [ICSCalendar calendarWithDictionaryInNodes:dictionaryInNodes];
						if (calendar) {
							[calendars addObject:calendar];
						}
					}
				}
			}
		}
		return calendars;
	} else {
		NSString		*iCalCalendarFolderPath	= [[NSHomeDirectory() 
			stringByAppendingPathComponent:@"Library"]
			stringByAppendingPathComponent:@"Calendars"];
		NSString		*iCalSourcePlistPath	= [[[NSHomeDirectory()
			stringByAppendingPathComponent:@"Library"]
			stringByAppendingPathComponent:@"Preferences"]
			stringByAppendingPathComponent:@"com.apple.iCal.sources.plist"];
		NSDictionary	*sourcesPlist			= [NSDictionary dictionaryWithContentsOfFile:iCalSourcePlistPath];
		NSDictionary	*sourcesView			= [sourcesPlist objectForKey:@"SourcesView"];
		NSEnumerator	*keyEnumerator			= [sourcesView keyEnumerator];
		NSString		*keyString;
		
		NSMutableArray	*calendars				= [NSMutableArray arrayWithCapacity:[sourcesView count]];
		
		while (keyString = [keyEnumerator nextObject]) {
			id			object				= [sourcesView objectForKey:keyString];
			NSString	*description		= [object objectForKey:@"Description"];
			if (description) {
				NSColor		*aColor;
				NSString	*colorString	= [object objectForKey:@"Color"];
				if (colorString) {
					if ([colorString isEqualToString:@"blue"]) {
						aColor = [NSColor colorWithCalibratedRed:(float)0x02 / 0xFF
														   green:(float)0x52 / 0xFF
															blue:(float)0xD4 / 0xFF
														   alpha:(float)0xFF / 0xFF];
					} else if ([colorString isEqualToString:@"orange"]) {
						aColor = [NSColor colorWithCalibratedRed:(float)0xF5 / 0xFF
														   green:(float)0x78 / 0xFF
															blue:(float)0x02 / 0xFF
														   alpha:(float)0xFF / 0xFF];
					} else if ([colorString isEqualToString:@"teal"]) {
						aColor = [NSColor colorWithCalibratedRed:(float)0xB0 / 0xFF
														   green:(float)0x27 / 0xFF
															blue:(float)0xAE / 0xFF
														   alpha:(float)0xFF / 0xFF];
					} else if ([colorString isEqualToString:@"red"]) {
						aColor = [NSColor colorWithCalibratedRed:(float)0xE5 / 0xFF
														   green:(float)0x17 / 0xFF
															blue:(float)0x17 / 0xFF
														   alpha:(float)0xFF / 0xFF];
					} else if ([colorString isEqualToString:@"green"]) {
						aColor = [NSColor colorWithCalibratedRed:(float)0x2C / 0xFF
														   green:(float)0xA1 / 0xFF
															blue:(float)0x0B / 0xFF
														   alpha:(float)0xFF / 0xFF];
					} else if ([colorString isEqualToString:@"purple"]) {
						aColor = [NSColor colorWithCalibratedRed:(float)0x49 / 0xFF
														   green:(float)0x2B / 0xFF
															blue:(float)0xA1 / 0xFF
														   alpha:(float)0xFF / 0xFF];
					}
				}
				
				NSString	*filename			= [description stringByAppendingString:@".ics"];
				NSString	*iCalCalendarPath	= [iCalCalendarFolderPath stringByAppendingPathComponent:filename];
				ICSCalendar	*calendar			= [ICSCalendar calendarWithIdentifier:keyString
																		   isWritable:YES // <- omit to check
																				title:description
																				color:aColor
																				notes:nil
																	   contentsOfFile:iCalCalendarPath];
				[calendars addObject:calendar];
			}
		}
		return calendars;
	}
}

+ (id)calendarWithIdentifier:(NSString *)anIdentifier isWritable:(BOOL)flag title:(NSString *)aTitle color:(NSColor *)aColor notes:(NSString *)aNotes contentsOfFile:(NSString *)aPath
{
	return [[[ICSCalendar alloc] initWithIdentifier:anIdentifier
										 isWritable:flag
											  title:aTitle
											  color:aColor
											  notes:aNotes
									 contentsOfFile:aPath] autorelease];
}

+ (id)calendarWithDictionaryInNodes:(NSDictionary *)dictionaryInNodes
{
	return [[[ICSCalendar alloc] initWithDictionaryInNodes:dictionaryInNodes] autorelease];
}

+ (NSColor *)colorWithString:(NSString *)string
{
	NSColor *aColor;
	
	if (string && [string length] == 9) {
		unsigned    red, green, blue, alpha;
		NSString	*colorString	= [NSString stringWithFormat:@"0x%@ 0x%@ 0x%@ 0x%@", 
			[string substringWithRange:NSMakeRange(1, 2)],
			[string substringWithRange:NSMakeRange(3, 2)],
			[string substringWithRange:NSMakeRange(5, 2)],
			[string substringWithRange:NSMakeRange(7, 2)]];
			
		NSScanner	*scanner = [NSScanner scannerWithString:colorString];
		if ([scanner scanHexInt:&red] && 
			[scanner scanHexInt:&green] &&
			[scanner scanHexInt:&blue] &&
			[scanner scanHexInt:&alpha]) {
			
			aColor = [NSColor colorWithCalibratedRed:(float)red / 0xff
											   green:(float)green / 0xff
												blue:(float)blue / 0xff
											   alpha:(float)alpha / 0xff];
		}
	}
	return aColor;
}

#pragma mark initialization and deallocation
- (id)initWithIdentifier:(NSString *)anIdentifier isWritable:(BOOL)flag title:(NSString *)aTitle color:(NSColor *)aColor notes:(NSString *)aNotes contentsOfFile:(NSString *)aPath
{
	self = [super init];
	
	if (self != nil) {
		identifier	= [anIdentifier copy];
		isWritable	= flag;
		path		= [aPath copy];
		
		[self setTitle:aTitle];
		[self setColor:aColor];
		[self setNotes:aNotes];
		
		if ([self reload] == NO) {
			[self release];
			return nil;
		}
	}
	return self;
}

- (id)initWithDictionaryInNodes:(NSDictionary *)dictionaryInNodes
{
	if (dictionaryInNodes == nil) {
		return nil;
	}
	
	NSString	*sourceKey				= [dictionaryInNodes objectForKey:@"SourceKey"];
	if (sourceKey == nil) {
		return nil;
	}
	
	NSString	*calendarFolderPath		= [[[[[NSHomeDirectory() 
		stringByAppendingPathComponent:@"Library"]
		stringByAppendingPathComponent:@"Application Support"]
		stringByAppendingPathComponent:@"iCal"]
		stringByAppendingPathComponent:@"Sources"]
		stringByAppendingPathComponent:[sourceKey stringByAppendingPathExtension:@"calendar"]];
	NSString	*calendarDataPath		= [calendarFolderPath stringByAppendingPathComponent:@"corestorage.ics"];
	NSString	*calendarInfoPlistPath	= [calendarFolderPath stringByAppendingPathComponent:@"Info.plist"];
	NSDictionary	*calendarInfoPlist	= [NSDictionary dictionaryWithContentsOfFile:calendarInfoPlistPath];
	
	// get identifier
	NSString	*anIdentifier			= [dictionaryInNodes objectForKey:@"Key"];
	// get color
	NSString	*colorString			= [dictionaryInNodes objectForKey:@"ThemeColor"];
	NSColor		*themeColor;
	if (colorString) {
		themeColor = [ICSCalendar colorWithString:colorString];
	}
	// get title
	NSString	*aTitle					= [calendarInfoPlist objectForKey:@"Title"];
	// get description
	NSString	*aNotes					= [calendarInfoPlist objectForKey:@"Notes"];
	// get isWritable
	NSString	*aType					= [calendarInfoPlist objectForKey:@"Type"];
	BOOL		flag = NO;
	if (aType != nil) {
		flag = [aType isEqualToString:@"com.apple.ical.sources.naivereadwrite"];
	}
	
	return [self initWithIdentifier:anIdentifier
						 isWritable:flag
							  title:aTitle
							  color:themeColor
							  notes:aNotes
					 contentsOfFile:calendarDataPath];
}


- (void) dealloc {
	[identifier release];
	[events	release];
	[todos release];
	[path release];
	
	[self setTitle:nil];
	[self setColor:nil];
	[self setNotes:nil];
	
	[super dealloc];
}

#pragma mark accessor methods
- (NSMutableArray *)events
{
	return events;
}

- (NSMutableArray *)todos
{
	return todos;
}

- (NSString *)identifier
{
	return identifier;
}

- (BOOL)isWritable
{
	return isWritable;
}

- (NSString *)title
{
	return title;
}
- (void)setTitle:(NSString *)value
{
	if (title != value) {
		[title release];
		title = [value copy];
	}
}

- (NSColor *)color
{
	return color;
}
- (void)setColor:(NSColor *)value
{
	if (color != value) {
		[color release];
		color = [value copy];
	}
}

- (NSString *)notes
{
	return notes;
}
- (void)setNotes:(NSString *)value
{
	if (notes != value) {
		[notes release];
		notes = [value copy];
	}
}

#pragma mark other methods
- (BOOL)reload
{
	[events release];
	[todos release];
	
	events	= [[NSMutableArray alloc] initWithCapacity:64];
	todos	= [[NSMutableArray alloc] initWithCapacity:8];
	
	// NSString	*icsString	= [NSString stringWithContentsOfFile:path encoding:NSUTF8StringEncoding error:nil];
	NSString	*icsString	= [[[NSString alloc] initWithData:[NSData dataWithContentsOfFile:path]
													 encoding:NSUTF8StringEncoding] autorelease];
	NSScanner	*scanner	= [NSScanner scannerWithString:icsString];
	
	[scanner setCharactersToBeSkipped:[NSCharacterSet whitespaceCharacterSet]];
	
	NSString *icsString2, *calendarItemString;
	[scanner scanUpToString:@"BEGIN:VCALENDAR\r\n" intoString:nil];
	
	if ([scanner isAtEnd] == NO &&
		[scanner scanString:@"BEGIN:VCALENDAR\r\n" intoString:nil] &&
		[scanner scanUpToString:@"END:VCALENDAR\r\n" intoString:&icsString2] &&
		[scanner scanString:@"END:VCALENDAR\r\n" intoString:nil]) {
		
		scanner = [NSScanner scannerWithString:icsString2];
		[scanner setCharactersToBeSkipped:[NSCharacterSet whitespaceCharacterSet]];
		
		while ([scanner isAtEnd] == NO) {
			NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
			
			[scanner scanUpToString:@"BEGIN:" intoString:nil];
			if ([scanner scanString:@"BEGIN:VEVENT\r\n" intoString:nil] &&
				[scanner scanUpToString:@"END:VEVENT\r\n" intoString:&calendarItemString] &&
				[scanner scanString:@"END:VEVENT\r\n" intoString:nil]) {
				
				ICSEvent *event = [[[ICSEvent alloc] initWithICalendarRepresentation:calendarItemString] autorelease];
				if (event) {
					[events addObject:event];
				}
			} else if ([scanner scanString:@"BEGIN:VTODO\r\n" intoString:nil] &&
				[scanner scanUpToString:@"END:VTODO\r\n" intoString:&calendarItemString] &&
				[scanner scanString:@"END:VTODO\r\n" intoString:nil]) {
				
				ICSTodo *todo = [[[ICSTodo alloc] initWithICalendarRepresentation:calendarItemString] autorelease];
				if (todo) {
					[todos addObject:todo];
				}
			} else {
				[scanner scanUpToString:@"\r\n" intoString:nil];
			}
			[pool release];
		}
		return YES;
	} else { // when the file does not start with "BEGIN:VCALENDAR\r\n" or end with "END:VCALENDAR\r\n"...
		return NO;
	}
}

@end
