//
//  ICSEvent.m
//  iCal to KS2
//
//  Created by FUJIDANA on 05/05/11.
//  Copyright 2005 FUJIDANA. All rights reserved.
//
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#import "ICSEvent.h"
#import "NSCalendarDate+ICalendarSupport.h"

#define kNumberOfSearchingNextMonth 6
#define kNumberOfSearchingNextYear 8

@interface ICSEvent (Private)

+ (int)dayOfWeekFromICSWeekdayString:(NSString *)value;
+ (BOOL)weekOfMonth:(int *)ordwk dayOfWeek:(int *)weekday fromICSByDayString:(NSString *)string;
+ (void)days:(int *)daysPointer hours:(int *)hoursPointer minites:(int *)minutesPointer seconds:(int *)secondsPointer fromDuration:(NSString *)duration;
//+ (NSCalendarDate *)dateNextToDate:(NSCalendarDate *)aDate frequency:(NSString *)freq interval:(int)interval byDay:(NSString *)byDay byMonthDay:(NSString *)byMonthDay byMonth:(NSString *)byMonth;

- (NSDictionary *)dictionaryOfRecurrence;
+ (NSArray *)datesByExpandingDailyEventWithOriginalDate:(NSCalendarDate *)originalDate interval:(int)interval count:(int)count until:(NSDate *)until;
+ (NSArray *)datesByExpandingWeeklyEventWithOriginalDate:(NSCalendarDate *)originalDate byDay:(NSString *)byDay wkst:(int)wkst interval:(int)interval count:(int)count until:(NSDate *)until;
+ (NSArray *)datesByExpandingMonthlyEventWithOriginalDate:(NSCalendarDate *)originalDate byDay:(NSString *)byDay byMonthDay:(NSString *)byMonthDay interval:(int)interval count:(int)count until:(NSDate *)until;
+ (NSArray *)datesByExpandingYearlyEventWithOriginalDate:(NSCalendarDate *)originalDate byDay:(NSString *)byDay byMonthDay:(NSString *)byMonthDay byMonth:(NSString *)byMonth interval:(int)interval count:(int)count until:(NSDate *)until;

@end

#pragma mark -

@implementation ICSEvent

#pragma mark initialization and deallocation

- (id)initWithICalendarRepresentation:(NSString *)string
{
	self = [super init];
	
	if (self != nil) {
		NSDictionary	*rootDict	= [ICSCalendarItem dictionaryWithRFC2425Representation:string];
		NSDictionary	*subDict;
		NSString		*valueTypeString, *tzidString, *valueString;
		//NSDictionary	*key;
		
		// -- set unique identifier (UID). value: text --
		if ((subDict		= [rootDict objectForKey:@"UID"]) &&
			(valueString	= [subDict objectForKey:@"_value"])) {
			
			identifier = [valueString copy];
		}
		
		// -- set summary (SUMMARY). value: text. param: altrepparam, languageparam (optional). --
		if ((subDict		= [rootDict objectForKey:@"SUMMARY"]) && 
			(valueString	= [subDict objectForKey:@"_value"])) {
			
			[self setSummary:[ICSCalendarItem stringWithEscapedString:valueString]];
		}
		
		// -- set description (DESCRIPTION). value: text. param: altrepparam, languageparam (optional). --
		if ((subDict		= [rootDict objectForKey:@"DESCRIPTION"]) &&
			(valueString	= [subDict objectForKey:@"_value"])) {
			
			[self setNotes:[ICSCalendarItem stringWithEscapedString:valueString]];
		}
		
		// -- set sequence (SEQUENCE). value: integer. --
		if ((subDict		= [rootDict objectForKey:@"SEQUENCE"]) && 
			(valueString	= [subDict objectForKey:@"_value"])) {
			
			[self setSequence:[valueString intValue]];
		}
		
		// -- set date-time stamp (DTSTAMP). value: date-time. param: time zone is not allowed. --
		if ((subDict		= [rootDict objectForKey:@"DTSTAMP"]) &&
			(valueString	= [subDict objectForKey:@"_value"])) {
			
			[self setStampDate:[NSCalendarDate dateWithICalendarDateTimeString:valueString
																	  timeZone:nil]];
		}
		
		// -- set URL (URL). value: uri. --
		if ((subDict		= [rootDict objectForKey:@"URL"]) &&
			(valueString	= [subDict objectForKey:@"_value"])) {
			
			[self setUrl:valueString];
		}
		
		// -- set start date (DTSTART). value: dtstval (date-time/date). param: VALUE=(DATE-TIME | DATE), tzidparam --
		if ((subDict		= [rootDict objectForKey:@"DTSTART"]) &&
			(valueString	= [subDict objectForKey:@"_value"])) {
			
			valueTypeString	= [subDict objectForKey:@"VALUE"];
			tzidString		= [subDict objectForKey:@"TZID"];
			if (valueTypeString && [valueTypeString isEqualToString:@"DATE"]) {
				[self setAllDayEvent:YES];
				[self setStartDate:[NSCalendarDate dateWithICalendarDateString:valueString]];
			} else {
				[self setStartDate:[NSCalendarDate dateWithICalendarDateTimeString:valueString
																	timeZoneString:tzidString]];
			}
		}
		// -- set end date (DTEND). value: dtstval (date-time/date). param: VALUE=(DATE-TIME | DATE), tzidparam --
		// -- or, set end date (DURATION) --
		if ((subDict		= [rootDict objectForKey:@"DTEND"]) &&
			(valueString	= [subDict objectForKey:@"_value"])) {
		
			valueTypeString	= [subDict objectForKey:@"VALUE"];
			tzidString		= [subDict objectForKey:@"TZID"];
			if (valueTypeString && [valueTypeString isEqualToString:@"DATE"]) {
				NSTimeInterval timeInterval = [[NSCalendarDate dateWithICalendarDateString:valueString] timeIntervalSinceDate:[self startDate]];
				[self setDuration:timeInterval];
			} else {
				NSTimeInterval timeInterval = [[NSCalendarDate dateWithICalendarDateTimeString:valueString
																				timeZoneString:tzidString] timeIntervalSinceDate:[self startDate]];
				[self setDuration:timeInterval];
			}
		}
		if ((subDict = [rootDict objectForKey:@"DURATION"]) && 
				   (valueString = [subDict objectForKey:@"_value"])) {

			int days, hours, minutes, seconds;
			[ICSEvent days:&days
					 hours:&hours
				   minites:&minutes
				   seconds:&seconds
			  fromDuration:valueString];
			  
			[self setDuration:(double)(seconds + 60 * (minutes + 60 * (hours + 24 * days)))];
		}
		
		// -- set recurrence (RRULE) --
		if ((subDict		= [rootDict objectForKey:@"RRULE"]) &&
			(valueString	= [subDict objectForKey:@"_value"])) {
			
			[self setRecurrence:valueString];
		}
		
		// -- set excluded dates (EXDATE) --
		if ((subDict		= [rootDict objectForKey:@"EXDATE"]) &&
			(valueString	= [subDict objectForKey:@"_value"])) {
		
			valueTypeString	= [subDict objectForKey:@"VALUE"];
			tzidString		= [subDict objectForKey:@"TZID"];
			
			NSMutableArray	*tempArray	= [NSMutableArray arrayWithCapacity:1];
			NSArray			*strArray	= [valueString componentsSeparatedByString:@","];
			NSEnumerator	*enumerator	= [strArray objectEnumerator];
			NSString		*currentString;
			NSCalendarDate	*tempDate;
			if (valueTypeString && [valueTypeString isEqualToString:@"DATE"]) {
				while (currentString = [enumerator nextObject]) {
					tempDate = [NSCalendarDate dateWithICalendarDateString:currentString];
					if (tempDate) {
						[tempArray addObject:tempDate];
					}
				}
			} else {
				while (currentString = [enumerator nextObject]) {
					tempDate = [NSCalendarDate dateWithICalendarDateTimeString:currentString
					 timeZoneString:tzidString];
					 if (tempDate) {
						[tempArray addObject:tempDate];
					}
				}
			}
			[self setExcludedDates:[NSArray arrayWithArray:tempArray]];
		}
		// -- set status (STATUS) --
		if ((subDict		= [rootDict objectForKey:@"STATUS"]) &&
			(valueString	= [subDict objectForKey:@"_value"])) {

			int theStatus = AHKNoneStatus;
			if ([valueString isEqualToString:@"CANCELLED"] ) {
				theStatus = AHKCancelledStatus;
			} else if ([valueString isEqualToString:@"CONFIRMED"]) {
				theStatus = AHKConfirmedStatus;
			} else if ([valueString isEqualToString:@"TENTATIVE"]) {
				theStatus = AHKTentativeStatus;
			}
			[self setStatus:theStatus];
		}
		
		// -- set location (LOCATION) --
		if ((subDict		= [rootDict objectForKey:@"LOCATION"]) &&
			(valueString	= [subDict objectForKey:@"_value"])) {
		
			[self setLocation:[ICSCalendarItem stringWithEscapedString:valueString]];
		}
	}
	return self;
	/*
	{
		NSArray			*array		= [ICSCalendarItem arraysWithRFC2425Representation:string];
		NSEnumerator	*enumerator	= [array objectEnumerator];
		NSDictionary	*dictionary;
		NSString		*name, *value;
		NSDictionary	*param;
		while (dictionary = [enumerator nextObject]) {
			name	= [dictionary objectForKey:@"name"];
			param	= [dictionary objectForKey:@"param"];
			value	= [dictionary objectForKey:@"value"];
			
			if (name == nil) {
				continue;
				
			// -- set unique identifier (UID). value: text --
			} else if ([name isEqualToString:@"UID"]) {
				identifier = [value copy];
			
			// -- set summary (SUMMARY). value: text. param: altrepparam, languageparam (optional). --
			} else if ([name isEqualToString:@"SUMMARY"]) {
				[self setSummary:value];
			
			// -- set description (DESCRIPTION). value: text. param: altrepparam, languageparam (optional). --
			} else if ([name isEqualToString:@"DESCRIPTION"]) {
				[self setNotes:value];
			
			// -- set sequence (SEQUENCE). value: integer. --
			} else if ([name isEqualToString:@"SEQUENCE"]) {
				[self setSequence:[value intValue]];
			
			// -- set date-time stamp (DTSTAMP). value: date-time. param: time zone is not allowed. --
			} else if ([name isEqualToString:@"DTSTAMP"]) {
				[self setStampDate:[NSCalendarDate dateWithICalendarDateTimeString:value
																		  timeZone:nil]];
				
			// -- set URL (URL). value: uri. --
			} else if ([name isEqualToString:@"URL"]) {
				[self setUrl:value];
				
			// -- set start date (DTSTART). value: dtstval (date-time/date). param: VALUE=(DATE-TIME | DATE), tzidparam --
			} else if ([name isEqualToString:@"DTSTART"]) {
				if (param && [[param objectForKey:@"VALUE"] isEqualToString:@"DATE"]) {
					[self setAllDayEvent:YES];
					[self setStartDate:[NSCalendarDate dateWithICalendarDateString:value]];
				} else if (param && [param objectForKey:@"TZID"]) {
					[self setStartDate:[NSCalendarDate dateWithICalendarDateTimeString:value
																		timeZoneString:[param objectForKey:@"TZID"]]];
				}
			
			// -- set end date (DTEND). value: dtstval (date-time/date). param: VALUE=(DATE-TIME | DATE), tzidparam --
			} else if ([name isEqualToString:@"DTEND"]) {
				if (param && [[param objectForKey:@"VALUE"] isEqualToString:@"DATE"]) {
					[self setAllDayEvent:YES];
					[self setEndDate:[NSCalendarDate dateWithICalendarDateString:value]];
				} else if (param && [param objectForKey:@"TZID"]) {
					[self setEndDate:[NSCalendarDate dateWithICalendarDateTimeString:value
																	  timeZoneString:[param objectForKey:@"TZID"]]];
				}
			
			// -- set end date (DURATION) --
			} else if ([name isEqualToString:@"DURATION"]) {
				int days, hours, minutes, seconds;
				[ICSEvent days:&days
						 hours:&hours
					   minites:&minutes
					   seconds:&seconds
				  fromDuration:value];
				
				if ([self startDate]) {
					NSCalendarDate *aDate = [[self startDate] dateByAddingYears:0
																		 months:0
																		   days:days
																		  hours:hours
																		minutes:minutes 
																		seconds:seconds];
					[self setEndDate:aDate];
				}
			
			// -- set recurrence (RRULE) --
			} else if ([name isEqualToString:@"RRULE"]) {
				[self setRecurrence:value];
				
			// -- set excluded dates (EXDATE) --
			} else if ([name isEqualToString:@"EXDATE"]) {
				NSMutableArray	*tempArray	= [NSMutableArray arrayWithCapacity:2];
				
				NSArray			*strArray	= [value componentsSeparatedByString:@","];
				NSEnumerator	*enumerator	= [strArray objectEnumerator];
				NSString		*string;
				
				if (param && [[param objectForKey:@"VALUE"] isEqualToString:@"DATE"]) {
					while (string = [enumerator nextObject]) {
						NSCalendarDate *date = [NSCalendarDate dateWithString:value 
															   calendarFormat:@"%Y%m%d"];
						if (date) [tempArray addObject:date];
					}
				} else if (param && [param objectForKey:@"TZID"]) {
					while (string = [enumerator nextObject]) {
						string = [NSString stringWithFormat:@"%@ %@", string, [param objectForKey:@"TZID"]];
						NSCalendarDate *date = [NSCalendarDate dateWithString:string
															   calendarFormat:@"%Y%m%dT%H%M%S %Z"];
						if (date) [tempArray addObject:date];
					}
				}
				
				[self setExcludedDates:tempArray];
				
			// -- set status (STATUS) --
			} else if ([name isEqualToString:@"STATUS"]) {
				int theStatus = AHKNoneStatus;
				if (value == nil) {
					theStatus = AHKNoneStatus;
				} else if ([value isEqualToString:@"CANCELLED"] ) {
					theStatus = AHKCancelledStatus;
				} else if ([value isEqualToString:@"CONFIRMED"]) {
					theStatus = AHKConfirmedStatus;
				} else if ([value isEqualToString:@"TENTATIVE"]) {
					theStatus = AHKTentativeStatus;
				}
				[self setStatus:theStatus];
				
			// -- set location (LOCATION) --
			} else if ([name isEqualToString:@"LOCATION"]) {
				[self setLocation:value];
			}
		}
	}	
	return self;
	*/
}

- (void) dealloc {
	[self setStartDate:nil];
	//[self setEndDate:nil];
	[self setRecurrence:nil];
	[self setExcludedDates:nil];
	[self setLocation:nil];
	
	[super dealloc];
}

#pragma mark implementation method for NSCopying
- (id)copyWithZone:(NSZone *)zone
{
    id newObject = [super copyWithZone:zone];
	[newObject setStartDate:[self startDate]];
	[newObject setDuration:[self duration]];
	//[newObject setEndDate:[self endDate]];
	[newObject setAllDayEvent:[self isAllDayEvent]];
	[newObject setRecurrence:[self recurrence]];
	[newObject setExcludedDates:[self excludedDates]];
	[newObject setStatus:[self status]];
	[newObject setLocation:[self location]];
	
	return newObject;
}

#pragma mark accessor methods

- (NSCalendarDate *)startDate
{
	return startDate;
}

- (void)setStartDate:(NSCalendarDate *)value
{
	if (startDate != value) {
		[startDate release];
		startDate = [value copy];
	}
}

- (double)duration
{
	return duration;
}
- (void)setDuration:(double)value
{
	duration = value;
}

- (NSCalendarDate *)endDate
{
	//return endDate;
	return [[self startDate] dateByAddingYears:0
										months:0
										  days:0
										 hours:0
									   minutes:0
									   seconds:duration];
}
/*
- (void)setEndDate:(NSCalendarDate *)value
{
	//if (endDate != value) {
	//	[endDate release];
	//	endDate = [value copy];
	//}
	if (value != nil && [self startDate] != nil) {
		duration = [value timeIntervalSinceDate:[self startDate]];
	}
}
*/

- (BOOL)isAllDayEvent
{
	return isAllDayEvent;
}
- (void)setAllDayEvent:(BOOL)value
{
	isAllDayEvent = value;
}

- (NSString *)recurrence
{
	return recurrence;
}

- (void)setRecurrence:(NSString *)value
{
	if (recurrence != value) {
		[recurrence release];
		recurrence = [value copy];
	}
}

- (NSArray *)excludedDates
{
	return excludedDates;
}
- (void)setExcludedDates:(NSArray *)value
{
	if (excludedDates != value) {
		[excludedDates release];
		excludedDates = [value copy];
	}
}

- (int)status
{
	return status;
}
- (void)setStatus:(int)value
{
	status = value;
}

- (NSString *)location
{
	return location;
}

- (void)setLocation:(NSString *)value
{
	if (location != value) {
		[location release];
		location = [value copy];
	}
}

#pragma mark other methods

- (NSArray *)eventsByExpandingRecurrenceUntilDate:(NSDate *)untilDate
{
	NSEnumerator	*enumerator;
	id				object;
	// -- if the event is not recursive, return an array witch contains only the event itself. --
	if ([self recurrence] == nil || [[self recurrence] isEqualToString:@""]) {
		return [NSArray arrayWithObject:self];
	}
	
	//  -- make a NSDictionary object from the 'recurrence' string --
	NSDictionary *recurrenceDictionary = [self dictionaryOfRecurrence];	
	
	// -- obtain paramaters from the dictionary --
	NSString	*tempString;
	
	NSString	*freq		= [recurrenceDictionary objectForKey:@"FREQ"];
	int			interval	= [[recurrenceDictionary objectForKey:@"INTERVAL"] intValue];
	if (interval == 0) {
		interval = 1;
	}
	
	int			count		= 0x0fffff;
	if (tempString = [recurrenceDictionary objectForKey:@"COUNT"]) {
		count = [tempString intValue];
	}
	
	// set the limit for expanding dates.
	NSDate		*until		= nil;
	if (tempString = [recurrenceDictionary objectForKey:@"UNTIL"]) {
		until = [NSCalendarDate dateWithICalendarDateTimeString:tempString timeZoneString:nil];
		if (until == nil) {
			until = [NSCalendarDate dateWithICalendarDateString:tempString];
		}
	}
	if (until == nil) {
		until = untilDate;
	} else {
		until = [until earlierDate:untilDate];
	}
	
	NSString	*byDay		= [recurrenceDictionary objectForKey:@"BYDAY"];
	NSString	*byMonthDay	= [recurrenceDictionary objectForKey:@"BYMONTHDAY"];
	NSString	*byMonth	= [recurrenceDictionary objectForKey:@"BYMONTH"];
	
	int			wkst		= 0;
	if (tempString = [recurrenceDictionary objectForKey:@"WKST"]) {
		wkst = [ICSEvent dayOfWeekFromICSWeekdayString:tempString];
	}
	
	// -- initialize local variables --
	if ([[self startDate] compare:until] == NSOrderedDescending) { // object > until
		return [NSArray array];
	}
	NSMutableSet *startDateSet = [NSMutableSet setWithCapacity:16];
	[startDateSet addObject:[self startDate]];
	
	// -- expand recurrent date --
	if ([freq isEqualToString:@"DAILY"]) {
		NSArray *recurredDates = [ICSEvent datesByExpandingDailyEventWithOriginalDate:[self startDate]
																			 interval:interval
																				count:count
																				until:until];
		[startDateSet addObjectsFromArray:recurredDates];
	} else if ([freq isEqualToString:@"WEEKLY"]) {
		NSArray *recurredDates = [ICSEvent datesByExpandingWeeklyEventWithOriginalDate:[self startDate]
																				 byDay:byDay
																				  wkst:wkst
																			  interval:interval
																				 count:count
																				 until:until];
		[startDateSet addObjectsFromArray:recurredDates];
	} else if ([freq isEqualToString:@"MONTHLY"]) {
		NSArray *recurredDates = [ICSEvent datesByExpandingMonthlyEventWithOriginalDate:[self startDate]
																				  byDay:byDay
																			 byMonthDay:byMonthDay
																			   interval:interval
																				  count:count
																				  until:until];
		[startDateSet addObjectsFromArray:recurredDates];
	} else if ([freq isEqualToString:@"YEARLY"]) {
		NSArray *recurredDates = [ICSEvent datesByExpandingYearlyEventWithOriginalDate:[self startDate]
																				  byDay:byDay
																			 byMonthDay:byMonthDay
																				byMonth:byMonth
																			   interval:interval
																				  count:count
																				  until:until];
		[startDateSet addObjectsFromArray:recurredDates];
	}
	
	NSArray *startDates = [[startDateSet allObjects] sortedArrayUsingSelector:@selector(compare:)];
	
	// -- match sum of expanded events to COUNT, if sum is greater than COUNT. -- 
	if ([startDates count] > count) {
		startDates = [startDates subarrayWithRange:NSMakeRange(0, count)];
	}
	
	// -- remove EXDATE (excluded dates). -- 
	if ([self excludedDates] != nil && [[self excludedDates] count] > 0) {
		NSMutableArray	*tempArray = [startDates mutableCopy];
		[tempArray removeObjectsInArray:[self excludedDates]];
		startDates = [NSArray arrayWithArray:tempArray];
		[tempArray release];
	}
	
	enumerator = [startDates objectEnumerator];
	ICSEvent		*event;
	NSMutableArray	*resultArray = [NSMutableArray arrayWithCapacity:[startDateSet count]];
	
	while (object = [enumerator nextObject]) {
		event = [self copy];
		[event setStartDate:object];
		[resultArray addObject:event];
		[event release];
	}
	
	return resultArray;
}

#pragma mark private methods

// return 0--6 or NSNotFound
+ (int)dayOfWeekFromICSWeekdayString:(NSString *)value
{
	NSArray *array = [NSArray arrayWithObjects:@"SU", @"MO", @"TU", @"WE", @"TH", @"FR", @"SA", nil];
	return [array indexOfObject:value];
}

+ (BOOL)weekOfMonth:(int *)ordwk dayOfWeek:(int *)weekday fromICSByDayString:(NSString *)string
{
	NSScanner *scanner = [NSScanner scannerWithString:string];
	
	*ordwk = 0;
	[scanner scanInt:ordwk];
	
	if ([scanner scanString:@"SU" intoString:nil]) {
		*weekday = 0;
	} else if ([scanner scanString:@"MO" intoString:nil]) {
		*weekday = 1;
	} else if ([scanner scanString:@"TU" intoString:nil]) {
		*weekday = 2;
	} else if ([scanner scanString:@"WE" intoString:nil]) {
		*weekday = 3;
	} else if ([scanner scanString:@"TH" intoString:nil]) {
		*weekday = 4;
	} else if ([scanner scanString:@"FR" intoString:nil]) {
		*weekday = 5;
	} else if ([scanner scanString:@"SA" intoString:nil]) {
		*weekday = 6;
	} else {
		*weekday = NSNotFound;
		return NO;
	}
	return YES;
}


+ (void)days:(int *)daysPointer hours:(int *)hoursPointer minites:(int *)minutesPointer seconds:(int *)secondsPointer fromDuration:(NSString *)duration
{
	*daysPointer = 0;
	*hoursPointer = 0;
	*minutesPointer = 0;
	*secondsPointer = 0;
	
	// see RFC2445 for the detailed format of DURATION.
	NSScanner		*scanner			= [NSScanner scannerWithString:duration];
	NSCharacterSet	*daysCharacterSet	= [NSCharacterSet characterSetWithCharactersInString:@"WD"];
	NSCharacterSet	*timeCharacterSet	= [NSCharacterSet characterSetWithCharactersInString:@"HMS"];
	int				scanLocation;
	NSString		*numberString, *delimiterString;
	
	if ([scanner scanString:@"P" intoString:nil]) {
		scanLocation = [scanner scanLocation];
		
		if ([scanner isAtEnd] == NO && 
			[scanner scanUpToCharactersFromSet:daysCharacterSet intoString:&numberString] &&
			[scanner scanCharactersFromSet:daysCharacterSet intoString:&delimiterString]) {
			
			if ([delimiterString isEqualToString:@"W"]) {
				// dur-week, i.e. P3W
				*daysPointer = [numberString intValue] * 7;
				return;
			} else if ([delimiterString isEqualToString:@"D"]) {
				// dur-date, i.e. P12D, P2DT3H0M20S
				*daysPointer = [numberString intValue];
			}
		} else {
			// dur-time, i.e. PT3M, PT3H00M20S 
			[scanner setScanLocation:scanLocation];
		}
		
		if ([scanner scanString:@"T" intoString:nil]) {
			while ([scanner isAtEnd] == NO &&
				   [scanner scanUpToCharactersFromSet:timeCharacterSet intoString:&numberString] &&
				   [scanner scanCharactersFromSet:timeCharacterSet intoString:&delimiterString]) {
				
				if ([delimiterString isEqualToString:@"H"]) {
					// dur-hour
					*hoursPointer = [numberString intValue];
				} else if ([delimiterString isEqualToString:@"M"]) {
					// dur-minute
					*minutesPointer = [numberString intValue];
				} else if ([delimiterString isEqualToString:@"S"]) {
					// dur-second
					*secondsPointer = [numberString intValue];
				}
			}
		}
	}
}

- (NSDictionary *)dictionaryOfRecurrence
{
	NSArray				*paramArray			= [[self recurrence] componentsSeparatedByString:@";"];
	NSMutableDictionary	*paramDictionary	= [NSMutableDictionary dictionaryWithCapacity:[paramArray count]];
	
	NSEnumerator		*enumerator			= [paramArray objectEnumerator];
	id					object;
	
	while (object = [enumerator nextObject]) {
		NSArray *paramPairArray = [object componentsSeparatedByString:@"="];
		if ([paramPairArray count] == 2) {
			[paramDictionary setObject:[paramPairArray objectAtIndex:1]
								forKey:[paramPairArray objectAtIndex:0]];
		}
	}
	return [NSDictionary dictionaryWithDictionary:paramDictionary];
}

+ (NSArray *)datesByExpandingDailyEventWithOriginalDate:(NSCalendarDate *)originalDate interval:(int)interval count:(int)count until:(NSDate *)until
{
	NSCalendarDate	*currentDate	= originalDate;
	NSMutableArray	*recurredDates	= [NSMutableArray arrayWithCapacity:16];
	
	int i;
	for (i = 0; i < count; i++) {
		if ([currentDate compare:until] == NSOrderedDescending) { // object > until
			break;
		}
		[recurredDates addObject:currentDate];
		currentDate = [currentDate dateByAddingDays:interval];
	}
	return recurredDates;
}

+ (NSArray *)datesByExpandingWeeklyEventWithOriginalDate:(NSCalendarDate *)originalDate byDay:(NSString *)byDay wkst:(int)wkst interval:(int)interval count:(int)count until:(NSDate *)until
{
	if (byDay == nil) {
		return [ICSEvent datesByExpandingDailyEventWithOriginalDate:originalDate
														   interval:(interval * 7)
															  count:count
															  until:until];
	} else {
		NSMutableArray	*recurredDates	= [NSMutableArray arrayWithCapacity:16];
		
		int				startDayOfWeek		= [originalDate dayOfWeek];
		NSArray			*byDayArray			= [byDay componentsSeparatedByString:@","];
		NSEnumerator	*enumerator			= [byDayArray objectEnumerator];
		NSString		*string;
		
		while (string = [enumerator nextObject]) {
			int recurDayOfWeek = [ICSEvent dayOfWeekFromICSWeekdayString:string];
			if (recurDayOfWeek == NSNotFound) {
				continue;
			}
			int	daysBetweenStartDateAndRecurDate		= (recurDayOfWeek - startDayOfWeek + 7) % 7;
			int	daysBetweenStartDateAndWeekStartDate	= (wkst - startDayOfWeek + 7) % 7;
			
			if (daysBetweenStartDateAndRecurDate >=  daysBetweenStartDateAndWeekStartDate) {
				daysBetweenStartDateAndRecurDate += 7 * (interval - 1);
			}
			NSCalendarDate *currentOriginalDate = [originalDate dateByAddingDays:daysBetweenStartDateAndRecurDate];
			NSArray *array = [ICSEvent datesByExpandingDailyEventWithOriginalDate:currentOriginalDate
																		 interval:(interval * 7)
																			count:count
																			until:until];
			[recurredDates addObjectsFromArray:array];
		}
		return recurredDates;
	}
}

+ (NSArray *)datesByExpandingMonthlyEventWithOriginalDate:(NSCalendarDate *)originalDate byDay:(NSString *)byDay byMonthDay:(NSString *)byMonthDay interval:(int)interval count:(int)count until:(NSDate *)until
{
	if (byDay) {
		NSArray			*byDayArray		= [byDay componentsSeparatedByString:@","];
		NSEnumerator	*enumerator		= [byDayArray objectEnumerator];
		NSString		*byDayString;
		
		NSMutableArray	*recurredDates	= [NSMutableArray arrayWithCapacity:([byDayArray count] * 4)];
		
		while (byDayString = [enumerator nextObject]) {
			int weekOfMonth, dayOfWeek;
			[ICSEvent weekOfMonth:&weekOfMonth 
						dayOfWeek:&dayOfWeek
			   fromICSByDayString:byDayString];
			
			if (weekOfMonth == 0 || dayOfWeek == NSNotFound) {
				continue;
			}
			
			NSCalendarDate *currentDate;
			if (weekOfMonth > 0) {
				currentDate	= [originalDate dateAtBeginningOfMonth];
				currentDate	= [currentDate dateByAddingDays:((dayOfWeek - [currentDate dayOfWeek] + 7) % 7)];
				currentDate	= [currentDate dateByAddingDays:((weekOfMonth - 1) * 7)];
				
				int i, j= 0;
				for (i = 0; YES; i++) {
					if ([currentDate compare:until] == NSOrderedDescending || j >= count) {
						break;
					}
					if (i % interval == 0 && 
						[currentDate compare:originalDate] != NSOrderedAscending) { // tempDate is not older than originalDate
						[recurredDates addObject:currentDate];
						j++;
					}
					NSCalendarDate *newCurrentDate = [currentDate dateByAddingDays:7 * 4];
					if ([newCurrentDate weekOfMonth] == weekOfMonth) {
						currentDate = newCurrentDate;
					} else {
						currentDate = [currentDate dateByAddingDays:7 * 5];
					}
				}
			} else if (weekOfMonth < 0) {
				currentDate = [originalDate dateAtEndOfMonth];
				currentDate = [currentDate dateByAddingDays:-(([currentDate dayOfWeek] - dayOfWeek + 7) % 7)];
				currentDate = [currentDate dateByAddingDays:((weekOfMonth + 1) * 7)];
				
				int i, j = 0;
				for (i = 0; YES; i++) {
					if ([currentDate compare:until] == NSOrderedDescending || j >= count) {
						break;
					}
					if (i % interval == 0 && 
						[currentDate compare:originalDate] != NSOrderedAscending) { // tempDate is not older than originalDate
						[recurredDates addObject:currentDate];
						j++;
					}
					NSCalendarDate *newCurrentDate = [currentDate dateByAddingDays:7 * 5];
					if ([newCurrentDate reverseWeekOfMonth] == weekOfMonth) {
						currentDate = newCurrentDate;
					} else {
						currentDate = [currentDate dateByAddingDays:7 * 4];
					}
				}
			}
		}
		return recurredDates;
	} else {
		// if byDay is nil...
		NSArray			*byMonthDayArray;
		if (byMonthDay) {
			byMonthDayArray = [byMonthDay componentsSeparatedByString:@","];
		} else {
			byMonthDayArray = [NSArray arrayWithObject:[NSString stringWithFormat:@"%d", [originalDate dayOfMonth]]];
		}
		
		NSEnumerator	*byMonthDayEnumerator	= [byMonthDayArray objectEnumerator];
		NSString		*byMonthDayString;
		
		NSMutableArray	*originalDates			= [NSMutableArray arrayWithCapacity:[byMonthDayArray count]];
		
		while (byMonthDayString = [byMonthDayEnumerator nextObject]) {
			int currentDayOfMonth = [byMonthDayString intValue];
			
			int i = 0;
			if (currentDayOfMonth < [originalDate dayOfMonth]) {
				i = 1;
			}
			NSCalendarDate *date;
			while (i < kNumberOfSearchingNextMonth) {
				date = [originalDate dateByAddingDays:(1 - [originalDate dayOfMonth])];
				date = [date dateByAddingYears:0
										months:(interval * i)
										  days:(currentDayOfMonth - 1)
										 hours:0
									   minutes:0
									   seconds:0];
				if ([date dayOfMonth] == currentDayOfMonth) {
					[originalDates addObject:date];
					break;
				}
				i++;
			}
		}
		
		NSEnumerator	*originalDateEnumerator	= [originalDates objectEnumerator];
		NSCalendarDate	*currentOriginalDate;
		NSMutableArray	*recurredDates			= [NSMutableArray arrayWithCapacity:16];
		while (currentOriginalDate = [originalDateEnumerator nextObject]) {
			NSCalendarDate *currentDate = currentOriginalDate;
			int i;
			for (i = 0; i < count; i++) {
				if ([currentDate compare:until] == NSOrderedDescending) { // object > until
					break;
				}
				int j;
				for (j = 1; j < kNumberOfSearchingNextMonth; j++) {
					currentDate = [currentOriginalDate dateByAddingMonths:(interval * i)];
					if (currentDate != nil) {
						if ([currentDate compare:until] != NSOrderedDescending) { // object <= until
							[recurredDates addObject:currentDate];
						}
						break;
					}
				}
			}
		}
		return recurredDates;
	}
	return nil;
}

+ (NSArray *)datesByExpandingYearlyEventWithOriginalDate:(NSCalendarDate *)originalDate byDay:(NSString *)byDay byMonthDay:(NSString *)byMonthDay byMonth:(NSString *)byMonth interval:(int)interval count:(int)count until:(NSDate *)until
{
	if (byDay == nil && byMonthDay == nil) {
		byMonthDay = [NSString stringWithFormat:@"%d", [originalDate dayOfMonth]];
	}
	
	NSArray			*byMonthArray;
	if (byMonth) {
		byMonthArray = [byMonth componentsSeparatedByString:@","];
	} else {
		byMonthArray = [NSArray arrayWithObject:[NSString stringWithFormat:@"%d", [originalDate monthOfYear]]];
	}
	
	NSMutableArray	*recurredDates		= [NSMutableArray arrayWithCapacity:[byMonthArray count]];
	NSEnumerator	*byMonthEnumerator	= [byMonthArray objectEnumerator];
	NSString		*currentByMonthString;
	
	while (currentByMonthString = [byMonthEnumerator nextObject]) {
		int currentByMonth = [currentByMonthString intValue];
		
		NSCalendarDate *currentOriginalDate;
		
		int months = currentByMonth - [originalDate monthOfYear];
		
		if (months == 0) {
			currentOriginalDate = originalDate;
		} else if (months > 0) {
			currentOriginalDate = [[originalDate dateAtBeginningOfMonth] dateByAddingMonths:months];
		} else {
			currentOriginalDate = [[originalDate dateAtBeginningOfMonth] dateByAddingMonths:(months + 12 * interval)];
		}
		
		NSArray *array = [self datesByExpandingMonthlyEventWithOriginalDate:currentOriginalDate
																	  byDay:byDay
																 byMonthDay:byMonthDay
																   interval:(interval * 12)
																	  count:count
																	  until:until];
		if (array) [recurredDates addObjectsFromArray:array];
	}
	return recurredDates;
}

@end
