//
//  AHKCALMainController.m
//  iCal to KS2
//
//  Created by FUJIDANA on 05/05/21.
//  Copyright 2005 FUJIDANA. All rights reserved.
//
//
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#import "AHKCALMainController.h"
#import <AddressBook/AddressBook.h>
#import "ICSCalendar.h"
#import "AHKCALCsvExporter.h"

#import "NSAppleScript+HandlerCalls.h"

static NSString*	AHKCALCalendarsToBeExported				= @"AHKCALCalendarUIDsToBeExported";
static NSString*	AHKCALFilenameToBeExported				= @"AHKCALFilenameToBeExported";
static NSString*	AHKCALDaysBeforeToday					= @"AHKCALDaysBeforeToday";
static NSString*	AHKCALDaysAfterToday					= @"AHKCALDaysAfterToday";
static NSString*	AHKCALExportsColor						= @"AHKCALExportsColor";

static NSString*	AHKCALOmitsTimeOfAllDayEvents			= @"AHKCALOmitsTimeOfAllDayEvents";
static NSString*	AHKCALDividesCrossDayEvents				= @"AHKCALDividesCrossDayEvents";
static NSString*	AHKCALOmitsInterludeOfLongEvents		= @"AHKCALOmitsInterludeOfLongEvents";
static NSString*	AHKCALMaxDaysForNotOmittingInterlude	= @"AHKCALMaxDaysForNotOmittingInterlude";
static NSString*	AHKCALExpandsRecurrentEvents			= @"AHKCALExpandsRecurrentEvents";

static NSString*	AHKCALExportsTodos						= @"AHKCALExportsTodos";
static NSString*	AHKCALExportsCompletedTodos				= @"AHKCALExportsCompletedTodos";
static NSString*	AHKCALSetsDueDatesOfTermlessTodos		= @"AHKCALSetsDueDatesOfTermlessTodos";
static NSString*	AHKCALDueDateOfTermlessTodos			= @"AHKCALDueDateOfTermlessTodos";

static NSString*	AHKCALLaunchesMailApplication			= @"AHKCALLaunchesMailApplication";
static NSString*	AHKCALUsesAppleMail						= @"AHKCALUsesAppleMail";
static NSString*	AHKCALSendsAutomatically				= @"AHKCALSendsAutomatically";
static NSString*	AHKCALEmailAddress						= @"AHKCALEmailAddress";

static NSString*	AHKCALCalendarTableColumnIdentifier		= @"calendar";
static NSString*	AHKCALStateTableColumnIdentifier		= @"state";
static NSString*	AHKCALTitleTableColumnIdentifier		= @"title";
static NSString*	AHKCALIdentifierTableColumnIdentifier	= @"identifier";

#define noScriptErr 0

@interface AHKCALMainController (Private)

- (void)exportAlertDidEnd:(NSAlert *)alert returnCode:(int)returnCode contextInfo:(void *)contextInfo;
- (void)storeDefaults;
- (void)sendEmailWithAttachmentFile:(NSString *)path;
- (void)handleScriptError:(NSDictionary *)errorInfo;

@end

#pragma mark -

@implementation AHKCALMainController

#pragma mark class methods
+ (void)initialize {
	NSString *path = [[NSHomeDirectory() stringByAppendingPathComponent:@"Desktop"]
		stringByAppendingPathComponent:@"SCHEDULE.TXT"];
	
	// -- register defaults --
	NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
	NSDictionary *defaultsDictionary = [NSDictionary dictionaryWithObjectsAndKeys:[NSArray array], AHKCALCalendarsToBeExported,
		path, AHKCALFilenameToBeExported,
		[NSNumber numberWithInt:10], AHKCALDaysBeforeToday, 
		[NSNumber numberWithInt:30], AHKCALDaysAfterToday, 
		[NSNumber numberWithInt:NSOnState], AHKCALExportsColor,
		[NSNumber numberWithInt:NSOnState], AHKCALOmitsTimeOfAllDayEvents,
		[NSNumber numberWithInt:NSOnState], AHKCALDividesCrossDayEvents,
		[NSNumber numberWithInt:NSOnState], AHKCALOmitsInterludeOfLongEvents,
		[NSNumber numberWithInt:3], AHKCALMaxDaysForNotOmittingInterlude,
		[NSNumber numberWithInt:NSOffState], AHKCALExpandsRecurrentEvents,
		[NSNumber numberWithInt:NSOnState], AHKCALExportsTodos,
		[NSNumber numberWithInt:NSOnState], AHKCALExportsCompletedTodos, 
		[NSNumber numberWithInt:NSOnState], AHKCALSetsDueDatesOfTermlessTodos,
		[NSCalendarDate dateWithYear:2005
							   month:12
								 day:31
								hour:0 minute:0 second:0
							timeZone:nil], AHKCALDueDateOfTermlessTodos,
		[NSNumber numberWithInt:NSOffState], AHKCALLaunchesMailApplication,
		[NSNumber numberWithInt:NSOffState], AHKCALUsesAppleMail,
		[NSNumber numberWithInt:NSOffState], AHKCALSendsAutomatically,
		[NSString string], AHKCALEmailAddress,
		nil];
	[defaults registerDefaults:defaultsDictionary];
}

#pragma mark -
#pragma mark initialization/deallocation methods
- (id)init
{
	self = [self initWithWindowNibName:@"MainWindow"];
	if (self) {
		[self setWindowFrameAutosaveName:@"AHKCALMainWindow"];
		// custom initialization will be written here...
		[[self window] setExcludedFromWindowsMenu:YES];
		[self showWindow:self];
	}
	return self;
}

- (void)windowDidLoad
{
	[super windowDidLoad];
	// 
	[preferencesWindow center];
	
	// -- load user defaults --
    NSUserDefaults		*defaults					= [NSUserDefaults standardUserDefaults];
	NSArray				*calendarUIDsToBeExported	= [defaults stringArrayForKey:AHKCALCalendarsToBeExported];
	[daysBeforeTodayTextField setIntValue:[defaults integerForKey:AHKCALDaysBeforeToday]];
	[daysAfterTodayTextField setIntValue:[defaults integerForKey:AHKCALDaysAfterToday]];
	
	// -- load calendars --
	NSArray				*allCalendars				= [ICSCalendar calendars];
	NSEnumerator		*calendarEnumearator		= [allCalendars objectEnumerator];
	
	ICSCalendar			*calendar;
	NSMutableDictionary	*calendarDict;
	
	dataSource		= [[NSMutableArray alloc] initWithCapacity:[allCalendars count]];
	
	while (calendar = [calendarEnumearator nextObject]) {
		calendarDict = [NSMutableDictionary dictionaryWithCapacity:4];
		[calendarDict setObject:calendar forKey:AHKCALCalendarTableColumnIdentifier];
		[calendarDict setObject:[calendar identifier] forKey:AHKCALIdentifierTableColumnIdentifier];
		[calendarDict setObject:[calendar title] forKey:AHKCALTitleTableColumnIdentifier];
		
		if ([calendarUIDsToBeExported containsObject:[calendar identifier]]) {
			[calendarDict setObject:[NSNumber numberWithInt:NSOnState] forKey:AHKCALStateTableColumnIdentifier];
		} else {
			[calendarDict setObject:[NSNumber numberWithInt:NSOffState] forKey:AHKCALStateTableColumnIdentifier];
		}
		
		[dataSource addObject:calendarDict];
	}
	[calendarTableView reloadData];
}

- (void) dealloc {
	[dataSource release];
	
	[super dealloc];
}

#pragma mark delegate and implementation of protocol
#pragma mark delegated method by NSApplication
- (void)applicationWillTerminate:(NSNotification *)aNotification
{
	// -- store parameters in user defaluts --
	[self storeDefaults];
}

- (BOOL)applicationShouldTerminateAfterLastWindowClosed:(NSApplication *)theApplication
{
	return NO;//return YES;
}

#pragma mark implementations methods for NSMenuValidation (informal protocol)
- (BOOL)validateMenuItem:(id <NSMenuItem>)menuItem
{
	if ([menuItem action] == @selector(showWindow:)) {
		[menuItem setState:([[self window] isVisible]? NSOnState: NSOffState)];
	}
	return YES;
}

#pragma mark implementations methods for NSTableViewDataSource (informal protocol)
- (int)numberOfRowsInTableView:(NSTableView *)aTableView
{
	return [dataSource count];
}

- (id)tableView:(NSTableView *)aTableView objectValueForTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
	id theRecord, theValue;
	
	NSParameterAssert(rowIndex >= 0 && rowIndex < [dataSource count]);
	theRecord = [dataSource objectAtIndex:rowIndex];
	theValue = [theRecord objectForKey:[aTableColumn identifier]];
	return theValue;
}


- (void)tableView:(NSTableView *)aTableView setObjectValue:anObject forTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
	id theRecord;
	
	NSParameterAssert(rowIndex >= 0 && rowIndex < [dataSource count]);
	theRecord = [dataSource objectAtIndex:rowIndex];
	[theRecord setObject:anObject forKey:[aTableColumn identifier]];
}

#pragma mark IBActions and their delegation methods
- (IBAction)exportAction:(id)sender
{
	// -- check whether iCal is launched or not--
	NSWorkspace		*workspace						= [NSWorkspace sharedWorkspace];
	NSEnumerator	*launchedApplicationEnumerator	= [[workspace launchedApplications] objectEnumerator];
	id object;
	while (object = [launchedApplicationEnumerator nextObject]) {
		if ([[object objectForKey:@"NSApplicationBundleIdentifier"] isEqualToString:@"com.apple.iCal"]) {
			// -- if iCal is launched, show an alert.
			NSAlert *alert = [[[NSAlert alloc] init] autorelease];
			[alert setMessageText:NSLocalizedString(@"Calendar files this application is going to read in might not have refreshed with latest changes made on iCal.", @"mainController.iCalIsLaunched.message")];
			[alert setInformativeText:NSLocalizedString(@"iCal does not seem to update calendar files immediately you modify calendar data. If you want to ensure a file to be exported mirrors the latest changes, quit iCal before exporting a file.", @"mainController.iCalIsLaunched.informative")];
			[alert addButtonWithTitle:NSLocalizedString(@"Export", @"mainController.iCalIsLaunched.firstButton")];
			[alert addButtonWithTitle:NSLocalizedString(@"Cancel", @"mainController.iCalIsLaunched.secondButton")];
			[alert addButtonWithTitle:NSLocalizedString(@"Quit iCal and Export", @"mainController.iCalIsLaunched.thirdButton")];
			
			[alert beginSheetModalForWindow:[self window]
							  modalDelegate:self
							 didEndSelector:@selector(exportAlertDidEnd:returnCode:contextInfo:) 
								contextInfo:nil];
			return;
		}
	}
	// -- if iCal is not launched, execute the exporting process.
	[self exportAlertDidEnd:nil returnCode:NSAlertFirstButtonReturn contextInfo:nil];
}

- (void)exportAlertDidEnd:(NSAlert *)alert returnCode:(int)returnCode contextInfo:(void *)contextInfo
{
	[[alert window] orderOut:nil];
	
	if (returnCode == NSAlertSecondButtonReturn) {
		return;
	} else if (returnCode == NSAlertThirdButtonReturn) {
		NSAppleScript *appleScript = [[[NSAppleScript alloc] initWithSource:@"tell application \"iCal\" to quit"] autorelease];
		[appleScript executeAndReturnError:nil];
		[NSThread sleepUntilDate:[NSDate dateWithTimeIntervalSinceNow:1]];
	}
	
	[progressIndicator startAnimation:self];
	NSModalSession session = [[NSApplication sharedApplication] beginModalSessionForWindow:[self window]];
	[[NSApplication sharedApplication] runModalSession:session];
	
	// -- store parameters in user defaluts --
	[self storeDefaults];
	
	// -- get the span in which calendar will be exported
	NSCalendarDate	*currentDate			= [NSCalendarDate calendarDate];
	
	int				hours					= [currentDate hourOfDay];
	int				minutes					= [currentDate minuteOfHour];
	int				seconds					= [currentDate secondOfMinute];
	
	NSCalendarDate	*startingDate			= [currentDate dateByAddingYears:0
																	  months:0
																		days:(- [daysBeforeTodayTextField intValue])
																	   hours:(- hours)
																	 minutes:(- minutes)
																	 seconds:(- seconds)];
	
	NSCalendarDate	*endingDate				= [currentDate dateByAddingYears:0
																	  months:0
																		days:[daysAfterTodayTextField intValue]
																	   hours:(23 - hours)
																	 minutes:(59 - minutes)
																	 seconds:(59 - seconds)];
	
	NSEnumerator	*calendarEnumerator		= [dataSource objectEnumerator];
	NSDictionary	*calendarDict;
	NSMutableArray	*calendarsToBeExported	= [NSMutableArray arrayWithCapacity:[dataSource count]];
	
	while (calendarDict = [calendarEnumerator nextObject]) {
		NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
		if ([[calendarDict objectForKey:AHKCALStateTableColumnIdentifier] isEqualToNumber:[NSNumber numberWithInt:NSOnState]]) {
			ICSCalendar *calendar = [calendarDict objectForKey:AHKCALCalendarTableColumnIdentifier];
			if (calendar) {
				[calendar reload];
				[calendarsToBeExported addObject:calendar];
			}
		}
		[pool release];
	}
	
	// -- export --
	AHKCALCsvExporter	*csvExporter	= [[[AHKCALCsvExporter alloc] initWithCalendars:calendarsToBeExported] autorelease];
	NSUserDefaults		*defaults		= [NSUserDefaults standardUserDefaults];
	
	[csvExporter setStartingDateForExport:startingDate];
	[csvExporter setEndingDateForExport:endingDate];
	[csvExporter setExportsColor:[defaults boolForKey:AHKCALExportsColor]];
	[csvExporter setOmitsTimeOfAllDayEvents:[defaults boolForKey:AHKCALOmitsTimeOfAllDayEvents]];
	[csvExporter setDividesCrossDayEvents:[defaults boolForKey:AHKCALDividesCrossDayEvents]];
	[csvExporter setOmitsInterludeOfLongLastingEvents:[defaults boolForKey:AHKCALOmitsInterludeOfLongEvents]];
	[csvExporter setMaxDaysForNotOmittingInterlude:[defaults integerForKey:AHKCALMaxDaysForNotOmittingInterlude]];
	[csvExporter setExpandsRecurrentEvents:[defaults boolForKey:AHKCALExpandsRecurrentEvents]];
	[csvExporter setExportsTodos:[defaults boolForKey:AHKCALExportsTodos]];
	[csvExporter setExportsCompletedTodos:[defaults boolForKey:AHKCALExportsCompletedTodos]];
	[csvExporter setAsignsDueDatesOfTermlessTodos:[defaults boolForKey:AHKCALSetsDueDatesOfTermlessTodos]];
	NSCalendarDate *date = [NSCalendarDate dateWithTimeIntervalSinceNow:[[defaults objectForKey:AHKCALDueDateOfTermlessTodos] timeIntervalSinceNow]];
	[csvExporter setDueDateOfTermlessTodos:date];
	
	NSString	*scheduleTextString	= [csvExporter csvRepresentation];
	NSData		*scheduleTextData	= [scheduleTextString dataUsingEncoding:NSShiftJISStringEncoding];
	
	NSString	*path				= [defaults stringForKey:AHKCALFilenameToBeExported];
	if (path != nil) {
		if ([scheduleTextData writeToFile:path atomically:YES] == NO) {
				//NSLog(@"failed to write SCHEDULE.TXT");
		} else if ([defaults boolForKey:AHKCALLaunchesMailApplication]) {
			[self sendEmailWithAttachmentFile:path];
		}
	}
	
	//[NSThread sleepUntilDate:[NSDate dateWithTimeIntervalSinceNow:5]];
	[progressIndicator stopAnimation:self];
	[[NSApplication sharedApplication] requestUserAttention:NSInformationalRequest];
	[[NSApplication sharedApplication] endModalSession:session];
}

- (IBAction)showPreferencesWindow:(id)sender
{
	[preferencesWindow makeKeyAndOrderFront:sender];
}

- (IBAction)selectFilenameToBeExported:(id)sender
{
	NSSavePanel		*savePanel			= [NSSavePanel savePanel];
	NSUserDefaults	*userDefaults		= [NSUserDefaults standardUserDefaults];
	NSString		*calendarFilePath	= [userDefaults stringForKey:AHKCALFilenameToBeExported];
	
	[savePanel setAllowedFileTypes:[NSArray arrayWithObjects:@"TXT", @"TSV", nil]];
	[savePanel setCanSelectHiddenExtension:YES];
	
	int returnCode = [savePanel runModalForDirectory:[calendarFilePath stringByDeletingLastPathComponent] 
												file:[calendarFilePath lastPathComponent]];
	if (returnCode == NSFileHandlingPanelOKButton) {
		[userDefaults setObject:[savePanel filename]
						 forKey:AHKCALFilenameToBeExported];
	}
	
}

- (IBAction)guessMailAddress:(id)sender
{
	ABPerson *me = [[ABAddressBook sharedAddressBook] me];
	if (me) {
		ABMultiValue *mailAddresses = [me valueForProperty:kABEmailProperty];
		NSString *string, *substring;
		int i;

		for (i = 0; i < [mailAddresses count]; i++) {
			string = [mailAddresses valueAtIndex:i];
			if ([string length] >= 10) {
				substring = [string substringFromIndex:[string length] - 10];
				if ([substring isEqualToString:@".pdx.ne.jp"] || [substring isEqualToString:@"@pdx.ne.jp"]) {
					[mailAddressTextField setStringValue:string];
					[[NSUserDefaults standardUserDefaults] setObject:string
															  forKey:AHKCALEmailAddress];
					return;
				}
			}
		}
	}
	
	NSAlert	*alert = [[[NSAlert alloc] init] autorelease];
	[alert setMessageText:NSLocalizedString(@"Your card in Address Book does not seem to contain an email address of the Air-EDGE Phone.", @"guessMailAddress.alert.message")];
	[alert setInformativeText:NSLocalizedString(@"Confirm your card is set in Address Book, and your card contains a email address that ends with pdx.ne.jp.", @"guessMailAddress.alert.informative")];
	[alert setAlertStyle:NSInformationalAlertStyle];
	[alert runModal];
}

#pragma mark other methods
- (void)storeDefaults
{
    NSUserDefaults		*defaults	= [NSUserDefaults standardUserDefaults];
	
	NSEnumerator		*enumerator	= [dataSource objectEnumerator];
	NSMutableDictionary	*calendarDict;
	
	[defaults synchronize];
	
	// -- get names of checked calendars, and store them into user defaults -- 
	NSMutableArray		*calendarUIDsToBeExported	= [NSMutableArray arrayWithCapacity:[dataSource count]];
	
	while (calendarDict = [enumerator nextObject]) {
		if ([[calendarDict objectForKey:AHKCALStateTableColumnIdentifier] isEqualToNumber:[NSNumber numberWithInt:NSOnState]]) {
			[calendarUIDsToBeExported addObject:[calendarDict objectForKey:AHKCALIdentifierTableColumnIdentifier]];
		}
	}
	[defaults setObject:calendarUIDsToBeExported forKey:AHKCALCalendarsToBeExported];
	
	[defaults setInteger:[daysBeforeTodayTextField intValue] forKey:AHKCALDaysBeforeToday];
	[defaults setInteger:[daysAfterTodayTextField intValue] forKey:AHKCALDaysAfterToday];
}

- (void)sendEmailWithAttachmentFile:(NSString *)path
{
	NSUserDefaults	*defaults		= [NSUserDefaults standardUserDefaults];
	
	NSString		*emailAddress	= [defaults stringForKey:AHKCALEmailAddress];
	NSString		*subject		= NSLocalizedString(@"iCal Schedule data", @"mainController.export.mailSubject");
	NSString		*body			= [NSString stringWithFormat:NSLocalizedString(@"Exported Date: %@", @"mainController.export.mailBody"), [NSDate date]];
	
	if ([defaults boolForKey:AHKCALUsesAppleMail]) {
		/* Locate the script within the bundle */
		NSString *scriptPath = [[NSBundle mainBundle] pathForResource:@"LaunchMail" 
															   ofType:@"scpt"
														  inDirectory:@"Scripts"];
		NSURL *scriptURL = [NSURL fileURLWithPath:scriptPath];
		
		NSDictionary *errorInfo = nil;
		
		/* Here I am using "initWithContentsOfURL:" to load a pre-compiled script, rather than using "initWithSource:" to load a text file with AppleScript source.  The main reason for this is that the latter technique seems to give rise to inexplicable -1708 (errAEEventNotHandled) errors on Jaguar. */
		NSAppleScript *script = [[NSAppleScript alloc] initWithContentsOfURL: scriptURL error: &errorInfo];
		
		/* See if there were any errors loading the script */
		if (!script || errorInfo) {
			[self handleScriptError: errorInfo];
		}
		
		/* We have to construct an AppleEvent descriptor to contain the arguments for our handler call.  Remember that this list is 1, rather than 0, based. */
		NSAppleEventDescriptor *arguments = [[NSAppleEventDescriptor alloc] initListDescriptor];
		[arguments insertDescriptor: [NSAppleEventDescriptor descriptorWithString: emailAddress] atIndex: 1];
		[arguments insertDescriptor:[NSAppleEventDescriptor descriptorWithString:subject] atIndex:2];
		[arguments insertDescriptor:[NSAppleEventDescriptor descriptorWithString:body] atIndex:3];
		[arguments insertDescriptor:[NSAppleEventDescriptor descriptorWithString:path] atIndex:4];
		Boolean boolean = [defaults boolForKey:AHKCALSendsAutomatically] == YES? true: false;
		[arguments insertDescriptor:[NSAppleEventDescriptor descriptorWithBoolean:boolean] atIndex:5];
		
		errorInfo = nil;
		
		/* Call the handler using the method in our special category */
		NSAppleEventDescriptor *result = [script callHandler:@"show_message"
											   withArguments:arguments
												   errorInfo:&errorInfo];
		
		int scriptResult = [result int32Value];
		
		/* Check for errors in running the handler */
		if (errorInfo) {
			[self handleScriptError: errorInfo];
		}
		/* Check the handler's return value */
		else if (scriptResult != noScriptErr) {
			NSRunAlertPanel(NSLocalizedString(@"Script Failure", @"Title on script failure window."), [NSString stringWithFormat: @"%@ %d", NSLocalizedString(@"The script failed:", @"Message on script failure window."), scriptResult], NSLocalizedString(@"OK", @""), nil, nil);
		}
		
		[script release];
		[arguments release];
		
	} else {
		NSString	*urlString	= [NSString stringWithFormat:@"mailto:%@?subject=%@&body=%@", emailAddress, subject, body];
		
		urlString = [urlString stringByAddingPercentEscapesUsingEncoding:NSISO2022JPStringEncoding];
		NSURL		*url		= [NSURL URLWithString:urlString];
		/*
		 NSURL	*url;
		 url = [NSURL URLWithString:@"mailto:dts@apple.com"
			 "?subject=Hello%20Cruel%20World!"
			 "&body=Share%20and%20Enjoy"
			 ];
		 */
		[[NSWorkspace sharedWorkspace] openURL:url];
	}
}

- (void)handleScriptError:(NSDictionary *)errorInfo
{
    NSString	*errorMessage	= [errorInfo objectForKey: NSAppleScriptErrorBriefMessage];
    NSNumber	*errorNumber	= [errorInfo objectForKey: NSAppleScriptErrorNumber];

    NSRunAlertPanel(NSLocalizedString(@"Script Error", @"Title on script error window."),
		[NSString stringWithFormat:@"%@: %@", NSLocalizedString(@"The script produced an error", @"Message on script error window."), errorNumber, errorMessage], 
		NSLocalizedString(@"OK", @""), 
		nil, 
		nil);
}

@end
