#include <errno.h>
#include <utime.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "ar.h"

static void
copy_stream_crc(FILE *arcfile, FILE *outfile,
                off_t remainder, uint16_t *crc)
{
    char buf[BUFSIZ];

    while (remainder != 0) {
        size_t n = (size_t)MIN(remainder, BUFSIZ);

        /* no compress */
        if (fread(buf, 1, n, arcfile) != n)
            error("Can't read");
        fwrite_crc(buf, n, outfile, crc);
        if (outfile != stdout && opts.quiet < 1) {
            putc('.', stdout);
        }
        remainder -= n;
    }
}

static void
extract(struct lzh_istream *rp, FILE *outfile, struct lzh_header *h)
{
    uint16_t crc;
    crc = INIT_CRC;

    if (opts.method->dicbit != 0)
        decode(rp, outfile, h->origsize, &crc);
    else
        copy_stream_crc(rp->fp, outfile, h->origsize, &crc);

    if ((crc ^ INIT_CRC) != h->file_crc)
        error("CRC error");
}

void
extract_to_file(struct lzh_istream *rp, struct lzh_header *h)
{
    FILE *outfile;
    char filename[1024];

    if (opts.outdir)
        makepath(filename, sizeof(filename),
                 opts.outdir, h->filename, NULL);
    else
        string_copy(filename, h->filename, sizeof(filename));

    if (memcmp(h->method, "-lhd-", sizeof(h->method)) == 0) {
        /* directory */
        if (opts.quiet < 2)
            printf("Extracting %s\n", filename);
        return;
    }

    /* create regular file */
    if (file_exists(filename)) {
        if (!opts.force_extract) {
            message("'%s' has been already exist. skip", filename);
            skip(rp->fp, h);
            return;
        }
    }

    if (mkdir_parent(filename) == -1) {
        if (errno != EEXIST)
            error("cannot make directory \"%s\"", opts.outdir);
    }

    outfile = fopen(filename, "wb");
    if (outfile == NULL) {
        fprintf(stderr, "Can't open %s (skip)\n", filename);

        skip(rp->fp, h);
        return;
    }
    if (opts.quiet < 2)
        printf("Extracting %s ", filename);

    extract(rp, outfile, h);
    fclose(outfile);

    /* adjust file information (timestamp etc) */
    {
        struct utimbuf ut;

        ut.actime = ut.modtime = h->mtime;
        utime(filename, &ut);
    }

    if (opts.quiet < 2)
        fprintf(stdout, "\n");
}

void
extract_to_stdout(struct lzh_istream *rp, struct lzh_header *h)
{
    if (memcmp(h->method, "-lhd-", sizeof(h->method)) == 0) {
        return;
    }

    if (opts.quiet < 2)
        printf("===== %s =====\n", h->filename);

    /* extract */
    extract(rp, stdout, h);
}
