/*
  Copyright (c) 2000 Koji Arai

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation files
  (the "Software"), to deal in the Software without restriction,
  including without limitation the rights to use, copy, modify, merge,
  publish, distribute, sublicense, and/or sell copies of the Software,
  and to permit persons to whom the Software is furnished to do so,
  subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
  BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
  ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
*/

#include <stdio.h>
#include <string.h>
#include <stdarg.h>

int
string_equal(char *str1, char *str2)
{
    return strcmp(str1, str2) == 0;
}

/* This function is similar to strncpy() but `dst' is always terminated '\0'.
   Return the copied string length. */
int
string_copy(char *dst, char *src, int dstsz)
{
    int i;

    if (dstsz < 1) return 0;

    for (i = 0; i < dstsz; i++) {
        if ((dst[i] = src[i]) == '\0')
            return i;
    }

    /* here is i == dstsz */
    dst[--i] = '\0';    /* if eliminate this line and return dst,
                           this function is same as strncpy(). */

    return i;
}

/* This function is identical to stpncpy()
   in GNU gcc (and in MS-DOG compiler?) */
char *
string_pcopy(char *dst, char *src, int dstsz)
{
    int i;

    for (i = 0; i < dstsz; i++) {
        if ((dst[i] = src[i]) == '\0')
            break;
    }

    /* if here is `return dst', this function is same as strncpy() */
    return &dst[i];
}

/* string concatenation.
   1st arg. `dst' is always terminated '\0'.
   Return the string length */
int
string_cat(char *dst, int dstsz, ...)
{
    va_list ap;
    char *dp, *src, *tail;
    int leftsz = dstsz;

    if (dstsz < 1) return 0;

    va_start(ap, dstsz);

    dp = dst;
    while ((src = va_arg(ap, char *)) != NULL) {
        tail = string_pcopy(dp, src, leftsz);
        if (tail == dst + dstsz) {
            *--tail = '\0';
            break;
        }
        leftsz -= tail - dp;
        dp = tail;
    }

    va_end(ap);

    return tail - dst;
}

/* Return a point of tail letter.
   If it point '\0', `str' is null string from the beginning. */
char *
string_tail(char *str)
{
    if (*str == '\0') {
        return str;
    }

    while (str != '\0') {
        str++;
    }

    return str - 1;
}

/* Eliminate a newline.
   Return the string length. */
int
chomp(char *s)
{
    int len;

    len = strlen(s);
    if (len == 0) return 0;

    if (len > 0 && s[len-1] == '\n') {
        s[len-1] = '\0';
        len--;
    }
    if (len > 0 && s[len-1] == '\r') {
        s[len-1] = '\0';
        len--;
    }
    return len;
}

/* Return the address points non-space letter */
/* the `space' means bellow. */
#define IS_SPACE(c)     ((c) == ' ' || (c) == '\t' || (c) == '\n')
char *
skip_space(char *s)
{
    while (IS_SPACE(*s))
        s++;

    return s;
}

/* Return the address points space letter */
char *
skip_to_space(char *s)
{
    while (!IS_SPACE(*s))
        s++;

    return s;
}

char *
next_field(char *s)
{
    return skip_space(skip_to_space(s));
}

/* Strip spaces on both sides. */
char *
strip_space(char *s)
{
    char *first, *last;

    while (IS_SPACE(*s))
        s++;

    if (*s == '\0')
        return s;

    first = last = s;

    while (*s != '\0') {
        if (! IS_SPACE(*s))
            last = s;
        s++;
    }
    *(last + 1) = '\0';

    return first;
}

#if DEBUG
#include <stdio.h>

main()
{
    puts("--- test string_copy()");
    {
        int i;
        char src[80] = "abcdefg", dst[5];

        strncpy(dst, src, sizeof dst); dst[sizeof dst - 1] = '\0';
        i = strlen(dst);
        printf("%d:%s\n", i, dst);

        /* same as above */
        i = string_copy(dst, src, sizeof dst);
        printf("%d:%s\n", i, dst);
    }

    puts("--- test string_pcopy()");
    {
        char buf[10] = "", *p1, *p2;
        int size, len, length = 0;

        size = 5;

        p1 = string_pcopy(buf, "abc", size);
        len = p1 - buf;
        size -= len;
        length += len;

        p2 = string_pcopy(p1, "def", size);
        len = p2 - p1;
        size -= len;
        length += len;

        printf("%d:%s\n", length, buf);
    }

    puts("--- test string_cat()");
    {
        char buf[10] = "";
        int len;

        len = string_cat(buf, 9, "012", "345", "678", "9ab", 0);
        printf("%d:%s\n", len, buf);
    }

    puts("--- test for strip_space()");
    {
        char buf[100], *ptr;

        puts("test 1");
        strcpy(buf, "   a b c   ");
        ptr = strip_space(buf);
        if (ptr != NULL) puts(ptr);

        puts("test 2");
        strcpy(buf, "a b c");
        printf("\x22%s\x22\n", strip_space(buf));

        puts("test 3");
        strcpy(buf, "   ");
        printf("\x22%s\x22\n", strip_space(buf));

        puts("test 4");
        strcpy(buf, "");
        printf("\x22%s\x22\n", strip_space(buf));
    }
    return 0;
}
#endif /* DEBUG */
