#pragma once
#include<WaveStream/Base/IWaveStream.hpp>
#include<WaveStream/Util/BiquadraticFilter.hpp>
#include<vector>

namespace LibSynthPP{
   namespace DSP{
      template<class Tsig,class Tparam>
      class BiquadraticFilterStream : public Base::IWaveStream{
	 Base::IWaveStream *_downstream,*_upstream;
	 Base::ControlMessageManager _msgMgr;
	 Base::WaveFormat *_waveFormat;
	 int64_t _position;
	 Util::BiquadraticFilter<Tsig,Tparam> **_bqf;
	 BiquadraticFilterStream(){}
      public:
	 BiquadraticFilterStream(int32_t frequency,int32_t channelNum){
	    _waveFormat=new Base::WaveFormat(frequency,channelNum);
	    _downstream=_upstream=NULL;
	    //双二次フィルタの生成
	    _bqf=new Util::BiquadraticFilter<Tsig,Tparam>*[channelNum];
	    for(int32_t i=0;i<channelNum;++i){
	       _bqf[i]=new Util::BiquadraticFilter<Tsig,Tparam>();
	    }
	    clear();
	 }
	 ~BiquadraticFilterStream(){
	    if(_downstream!=NULL){
	       disconnect(_downstream);
	    }
	    //双二次フィルタの破棄
	    int32_t channelNum=_waveFormat->getChannelNum();
	    for(int32_t i=0;i<channelNum;++i){
	       delete _bqf[i];
	    }
	    delete _bqf;
	    delete _waveFormat;
	 }
	 int64_t getPosition()const{
	    return _position;
	 }
	 const Base::WaveFormat& getOutputWaveFormat()const{
	    return *_waveFormat;
	 }
	 const Base::WaveFormat& getInputWaveFormat()const{
	    return *_waveFormat;
	 }
	 void clear(){
	    _sendControlMessage(new Base::ControlMessage(-1,this,0,NULL));
	 }
	 void reset(){
	    _sendControlMessage(new Base::ControlMessage(-2,this,0,NULL));
	 }
	 void connect(IWaveStream *downstream){
	    if(_downstream!=NULL){
	       THROW(Base::invalid_operation,"Already Connected.");
	    }else{
	       downstream->_sendControlMessage(
		  new Base::ControlMessage(-3,this,0,NULL)
		  );
	       _downstream=downstream;
	    }
	 }
	 void disconnect(IWaveStream *downstream){
	    if(_downstream != downstream){
	       THROW(Base::invalid_argument,"Not Connected.");
	    }else{
	       downstream->_sendControlMessage(
		  new Base::ControlMessage(-4,this,0,NULL)
		  );
	       _downstream=NULL;
	    }
	 }
	 void _write(IWaveStream *_writerStream,void *signal, int64_t length,int64_t offset){
	    //双二次フィルタを適用して出力
	    if(_downstream==NULL){
	       THROW(Base::invalid_operation,"Not Connected.");
	    }
	    Tsig *ptr=(Tsig*)signal;
	    int32_t channelNum=_waveFormat->getChannelNum();
	    int64_t processed=0;
	    //制御メッセージ取得
	    std::list<Base::ControlMessage*> *msgList=_msgMgr.getMessage(_position+length);
	    for(std::list<Base::ControlMessage*>::iterator iter=msgList->begin();
		iter!=msgList->end();++iter){
	       int64_t size=(*iter)->getTime()-processed-_position;
	       for(int64_t i=0;i<size*channelNum;++i,++ptr){
		  *ptr=_bqf[i%channelNum]->update(*ptr);
	       }
	       if((*iter)->getMessageID()>0 && (*iter)->getMessageID()<=10){
		  Tparam *paramArr=(Tparam*)(*iter)->getParam();
		  for(int32_t i=0;i<_waveFormat->getChannelNum();++i){
		     switch((*iter)->getMessageID()){
			case 1://パラメータリセット
			   _bqf[i]->resetParam();
			   break;
			case 2://ローパスフィルタ設定
			   _bqf[i]->setLopassParam(paramArr[0],paramArr[1],paramArr[2]);
			   break;
			case 3://ハイパスフィルタ設定
			   _bqf[i]->setHighpassParam(paramArr[0],paramArr[1],paramArr[2]);
			   break;
			case 4://バンドパスフィルタ設定1
			   _bqf[i]->setBandpass1Param(paramArr[0],paramArr[1],paramArr[2]);
			   break;
			case 5://バンドパスフィルタ設定2
			   _bqf[i]->setBandpass2Param(paramArr[0],paramArr[1],paramArr[2]);
			   break;
			case 6://バンドストップフィルタ設定
			   _bqf[i]->setBandstopParam(paramArr[0],paramArr[1],paramArr[2]);
			   break;
			case 7://オールパスフィルタ設定
			   _bqf[i]->setAllpassParam(paramArr[0],paramArr[1],paramArr[2]);
			   break;
			case 8://ピーキングフィルタ設定
			   _bqf[i]->setPeakingParam(paramArr[0],paramArr[1],paramArr[2],paramArr[3]);
			   break;
			case 9://ローシェルフフィルタ設定
			   _bqf[i]->setLoshelfParam(paramArr[0],paramArr[1],paramArr[2],paramArr[3]);
			   break;
			case 10://ハイシェルフフィルタ設定
			   _bqf[i]->setHighshelfParam(paramArr[0],paramArr[1],paramArr[2],paramArr[3]);
			   break;
		     }		   
		  }
		  if(paramArr!=NULL){
		     delete paramArr;
		  }
	       }else if((*iter)->getMessageID()==0){
		  break;
	       }

	       processed+=size;
	       delete *iter;
 	    }
	    delete msgList;
	    _downstream->_write(this,signal,length,offset);
	    _position+=length;
	 }

	 void _sendControlMessage(Base::ControlMessage *c){
	    switch(c->getMessageID()){
	       case -1: //clear
		  _position=0;
		  _msgMgr.clear();
	       case -2: //reset。clearから流れてくる。
		  for(int32_t i=0;i<_waveFormat->getChannelNum();++i){
		     _bqf[i]->clear();
		  }
		  break;
	       case -3: //connect
		  //チャネル数が一致すれば接続可能。
	       {
		  Base::IWaveStream *sender=c->getSender();
		  if(!(sender->getOutputWaveFormat()==_waveFormat)){
		     THROW(Base::invalid_operation,"Wave format missmatch.");
		  }
		  _upstream=sender;
	       }
	       break;
	       case -4: //disconnect
	       {
		  //接続中のストリームであれば、切断する。
		  //下位ストリームと接続中は切断不可。
		  Base::IWaveStream *sender=c->getSender();
		  if(sender!=_upstream){
		     THROW(Base::invalid_operation,"Not Connected.");
		  }else{
		     _upstream=NULL;
		  }
	       }
	       break;
	       case 1://パラメータリセット
	       case 2://ローパスフィルタ設定
	       case 3://ハイパスフィルタ設定
	       case 4://バンドパスフィルタ設定1
	       case 5://バンドパスフィルタ設定2
	       case 6://バンドストップフィルタ設定
	       case 7://オールパスフィルタ設定
	       case 8://ピーキングフィルタ設定
	       case 9://ローシェルフフィルタ設定
	       case 10://ハイシェルフフィルタ設定
		  _msgMgr.addMessage(c);
		  break;
	       default:
		  THROW(Base::invalid_argument,"Undefined message.");
	    }
	 }
	 //パラメータリセット
	 void setDefaultParam(int64_t time){
	    _sendControlMessage(new Base::ControlMessage(1,this,time,NULL));
	 }
	 //ローパスフィルタ設定
	 void setLopassParam(int64_t time, Tparam cutOffFreq, Tparam Q) {
	    Tparam *paramArr=new Tparam[3];
	    paramArr[0]=(Tparam)_waveFormat->getFrequency();
	    paramArr[1]=cutOffFreq;
	    paramArr[2]=Q;
	    _sendControlMessage(new Base::ControlMessage(2,this,time,paramArr));
	 }
	 //ハイパスフィルタ設定
	 void setHighpassParam(int64_t time,Tparam cutOffFreq,Tparam Q) {
	    Tparam *paramArr=new Tparam[3];
	    paramArr[0]=(Tparam)_waveFormat->getFrequency();
	    paramArr[1]=cutOffFreq;
	    paramArr[2]=Q;
	    _sendControlMessage(new Base::ControlMessage(3,this,time,paramArr));
	 }
	 //バンドパス1設定
	 void setBandpass1Param(int64_t time,Tparam centerFreq, Tparam BW) {
	    Tparam *paramArr=new Tparam[3];
	    paramArr[0]=(Tparam)_waveFormat->getFrequency();
	    paramArr[1]=centerFreq;
	    paramArr[2]=BW;
	    _sendControlMessage(new Base::ControlMessage(4,this,time,paramArr));
	 }
	 //バンドパス2設定
	 void setBandpass2Param(int64_t time, Tparam centerFreq, Tparam BW) {
	    Tparam *paramArr=new Tparam[3];
	    paramArr[0]=(Tparam)_waveFormat->getFrequency();
	    paramArr[1]=centerFreq;
	    paramArr[2]=BW;
	    _sendControlMessage(new Base::ControlMessage(5,this,time,paramArr));
	 }
	 //バンドストップ設定
	 void setBandstopParam(int64_t time,Tparam centerFreq,Tparam BW) {
	    Tparam *paramArr=new Tparam[3];
	    paramArr[0]=(Tparam)_waveFormat->getFrequency();
	    paramArr[1]=centerFreq;
	    paramArr[2]=BW;
	    _sendControlMessage(new Base::ControlMessage(6,this,time,paramArr));
	 }
	 //オールパス設定
	 void setAllpassParam(int64_t time,Tparam cutOffFreq,float BW) {
	    Tparam *paramArr=new Tparam[3];
	    paramArr[0]=(Tparam)_waveFormat->getFrequency();
	    paramArr[1]=cutOffFreq;
	    paramArr[2]=BW;
	    _sendControlMessage(new Base::ControlMessage(7,this,time,paramArr));
	 }
	 //ピーキング設定
	 void setPeakingParam(int64_t time,Tparam centerFreq,Tparam BW,Tparam gain) {
	    Tparam *paramArr=new Tparam[4];
	    paramArr[0]=(Tparam)_waveFormat->getFrequency();
	    paramArr[1]=centerFreq;
	    paramArr[2]=BW;
	    paramArr[3]=gain;
	    _sendControlMessage(new Base::ControlMessage(8,this,time,paramArr));
	 }
	 //ローシェルフ設定
	 void setLoshelfParam(int64_t time,Tparam cutOffFreq,Tparam S,Tparam gain) {
	    Tparam *paramArr=new Tparam[4];
	    paramArr[0]=(Tparam)_waveFormat->getFrequency();
	    paramArr[1]=cutOffFreq;
	    paramArr[2]=S;
	    paramArr[3]=gain;
	    _sendControlMessage(new Base::ControlMessage(9,this,time,paramArr));
	 }
	 //ハイシェルフ設定
	 void setHighshelfParam(int64_t time,Tparam cutOffFreq,Tparam S,Tparam gain) {
	    Tparam *paramArr=new Tparam[4];
	    paramArr[0]=(Tparam)_waveFormat->getFrequency();
	    paramArr[1]=cutOffFreq;
	    paramArr[2]=S;
	    paramArr[3]=gain;
	    _sendControlMessage(new Base::ControlMessage(10,this,time,paramArr));
	 }
      };
   };
};
