#pragma once
#include<cstring>
#include<WaveStream/Base/typedef.hpp>
#include<WaveStream/Base/Exception.hpp>
#include<list>
#include<utility>

namespace LibSynthPP {
   namespace Util {
      //信号を蓄積するバッファ
      template<class T>
      class WaveBuffer{
	 int32_t _channel_num;
	 int64_t _length;
	 std::list<std::pair<T*,int64_t> >buf;
      public:
	 //コンストラクタ。
	 WaveBuffer(int32_t channel_num) {
	    if (channel_num<=0) {
	       THROW(Base::invalid_argument,"Invalid channel num.");
	    } else {
	       _channel_num=channel_num;
	       _length=0;
	    }
	 }
	 //デストラクタ
	 ~WaveBuffer(){
	    clear();
	 }
	 //チャネル数取得
	 int32_t getChannelNum() const{
	    return _channel_num;
	 }
	 //バッファに格納されている信号長
	 int64_t getLength()const{
	    return _length;
	 }
	 //バッファをクリア
	 void clear(){
	    for(typename std::list<std::pair<T*,int64_t> >::iterator iter=buf.begin();
		iter!=buf.end();++iter){
	       delete (*iter).first;
	    }
	    _length=0;
	 }
	 //信号の書き込み。
	 void write(T *signal,int64_t length) {
	    if(length<0){
	       THROW(Base::invalid_argument,"Invalid length.");
	    }
	    buf.push_back(std::pair<T*,int64_t>(signal,length));
	    _length+=length;
	 }
	 //信号の切り出し。
	 T* read(int64_t length) {
	    T *ret=NULL;
	    if (length<0 || length>_length) {
	       THROW(Base::invalid_argument,"Invalid length.");
	    }
	    ret=new T[length*_channel_num];
	    if(length!=0){
	       for (int64_t pos=0;pos<length;) {
		  std::pair<T*,int64_t> tmp=buf.front();
		  buf.pop_front();
		  int tmpLength=tmp.second;
		  if (tmpLength+pos<=length) {
		     //ぴったり収まるか、足りない場合。
		     for (int64_t j=0;j<_channel_num;++j) {
			for (int64_t i=0;i<tmpLength;++i) {
			   ret[(pos+i)*_channel_num+j]=tmp.first[i*_channel_num+j];
			}
		     }
		     delete tmp.first;
		     pos+=tmpLength;
		  } else {
		     //tmpの方が大きく、はみ出る場合。
		     //まずはretにコピー
		     for(int64_t j = 0;j < _channel_num;++j) {
			for(int64_t i = 0;i < length-pos;++i) {
			   ret[(pos+i)*_channel_num+j] = tmp.first[i*_channel_num+j];
			}
		     }
		     //残った部分を戻す
		     T *tmp2 = new T[(tmpLength - length + pos) * _channel_num];
		     for(int64_t j = 0;j < _channel_num;++j) {
			for(int64_t i = 0;i < tmpLength - length + pos;++i) {
			   tmp2[i*_channel_num+j] = tmp.first[(i+length-pos)*_channel_num+j];
			}
		     }
		     delete tmp.first;
		     buf.push_front(std::pair<T*,int64_t>(tmp2,tmpLength - length + pos));
		     pos += length - pos;
		  }
	       }
	       _length-=length;
	    }
	    return ret;
	 }
      };
      //信号をミキシングするバッファ
      template<class T>
      class WaveMixerBuffer {
	 int32_t _channel_num;
	 int64_t _length;
	 std::list<std::pair<T*,int64_t> >buf;
      public:
	 //コンストラクタ。
	 WaveMixerBuffer(int32_t channel_num) {
	    if (channel_num<=0) {
	       THROW(Base::invalid_argument,"Invalid channel num.");
	    } else {
	       _channel_num=channel_num;
	       _length=0;
	    }
	 }
	 //デストラクタ
	 ~WaveMixerBuffer(){
	    clear();
	 }
	 //チャネル数取得
	 int32_t getChannelNum() const{
	    return _channel_num;
	 }
	 //バッファに格納されている信号長
	 int64_t getLength() const{
	    return _length;
	 }
	 //バッファをクリア                                                                 
	 void clear(){
	    for(typename std::list<std::pair<T*,int64_t> >::iterator iter=buf.begin();
		iter!=buf.end();++iter){
	       delete (*iter).first;
	    }
	    _length=0;
	 }
	 //信号の書き込み。
	 void write(T *signal,int64_t length,int64_t offset) {
	    if(length>0){
	       //まずは、足りない分のスペースを追加。
	       if(offset + length > _length) {
		  //スペース追加により、この後でバッファが足りなくなることはなくなる。
		  T *tmp=new T[(offset + length - _length)*_channel_num];
		  std::memset(tmp,0,sizeof(T)*(offset + length - _length)*_channel_num);
		  buf.push_back(std::pair<T*,int64_t>(tmp,offset + length - _length));
		  _length+=offset + length - _length;
	       }

	       //次に、イテレータを取得して書き込み対象まで頭出し。
	       typename std::list<std::pair<T*,int64_t> >::iterator iter=buf.begin();
	       int64_t pos=0;
	       while(++iter!=buf.end()){
		  int64_t l=(*iter).second;
		  if(pos+l>= offset){
		     break;
		  }
		  pos+=l;
	       }
	       --iter;
	       pos=0;
	       int64_t ofs=offset-pos,rest=length;
	       while(rest!=0){
		  int64_t l=(*iter).second;
		  for(int64_t i=ofs;i<l && rest!=0;++i,--rest,++pos){
		     for(int64_t j=0;j<_channel_num;++j){
			(*iter).first[i*_channel_num+j]+=signal[pos*_channel_num+j];
		     }
		  }
		  ofs=0;
		  ++iter;
	       }

	       delete signal;
	    }else if(length==0){
	       //加算する必要がないので
	       delete signal;
	    }else{
	       THROW(Base::invalid_argument,"Invalid length.");
	    }
	 }
	 //信号の切り出し。
	 T* read(int64_t length) {
	    T *ret=NULL;
	    if (length<0) {
	       THROW(Base::invalid_argument,"Invalid length.");
	    }
	    //まずは、足りない分のスペースを追加。
	    if( length > _length) {
	       //スペース追加により、この後でバッファが足りなくなることはなくなる。
	       T *tmp=new T[(length - _length)*_channel_num];
	       std::memset(tmp,0,sizeof(T)*(length - _length)*_channel_num);
	       buf.push_back(std::pair<T*,int64_t>(tmp,length - _length));
	       _length+= length - _length;
	    }
	    ret=new T[length*_channel_num];
	    if(length!=0){
	       for (int64_t pos=0;pos<length;) {
		  std::pair<T*,int64_t> tmp=buf.front();
		  buf.pop_front();
		  int tmpLength=tmp.second;
		  if (tmpLength+pos<=length) {
		     //ぴったり収まるか、足りない場合。
		     for (int64_t j=0;j<_channel_num;++j) {
			for (int64_t i=0;i<tmpLength;++i) {
			   ret[(pos+i)*_channel_num+j]=tmp.first[i*_channel_num+j];
			}
		     }
		     delete tmp.first;
		     pos+=tmpLength;
		  } else {
		     //tmpの方が大きく、はみ出る場合。
		     //まずはretにコピー
		     for(int64_t j = 0;j < _channel_num;++j) {
			for(int64_t i = 0;i < length-pos;++i) {
			   ret[(pos+i)*_channel_num+j] = tmp.first[i*_channel_num+j];
			}
		     }
		     //残った部分を戻す
		     T *tmp2 = new T[(tmpLength - length + pos) * _channel_num];
		     for(int64_t j = 0;j < _channel_num;++j) {
			for(int64_t i = 0;i < tmpLength - length + pos;++i) {
			   tmp2[i*_channel_num+j] = tmp.first[(i+length-pos)*_channel_num+j];
			}
		     }
		     delete tmp.first;
		     buf.push_front(std::pair<T*,int64_t>(tmp2,tmpLength - length + pos));
		     pos += length - pos;
		  }
	       }
	       _length-=length;
	    }
	    return ret;
	 }
      };
   };
};
