;;; yasgml.el --- Yet Another SGML mode for LinuxDOC DTD

;; Copyright (C) 1999 by Free Software Foundation, Inc.

;; Author: Takashi Nishimoto <g96p0935@mse.waseda.ac.jp>
;; Keywords: sgml

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; (from the Debian package maintainer:
;;   Original yasgml.el has comments but it is all written in Japanese.
;;   So I translate it here into English to make it readable for more
;;   people. If you found the mistake in English sentences, please
;;   tell me <sano@debian.org> first using BTS or direct e-mail :)
;;
;; This is Yet Another SGML Mode for YaTeX users.
;;
;; Once writing SGML source, You can get the document in various
;; formats such as plain text, LaTeX, HTML, info, etc.
;;
;; I use YaTeX, a famous major-mode for LaTeX writing, and yahtml which 
;; comes with YaTeX for HTML writing. yahtml has so similar operation
;; to YaTeX that YaTeX user feels easy to use yahtml. 
;; When I learned SGML, I was going to use YaTeX like major-mode for 
;; SGML writing, but I can not found anyone, so I created this yasgml.
;;
;; This yasgml is written for LinuxDoc DTD. LinuxDoc DTD is created
;; for Linux HOWTO documents, but you can use it for general documents.
;;
;; (from the Debian package maintainer: 
;;   LinuxDoc DTD is now obsolete, and DocBook DTD is recommended.
;;   I have tried to contact the author to ask to support DocBook DTD,
;;   but I don't get his reply. I'll try to use psgml, and then decide
;;   to switch it or to enhance this by myself.)
;;
;; If you find bugs, or if you have suggestions, please mail me,
;; to <g96p0935@mse.waseda.ac.jp> I will support users' request.
;;
;; P.S.
;; I use memo-mode.el by Oshiro, to write the comments here.
;; It shows logical structure of sentences using left-margins and arrows.
;; Equal-indentation means the parallel or order relation.
;; memo-mode.el can be downloaded from:
;;  http://mibai.tec.u-ryukyu.ac.jp/%7Eoshiro/Programs/elisp/memo-mode.el

;; YaTeX 桼Τ sgml mode Ǥ
;;
;; SGML ȤΤϡ HTML ˻ʸˡΥ** plain text 
;; LaTeXHTMLinfo ʤɤΤΥեޥåȤʸͥʪ
;; ǤĤޤꡢĤΥ饪饤ʸ񡢤줤ʰʪۡ
;; ڡκ뤳ȤǤʸݼΩޤ
;;
;; LaTeX 񤯤Τ YaTeX ȤȤ䤹ͭ̾ major-mode 
;; ޤ HTML 񤯤Τˤ yahtml ޤȤˡ 
;; yahtml  YaTeX Ʊ褦ˡˤʤäƤ뤿ᡢ YaTeX 桼
;; ϤʤޤYaTeX 桼 SGML ʸ񤯤Ȥ YaTeX 
;; Ӥ major-mode ̵ΤȤȤˤʤޤǰ
;; ¸ߤʤ褦ʤΤǡ˻ޤ
;;
;;  yasgml ϡ linuxdoc DTD ˹碌ƺäƤޤ linuxdoc DTD 
;; ϡȤ Linux  HOWTO ʸ񤯤Τ˺줿ΤǤǤ
;; ̤ʸ񤭤ˤȤޤ
;;
;; Хոʤɤޤ顢 g96p0935@mse.waseda.ac.jp ޤǥ᡼
;; ꤤޤǤ¤˾ˤϱ褦ȻפäƤޤ
;;
;; ɥȽ񤭤ˤ뤵 memo-mode.el ѤƤޤ
;; ޡʸϤ¤ΤˤΤǤƱ̤
;; ǥȤʤ顢󤢤뤤ϽطɽƤޤ
;; memo-mode.el  
;; http://mibai.tec.u-ryukyu.ac.jp/%7Eoshiro/Programs/elisp/memo-mode.el
;; ˤޤ
;;

;;; Installation;

;;-->You have to install YaTeX. This yasgml.el use YaTeX and yahtml.
;;   yahtml is included in yatex package.
;;   
;;-->Then, install sgml-tools (v1)
;;   If you write SGML source in Japanese language and convert it
;;   using sgml2txt of sgml-tools, you needs JF patch. 
;;   That patch can be downloaded from
;;   http://jf.linux.or.jp/workshop/archives/sgml-tools-1.0.9j.diff.gz
;;   as well as original sgml-tools (1.0.9) tar archive:
;;   http://jf.linux.or.jp/workshop/archives/sgml-tools-1.0.9.tar.gz
;;   
;;-->Copy this file (yasgml.el) to some place specified in load-path.
;;   
;;-->Add settings for yasgml into your .emacs
;;  -->Add following two at first.
;;     (setq auto-mode-alist
;;       (nconc
;;         '(("\\.sgml$" . yasgml-mode)) auto-mode-alist))
;;     (autoload 'yasgml-mode "yasgml" "Major mode for editing files of SGML." t)
;;  -->See "About variables for customization" for other settings.
;;     
;;-->Byte Compile this
;;  -->execute "M-x byte-compile yasgml.el"

;;-->YaTeX 򥤥󥹥ȡ뤷ƤΥץ YaTeX Ȥ
;;   ° yahtml Ȥޤ
;;   
;;-->sgml-tools 򥤥󥹥ȡ뤷ƤꥸʥΤޤޤǤܸ
;;   бƤޤΤǡܸѥåƤ뤳ȡ
;;   ftp://jf.linux.or.jp/pub/JF/misc/sgml-tools/ ˤޤ
;;   
;;-->Υե (yasgml.el)  load-path ̤äǥ쥯ȥ˥ԡޤ
;;   
;;-->.emacs ꤷޤ
;;  -->(setq auto-mode-alist
;;       (nconc
;;         '(("\\.sgml$" . yasgml-mode)) auto-mode-alist))
;;     (autoload 'yasgml-mode "yasgml" "Major mode for editing files of SGML." t)
;;     2Ԥ .emacs ˲äޤ
;;  -->¾ΥޥˤĤƤϡ֥ޥǽѿˤĤơ
;;     򻲾ȤΤȡ
;;     
;;-->Хȥѥ뤷ޤ
;;  -->M-x byte-compile (Υե) ¹Ԥ뤳ȡ

;;-->about variables for customization : default is shown in ()
;;
;;  -->yasgml-author (format "%s <tt>&lt;%s&gt;</tt>" (user-full-name) user-mail-address)
;;    -->the author of the SGML documents
;;
;;  -->yasgml-default-article ("article")
;;    -->Default style for the SGML documents
;;
;;  -->yasgml-default-dtd ("linuxdoc")
;;    -->Default DTD (but now this can be used for LinuxDOC DTD only)
;;
;;  -->yasgml-prefix ("\C-c")
;;    -->Prefix-Key for yasgml
;;
;;  -->yasgml-fill-column (72)
;;    -->fill-column for yasgml
;;
;;  -->yasgml-fill-prefix (nil)
;;    --> fill-prefix for yasgml
;;
;;  -->yasgml-environment-indent (1)
;;    -->level (depth) of environment indent
;;
;;  -->yasgml-special-characters-input-method ('auto)
;;    -->Method for input special characters: either 'auto or 'bell
;;       please read "input method for special characters"
;;
;;  -->yasgml-do-special-characters-check (t)
;;    -->do or do not simple check for special characters before typesetting
;;
;;  -->yasgml-use-linuxdoc-sgml-ja (if (file-exists-p "/usr/bin/sgmls") t nil)
;;    -->use linuxdoc-sgml-ja or sgml-tools(v1). currently (1999 Dec. 06)
;;       the Debian package of sgml-tools(v1) can not handle Japanese characters
;;       correctly, and users use JP Package of linuxdoc-sgml-ja for stable
;;       (slink) system. Maybe we will not need this variable after release
;;       of potato with Japanese enhanced sgml-tools(v1), but now we do.
;;
;;  -->yasgml-typeset-program  (if yasgml-use-linuxdoc-sgml-ja 
;;                              "sgml2txt -f" "sgml2txt -f -l ja -c nippon")
;;    -->this is for typeset command. "-l ja -c nippon" is required
;;       only for Japanese SGML command, so you have to set this value
;;       differently if you don't use Japanese.
;;       You can preview text file generated by this command using
;;       "[prefix] tp"
;;
;;  -->yasgml-format-program-alist ("default is too long to show here" ;)
;;    -->This is alist for various formats and commands for them
;;
;;  -->yasgml-preview-function ('yasgml-show-txt-other-window)
;;    -->control the way to show buffer of preview result.
;;
;;      -->'yasgml-show-txt-other-window
;;           : split screen and show the preview on another window
;;
;;      -->'yasgml-show-txt-other-frame
;;           : show the preview on the other frame
;;
;;      -->'yasgml-show-txt-bully
;;           : show preview buffer in 1 window


;;-->ޥǽѿˤĤ  ()ϥǥե
;;
;;  -->yasgml-author (format "%s <tt>&lt;%s&gt;</tt>" (user-full-name) user-mail-address)
;;    -->SGMLʸκ(ʤʬ)̾
;;
;;  -->yasgml-default-article ("article")
;;    -->SGMLʸΥǥեȥ
;;
;;  -->yasgml-default-dtd ("linuxdoc")
;;    -->ǥեȤ DTDޤ linuxdoc ˤбƤޤ
;;
;;  -->yasgml-prefix ("\C-c")
;;    -->yasgml ΥץեåǤ
;;
;;  -->yasgml-fill-column (72)
;;    -->yasgml ˤ fill-column Ǥ
;;
;;  -->yasgml-fill-prefix (nil)
;;    -->yasgml ˤ fill-prefix Ǥ
;;
;;  -->yasgml-environment-indent (1)
;;    -->ĶΥǥȤοǤ
;;
;;  -->yasgml-special-characters-input-method ('auto)
;;    -->üʸˡǤ 'auto  'bell Τɤ餫Ǥ
;;       üʸˡפ򻲾ȤΤ
;;
;;  -->yasgml-do-special-characters-check (t)
;;    -->ץåüʸˤĤƤδñʥåԤɤ
;;
;;  -->yasgml-use-linuxdoc-sgml-ja (if (file-exists-p "/usr/bin/sgmls") t nil)
;;    -->JP ѥå linuxdoc-sgml-ja Ȥ뤤 JF ѥå
;;       Ƥ sgml-tools(v1) Ȥɤ餫򤷤ޤ
;;       Debian ѥå sgml-tools(v1)  JF ѥåޡ
;;       potato ꡼ǤСѿϤ餯ɬ̵ʤǤ礦
;;
;;  -->yasgml-typeset-program  (if yasgml-use-linuxdoc-sgml-ja 
;;                              "sgml2txt -f" "sgml2txt -f -l ja -c nippon")
;;    -->ץåȤԤޥɤǤ
;;       "[prefix] tp" Ǥˤä줿 .txt СΥץӥ塼
;;       Ԥޤ
;;
;;  -->yasgml-format-program-alist (ĹΤǾά)
;;    -->եޥåȤȤ뤿Υޥɤ alist Ǥ
;;
;;  -->yasgml-preview-function ('yasgml-show-txt-other-window)
;;    -->ץӥ塼η̤ɽХåեɤΤ褦ɽ뤫
;;       ޤ
;;      -->'yasgml-show-txt-other-window ʤ顢̤2ʬ䤷ɽޤ
;;      -->'yasgml-show-txt-other-frame ʤ顢̤ʥե졼ɽޤ
;;      -->'yasgml-show-txt-bully ʤ顢̤ǥץӥ塼Хåեɽ
;;         ޤ

;;-->Input Method for Special Characters
;;
;;  -->Some characters ( &, <, >, $, #, %, ", ~, and </ ) has special
;;     meanings in SGML source, therefore when they are needed in SGML
;;     source, author have to check their expression in SGML and convert
;;     special chacters into them. This yasgml does this conversion for
;;     you. 
;;
;;     (when in the environment of verb, code, or quote, this restriction
;;      is loosen, but & and </ can not be entered even there. They are
;;      to be converted into &amp; and &etago;)
;;
;;  -->default setting is automatic conversion ON
;;    -->i.e. set 'auto in yasgml-special-characters-input-method
;;    -->with default setting, when entered '&', it will be converted
;;       &amp; always.
;;    -->When enter / after < ("</"), they are converted into &etago; 
;;    -->for other special characters:
;;      -->when in the environment of verbcodequote, the entered
;;         character is not converted.
;;      -->otherwise, the character is converted automatically.
;;
;;    -->If you mind this default setting, then add
;;         (setq yasgml-special-characters-input-method 'bell)
;;       into your .emacs. This makes conversion off, and yasgml
;;       warn by ringing bell when the special character is entered.

;;-->üʸˡ
;;  -->SGML ˤơ &<>$#%"~</ üʸȤư 
;;     ޤ餽ΤޤϤǤޤ verbcodequote ĶǤ
;;      󤬴ˤʤޤϤ &  </ ϤΤޤޤǤϤǤ
;;      줾 &amp;&etago; (ʲ &foo; त&󥹡
;;     ȸƤӤޤ)ϤʤФʤޤüʸˤĤƲͤ
;;     ʸ񤤤ƤȡĤδ֤ˤüʸ򤽤ΤޤϤƤޤ
;;     ǽ⤤Ǥ⡢顼򸫤ĤΤϤȤƤ⺤Ǥ
;;     ϤǵդСä顼ǽ
;;     㤯ʤޤ
;;  -->ǥեȤǤüʸΥáȤʸʸɤ
;;     ˡ׽ФʸϤޤ
;;    -->yasgml-special-characters-input-method  'auto ˤʤäƤޤ
;;    -->& 򲡤ȤϤĤǤ &amp; Ϥޤ
;;      -->Ͻ鿴ԤΤθǤΤǡʤС
;;         (setq yasgml-special-characters-input-method 'bell)
;;          .emacs ˲äƤξ֤Ǥ & 򲡤Ȥ٥
;;         Ĥꡢ & üʸǤ뤳Ȥٹ𤹤褦ˤʤޤ
;;    -->< θ / ϤȤưŪ &etago; Ѥޤ
;;    -->ʳʸˤĤ
;;      -->⤷ verbcodequote ĶˤȤϤʸ򤽤Τ
;;         ޽Ϥޤ
;;      -->ǤʤȤϡʸб & Ϥ
;;         


;;; Usage:

;; In the following description, [prefix] is C-c key as default.
;; (you can customize this key using variable 'yasgml-prefix, see above)
;;
;;  * [prefix] b X	insert environment tag such as <enum> and </enum>
;;
;;  * [prefix] s	insert following tags with interactive prompt:
;;                        <url url="http://aaaa.com" name="test">
;;                        <sect>Section Title<p>
;;  * [prefix] l	insert tags for textstyle such as `<em> ... </em>' 
;;                      or  `<tag> ... </tag>'
;;
;;  * [prefix] m	insert tags such as `<toc>', `<item> ', or `<p>'
;;
;;  * [prefix] p	insert <p> at cursor position
;;  * [prefix] g	jump to the matching tag :
;;                        (<verb> <-> </verb>, or url="bbb.html": 
;;                         in this case, the place to go is bbb.html)
;;
;;  * [prefix] k	erase SGML tag at the position
;;			if C-u is entered as a prefix, then the contents
;;                      enclosed that tag are erased at once.
;;
;;  * [prefix] c	change the tag at the position

;;  * [prefix] t j	exec sgml2txt for current file.
;;  * [prefix] t p	preview text file generated by SGML source
;;  * [prefix] a	this is the same as completion of accent in YaTeX
;;			you can enter &lt;, &gt;, etc using this.
;;
;;  * Here is the tags completed by "[prefix] b" : short-cut key is shown in ()
;;                        abstract ([prefix] b a)
;;                        code     ([prefix] b c)
;;                        descrip  ([prefix] b d)
;;                        enum     ([prefix] b e)
;;                        itemize  ([prefix] b i)
;;                        quote    ([prefix] b q)
;;                        tscreen  ([prefix] b t)
;;                        verb     ([prefix] b v)
;;
;;  * Here is the tags completed by "[prefix] l"
;;      tag, bf, cparam, em, it, sf, sl, tt, bf
;;
;;  * Here is the tags completed by "[prefix] m"
;;      item, p, toc, idx, cdx, nidx, ncdx, file
;;
;;  * Here is the tags completed by "[prefix] s"
;;      sect, sect1, sect2, sect3, sect4, url, htmlurl, label, ref

;; ʲˤơä˥ޥ򤷤Ƥʤ¤ꡢ[prefix] 
;; C-c ̣ޤ
;;  * [prefix] b X	`</enum>' ȤäλɬפȤʤ`enum'Τ褦
;;			ʴĶ䴰Ϥޤ
;;  * [prefix] s	ʲΤ褦䴰Ԥޤ
;;                        <url url="http://aaaa.com" name="test">
;;                        <sect>̾<p>
;;  * [prefix] l	`<em> ... </em>'  `<tag> ... </tag>' Τ褦
;;			ʥƥȥΥ䴰ޤ
;;			䴰ǽ̾ [prefix] b 䴰ǤΤ
;;			ǽ񤭤ȤˤѤ뤳Ȥޤ
;;  * [prefix] m	`<toc>'  `<item> '`<p>' ñΥ䴰
;;			Ԥޤ
;;  * [prefix] p	֤<p>ޤ
;;  * [prefix] g	б륿<verb> <-> </verb>  url="xxx" 
;;			褦 TAG ˥פޤ
;;                        url=hoge.html ξhoge.htmlӤ
;;			
;;  * [prefix] k	ݥȾ SGML õޤ
;;			⤷ universal-argument դ(C-u˲
;;			)SGMLǰϤޤ줿ƤƱ˾õޤ
;;  * [prefix] c	ݥȾΥѹޤ
;;  * [prefix] t j	ȥեФ sgml2txt ƤӤޤ
;;  * [prefix] t p	SGML 줿 txt ǥץӥ塼ޤ
;;  * [prefix] a	YaTeX Υȵ䴰ƱǤ
;;			&lt; &gt; ϤǤޤ
;;
;;  * [prefix] b 䴰Ǥ륿()ϥ硼ȥåȥǤ
;;                        abstract ([prefix] b a)
;;                        code     ([prefix] b c)
;;                        descrip  ([prefix] b d)
;;                        enum     ([prefix] b e)
;;                        itemize  ([prefix] b i)
;;                        quote    ([prefix] b q)
;;                        tscreen  ([prefix] b t)
;;                        verb     ([prefix] b v)
;;  * [prefix] l 䴰Ǥ륿
;;                        tag
;;                        bf
;;                        cparam
;;                        em
;;                        it
;;                        sf
;;                        sl
;;                        tt
;;                        bf
;;                        [prefix] b 䴰Ǥ륿٤
;;  * [prefix] m 䴰Ǥ륿
;;                        item
;;                        p
;;                        toc
;;                        idx
;;                        cdx
;;                        nidx
;;                        ncdx
;;                        file
;;  * [prefix] s 䴰Ǥ륿
;;                        sect
;;                        sect1
;;                        sect2
;;                        sect3
;;                        sect4
;;                        url
;;                        htmlurl
;;                        label
;;                        ref

;;; *****Attention*****

;; Currently, sgml-tools (v1) with JF patch can be used to convert
;; SGML source written in Japanese, only into Plain text and HTML.
;; But recently the report of using sgml2latex successfully to convert
;; Japanese SGML into LaTeX is posted on some mailing list.
;; So maybe soon we can use sgml2latex as well as sgml2txt and sgml2html.
;; (from the Debian maintainer: we should transfer to sgmltools-v2 or
;;  jade and go from linuxdoc DTD to DocBook DTD. So working sgml-tools
;;  (v1) too much is not recommended. I wish only to implement current 
;;  support level in Debian so that users of Debian can use tools which
;;  they can use on other distributions.)

;; ߤΤȤ sgml-tools ܸбƤեޥåȤϡ 
;; Plain text  HTML ǤߤΤȤ Linux 
;; ʸ SGML 塹ȿʤǤꡢ sgml2info  sgml2latex ܸ
;; Сܸǽ񤫤줿 SGML ʸ⤢եޥåȤѴ
;; 뤳ȤǤ

;;; Code:

(require 'yahtml)
(require 'yatex)

;;; --- customizable variable starts here ---
(defvar yasgml-author (format "%s <tt>&lt;%s&gt;</tt>" (user-full-name) user-mail-address)
  "*Author of SGML document.")
(defvar yasgml-default-style "article"
  "*Default style of SGML document.")
(defvar yasgml-default-dtd "linuxdoc"
  "*Default DTD of SGML document. ")

(defvar yasgml-prefix "\C-c"
  "*Prefix key stroke of yasgml functions.")
(defvar yasgml-fill-column 72 "*fill culumn used for yasgml-mode")
(defvar yasgml-fill-prefix nil "*fill prefix for yasgml-mode")
(defvar yasgml-environment-indent 1
  "*Indentation depth of SGML's listing environment")
(defvar yasgml-special-characters-input-method 'auto
  "*How to input special characters, e.g. &amp; &lt; &gt; ...
auto     Auto conversion.
bell     Warn when you press '&' key.
")
(defvar yasgml-do-special-characters-check t
  "*If non-nil, check special characters before typesetting.
It is useful to find errors about special characters.")

;(defvar YaTeX-japan (or (boundp 'NEMACS) (boundp 'MULE) YaTeX-emacs-20)
;  "Whether yatex mode is running on Japanese environment or not.")

(defvar yasgml-use-linuxdoc-sgml-ja (if (file-exists-p "/usr/bin/sgmls") t nil)
  "Flag to decide which sgml package is used.
If you want using linuxdoc-sgml-ja package, set this variable non-nil value.
linuxdoc-sgml-ja package is distributed from JF before sgmltools-1.09j.")

(defvar yasgml-typeset-program
  (if yasgml-use-linuxdoc-sgml-ja 
     "sgml2txt -f" "sgml2txt -f -l ja -c nippon")
  "*Typesetting program for SGML.")
(defvar yasgml-format-program-alist
  (if yasgml-use-linuxdoc-sgml-ja 
      '((text . "sgml2txt")
	(man . "sgml2txt --man")
	(latex . "sgml2latex")
	(ps . "sgml2latex --output=ps")
	(dvi . "sgml2latex --output=dvi")
	(html . "sgml2html")
	(info . "sgml2info")
	(lyx . "sgml2lyx")
	(rtf . "sgml2rtf")
	(xml . "sgml2xml")
	(line . "sgmlcheck"))
    '((text . "sgml2txt -l ja -c nippon")
      (man . "sgml2txt --man")
      (latex . "sgml2latex")
      (ps . "sgml2latex --output=ps")
      (dvi . "sgml2latex --output=dvi")
      (html . "sgml2html -l ja -c nippon")
      (info . "sgml2info")
      (lyx . "sgml2lyx")
      (rtf . "sgml2rtf")
      (xml . "sgml2xml")
      (line . "sgmlcheck")))
  "*Program alist to convert SGML to various formats.")
(defvar yasgml-preview-function 'yasgml-show-txt-other-window
  "*Preview function.")

(defvar yasgml-lint-program "sgmlcheck"
  "*Program name to lint SGML file")

(defvar yasgml-error-line-regexp
  "^sgmls: SGML error at \\(.*\\), line +\\([0-9]+\\) "
  "*Regexp of error position which is produced by lint program.")

(defvar yasgml-use-font-lock (featurep 'font-lock)
  "*Use font-lock to fontify buffer or not.")

(defvar yasgml-fontify-preview-buffer t
  "*Fontify preview buffer or not.")

(defvar yasgml-overstrike-face 'bold
  "*Face to use when fontifying overstrike.")

(defvar yasgml-underline-face 'underline
  "*Face to use when fontifying underlining.")

;;; --- customizable variable ends here ---
(defvar yasgml-kanji-code 3
  "Kanji coding system number of sgml file; 1=sjis, 2=jis, 3=euc")
(defvar yasgml-coding-system
  (cdr (assq yasgml-kanji-code YaTeX-kanji-code-alist))
  "Kanji coding system")
(defvar yasgml-shell-command-option  (or (and (boundp 'shell-command-option) shell-command-option)
      (if (eq system-type 'ms-dos) "/c" "-c")))

;;; -------- tags ----------

(defvar yasgml-env-table
  '(("abstract") ("code") ("descrip") ("enum") ("itemize") ("quote") ("tscreen") ("verb")))
(defvar yasgml-typeface-table
  (append
   '(("tag") ("bf") ("cparam") ("em") ("it") ("sf") ("sl") ("tt") )
   yasgml-env-table)
  "Default completion table of typeface designator")
(defvar yasgml-single-cmd-table
  '(("item") ("p") ("toc")
    ("idx") ("cdx") ("nidx") ("ncdx") ("file"))
  "Default completion table of SGML single command.")
(defvar yasgml-sect-table
  '(("sect") ("sect1") ("sect2") ("sect3") ("sect4"))
  "Default completion table of sect commands.")
(defvar yasgml-form-table
  (append
   '(("url") ("htmlurl") ("label") ("ref"))
   yasgml-sect-table)
  "Default completion table of SGML form.")
(defvar yasgml-special-characters-alist
  '(("<" . "&lt;") (">" . "&gt;") ("$" . "&dollar;")
    ("#" . "&num;") ("%" . "&percnt;") ("\"" . "&dquot;") ("~" . "&tilde;"))
  "Default special characters.")
(defvar yasgml-user-env-table nil)
(defvar yasgml-tmp-env-table nil)

(defvar yasgml-user-typeface-table nil)
(defvar yasgml-tmp-typeface-table nil)
(defvar yasgml-last-typeface-cmd "em")

(defvar yasgml-user-single-cmd-table nil)
(defvar yasgml-tmp-single-cmd-table nil)

(defvar yahtml-user-form-table nil)
(defvar yahtml-tmp-form-table nil)
(defvar yahtml-last-form "sect")


(defvar yasgml-struct-name-regexp
  (concat
   "\\<\\("
   (mapconcat (function (lambda (x) (car x))) yasgml-typeface-table "\\|")
   "\\)\\b")
  "Regexp of structure beginning.")

(defconst yasgml-font-lock-keywords-1
  '(("<\\([!?][a-z][-.a-z0-9]*\\)" 1 font-lock-keyword-face)
    ("<\\(/?[a-z][-.a-z0-9]*\\)" 1 font-lock-function-name-face)
    ("[&%][a-z][-.a-z0-9]*;?" . font-lock-variable-name-face)
    ("<! *--.*-- *>" . font-lock-comment-face)))

(defconst yasgml-font-lock-keywords-2 yasgml-font-lock-keywords-1)

(defvar yasgml-font-lock-keywords yasgml-font-lock-keywords-1
  "*Rules for highlighting HTML code.")

;;; --------- keymaps -----------

(defvar yasgml-prefix-map nil)
(defvar yasgml-mode-map nil "Keymap used in yasgml-mode.")
(defvar yasgml-preview-buffer-map nil "Keymap used in preview buffer.")
(defvar yasgml-lint-buffer-map nil "Keymap used in lint buffer.")

(defun yasgml-define-begend-key-normal (key env &optional map)
  "Define short cut yasgml-insert-begend key."
  (YaTeX-define-key
   key
   (list 'lambda '(arg) '(interactive "P")
	 (list 'yasgml-insert-begend 'arg env))
   map))

(defun yasgml-define-begend-region-key (key env &optional map)
  "Define short cut yasgml-insert-begend-region key."
  (YaTeX-define-key key (list 'lambda nil '(interactive)
			      (list 'yasgml-insert-begend t env)) map))

(defun yasgml-define-begend-key (key env &optional map)
  "Define short cut key for begin type completion both for
normal and region mode.  To customize yasgml, user should use this function."
  (yasgml-define-begend-key-normal key env map)
  (if YaTeX-inhibit-prefix-letter nil
    (yasgml-define-begend-region-key
     (concat (upcase (substring key 0 1)) (substring key 1)) env map)))


(if yasgml-mode-map nil
  (setq yasgml-mode-map (make-sparse-keymap)
	yasgml-prefix-map (make-sparse-keymap))
  (define-key yasgml-mode-map yasgml-prefix yasgml-prefix-map)
  (define-key yasgml-mode-map "\M-f" 'yasgml-forward-word)
  (define-key yasgml-mode-map "\M-\C-@" 'yahtml-mark-begend)
  (if (and (boundp 'window-system) (eq window-system 'x) YaTeX-emacs-19)
      (define-key yasgml-mode-map [?\M-\C- ] 'yahtml-mark-begend))
  (define-key yasgml-mode-map "\M-\C-a" 'yasgml-beginning-of-environment)
  (define-key yasgml-mode-map "\M-\C-e" 'yasgml-end-of-environment)
  (define-key yasgml-mode-map "\M-\C-m" 'newline-and-indent)
  ;(define-key yasgml-mode-map "\C-i" 'yahtml-indent-line)
  ;; special characters replace
  (mapcar (lambda (x)
            (define-key yasgml-mode-map (car x)
              `(lambda ()
                 (interactive)
                 (insert (if (yasgml-in-verb-p) ,(car x) ,(cdr x))))))
          yasgml-special-characters-alist)
  (define-key yasgml-mode-map "&" 'yasgml-ampersand)
  (define-key yasgml-mode-map "/" 'yasgml-insert-etago)
  (let ((map yasgml-prefix-map))
    (YaTeX-define-key "^" 'yahtml-visit-main map)
    (YaTeX-define-key "4^" 'yahtml-visit-main-other-window map)
    (YaTeX-define-key "4g" 'yahtml-goto-corresponding-*-other-window map)
    (YaTeX-define-key "44" 'YaTeX-switch-to-window map)
    (and YaTeX-emacs-19 window-system
	 (progn
	   (YaTeX-define-key "5^" 'yahtml-visit-main-other-frame map)
	   (YaTeX-define-key "5g" 'yahtml-goto-corresponding-*-other-frame map)
	   (YaTeX-define-key "55" 'YaTeX-switch-to-window map)))
    (YaTeX-define-key "v" 'YaTeX-version map)
    (YaTeX-define-key "}" 'YaTeX-insert-braces-region map)
    (YaTeX-define-key "]" 'YaTeX-insert-brackets-region map)
    (YaTeX-define-key ")" 'YaTeX-insert-parens-region map)
    (YaTeX-define-key "s" 'yasgml-insert-form map)
    (YaTeX-define-key "l" 'yahtml-insert-tag map)
    (YaTeX-define-key "L" 'yahtml-insert-tag-region map)
    (YaTeX-define-key "m" 'yahtml-insert-single map)
    (YaTeX-define-key "p" 'yasgml-insert-p map)
    (if YaTeX-no-begend-shortcut
	(progn
	  (YaTeX-define-key "B" 'yasgml-insert-begend-region map)
	  (YaTeX-define-key "b" 'yasgml-insert-begend map))
      (yasgml-define-begend-key "ba" "abstract" map)
      (yasgml-define-begend-key "bc" "code" map)
      (yasgml-define-begend-key "bd" "descrip" map)
      (yasgml-define-begend-key "be" "enum" map)
      (yasgml-define-begend-key "bi" "itemize" map)
      (yasgml-define-begend-key "bq" "quote" map)
      (yasgml-define-begend-key "bt" "tscreen" map)
      (yasgml-define-begend-key "bv" "verb" map)
      (YaTeX-define-key "b " 'yasgml-insert-begend map)
      (YaTeX-define-key "B " 'yasgml-insert-begend-region map)
      )
    (YaTeX-define-key "e" 'YaTeX-end-environment map)
    (YaTeX-define-key ">" 'yahtml-comment-region map)
    (YaTeX-define-key "<" 'yahtml-uncomment-region map)
    (YaTeX-define-key "g" 'yasgml-goto-corresponding-* map)
    (YaTeX-define-key "k" 'yahtml-kill-* map)
    (YaTeX-define-key "c" 'yahtml-change-* map)
    (YaTeX-define-key "t" 'yasgml-typeset-menu map)
    (YaTeX-define-key "a" 'yasgml-complete-mark map)
    (YaTeX-define-key "'" 'yahtml-prev-error map)
    ;;;;;(YaTeX-define-key "i" 'yahtml-fill-item map)
    ))

;;; ------- paragraph -----------


(defvar yasgml-paragraph-start
  (concat
   "^$\\|<!--\\|^[ \t]*</?\\(sect\\|sect[1-4]\\|article\\|book\\|report\\|manpage\\|abstract\\|code\\|descrip\\|enum\\|itemize\\|quote\\|tscreen\\|verb\\|p\\)\\b")
  "*Regexp of sgml paragraph separater")
(defvar yasgml-paragraph-separate
  (concat
   "^$\\|<!--\\|^[ \t]*</?\\(sect\\|sect[1-4]\\|article\\|book\\|report\\|manpage\\|abstract\\|code\\|descrip\\|enum\\|itemize\\|quote\\|tscreen\\|verb\\|p\\|!--\\)\\b")
  "*Regexp of sgml paragraph separater")

;;; ------- yasgml-mode ---------

(defun yasgml-mode ()
  " Yet Another SGML mode: Major mode for editing LinuxDoc DTD SGML files.

 In the following description, [prefix] is C-c key as default.
 (you can customize this key using variable 'yasgml-prefix, see above)

  * [prefix] b X	insert environment tag such as <enum> and </enum>

  * [prefix] s	insert following tags with interactive prompt:
                        <url url=\"http://aaaa.com\" name=\"test\">
                        <sect>Section Title<p>
  * [prefix] l	insert tags for textstyle such as `<em> ... </em>' 
                      or  `<tag> ... </tag>'

  * [prefix] m	insert tags such as `<toc>', `<item> ', or `<p>'

  * [prefix] p	insert <p> at cursor position
  * [prefix] g	jump to the matching tag :
                        (<verb> <-> </verb>, or url=\"bbb.html\": 
                         in this case, the place to go is bbb.html)

  * [prefix] k	erase SGML tag at the position
			if C-u is entered as a prefix, then the contents
                      enclosed that tag are erased at once.

  * [prefix] c	change the tag at the position

  * [prefix] t j	exec sgml2txt for current file.
  * [prefix] t p	preview text file generated by SGML source
  * [prefix] a	this is the same as completion of accent in YaTeX
			you can enter &lt\;, &gt\;, etc using this.

  * Here is the tags completed by \"[prefix] b\" : short-cut key is shown in ()
                        abstract ([prefix] b a)
                        code     ([prefix] b c)
                        descrip  ([prefix] b d)
                        enum     ([prefix] b e)
                        itemize  ([prefix] b i)
                        quote    ([prefix] b q)
                        tscreen  ([prefix] b t)
                        verb     ([prefix] b v)

  * Here is the tags completed by \"[prefix] l\"
      tag, bf, cparam, em, it, sf, sl, tt, bf

  * Here is the tags completed by \"[prefix] m\"
      item, p, toc, idx, cdx, nidx, ncdx, file

  * Here is the tags completed by \"[prefix] s\"
      sect, sect1, sect2, sect3, sect4, url, htmlurl, label, ref

  Those maybe enough for fastening your writing of SGML source.
If you have more request, please tell me. Thanks.
"
  (interactive)
  (cond
   ((and YaTeX-emacs-20 (fboundp 'coding-system-equal))
    (if t (or (coding-system-equal
	       yasgml-coding-system buffer-file-coding-system)
	      (set-buffer-file-coding-system yasgml-coding-system))
      ;;^v which is better?
      (let ((bmp (buffer-modified-p)))
	(set-buffer-file-coding-system yasgml-coding-system)
	(set-buffer-modified-p bmp))))
   ((featurep 'mule)
    (set-file-coding-system yasgml-coding-system))
   ((boundp 'NEMACS)
    (make-local-variable 'kanji-fileio-code)
    (setq kanji-fileio-code yasgml-kanji-code)))
  (setq major-mode 'yasgml-mode
	mode-name "yasgml")
  (mapcar
   (function (lambda (x)
	       (make-local-variable (car x))
	       (set (car x) (if (and (symbolp (cdr x))
				     (boundp (cdr x)))
				(symbol-value (cdr x))
			      (cdr x)))))
   '((YaTeX-ec . "")
     (YaTeX-struct-begin . "<%1%2")
     (YaTeX-struct-end . "</%1>")
     (YaTeX-struct-name-regexp . yasgml-struct-name-regexp)
     (YaTeX-comment-prefix . "<!--")
     (YaTeX-coding-system . yasgml-coding-system)
     (YaTeX-typesetting-mode-map . yasgml-lint-buffer-map)
     (fill-prefix . yasgml-fill-prefix) (fill-column . yasgml-fill-column)
     (paragraph-start . yasgml-paragraph-start)
     (paragraph-separate . yasgml-paragraph-separate)
     (comment-start . "<!-- ") (comment-end . " -->")
     (comment-start-skip . comment-start)
     (indent-line-function . indent-relative-maybe)
     ;;
     (yahtml-env-table . yasgml-env-table)
     (yahtml-user-env-table . nil)
     (yahtml-tmp-env-table . nil)
     ;;
     (yahtml-typeface-table . yasgml-typeface-table)
     (yahtml-user-typeface-table . nil)
     (yahtml-tmp-typeface-table . nil)
     (yahtml-last-typeface-cmd . "em")
     ;;
     (yahtml-single-cmd-table . yasgml-single-cmd-table)
     (yahtml-user-single-cmd-table . nil)
     (yahtml-tmp-single-cmd-table . nil)
     (yahtml-last-single-cmd . "toc")
     ;;
     (yahtml-form-table . yasgml-form-table)
     (yahtml-user-form-table . nil)
     (yahtml-tmp-form-table . nil)
     (yahtml-last-form . "sect")
     ;;
     (yahtml-struct-name-regexp . yasgml-struct-name-regexp)
     (yasgml-preview-point . 1)
     (yahtml-prefer-upcases . nil)
      ))

  (if yasgml-use-font-lock
      (progn
	(make-local-variable 'font-lock-defaults)
	(setq font-lock-defaults '((yasgml-font-lock-keywords
				    yasgml-font-lock-keywords-1
				    yasgml-font-lock-keywords-2)
				   nil t))
	(turn-on-font-lock)))

  (set-syntax-table yahtml-syntax-table)
  (use-local-map yasgml-mode-map)
  (YaTeX-read-user-completion-table)
  (turn-on-auto-fill)			;Sorry, this is prerequisite
  (and (= 0 (buffer-size)) (call-interactively 'yasgml-make-document))
  (run-hooks 'text-mode-hook 'yasgml-mode-hook))

;;; ------ menu-bar ------
(defvar yasgml-menu-map nil "Menu map of yasgml")
(defvar yasgml-menu-map-sectioning nil "Menu map of yasgml(sectioning)")
(defvar yasgml-menu-map-environment nil "Menu map of yasgml(environment)")
(defvar yasgml-menu-map-listing nil "Menu map of yasgml(listing)")
(defvar yasgml-menu-map-typeface nil "Menu map of yasgml(typeface tags)")
(defvar yasgml-menu-map-form nil "Menu map of yasgml(URL, label, ref)")
(defvar yasgml-menu-map-item nil "Menu map of yasgml(single tags)")

(cond
 ((and YaTeX-emacs-19 (null yasgml-menu-map))
  (setq yasgml-menu-map (make-sparse-keymap "yasgml"))
  (setq yasgml-menu-map-sectioning (make-sparse-keymap "sectioning menu"))
  (YaTeX-define-menu
   'yasgml-menu-map-sectioning
   (nreverse
    '((1 "sect"  . (lambda () (interactive) (yasgml-insert-sect "sect")))
      (2 "sect1" . (lambda () (interactive) (yasgml-insert-sect "sect1")))
      (3 "sect2" . (lambda () (interactive) (yasgml-insert-sect "sect2")))
      (4 "sect3" . (lambda () (interactive) (yasgml-insert-sect "sect3")))
      (5 "sect4" . (lambda () (interactive) (yasgml-insert-sect "sect4")))
      )))

  (setq yasgml-menu-map-typeface (make-sparse-keymap "typeface tags"))
  (YaTeX-define-menu
   'yasgml-menu-map-typeface
   (nreverse
    '((bf	"Bold" .
	  (lambda () (interactive) (yasgml-insert-tag nil "bf")))
      (tt	"Typewriter" .
	  (lambda () (interactive) (yasgml-insert-tag nil "tt")))
      (it	"Italic" .
	  (lambda () (interactive) (yasgml-insert-tag nil "it")))
      ;; Other fonts should not be used.
      )))
  (setq yasgml-menu-map-environment (make-sparse-keymap "environment"))
  (YaTeX-define-menu
   'yasgml-menu-map-environment
   (nreverse
    '((abstract	"Abstract" .
		(lambda () (interactive) (yasgml-insert-begend nil "abstract")))
      (code	"Code" .
		(lambda () (interactive) (yasgml-insert-begend nil "code")))
      (quote	"Quote" .
		(lambda () (interactive) (yasgml-insert-begend nil "quote")))
      (tscreen	"Typewriter screen" .
		(lambda () (interactive) (yasgml-insert-begend nil "tscreen")))
      (verb	"Verbatim" .
		(lambda () (interactive) (yasgml-insert-begend nil "verb")))
      )))
  (setq yasgml-menu-map-listing (make-sparse-keymap "listing"))
  (YaTeX-define-menu
   'yasgml-menu-map-listing
   (nreverse
    '((descrip	"Description" .
		(lambda () (interactive) (yasgml-insert-begend nil "descrip")))
      (enum	"Enumerate" .
		(lambda () (interactive) (yasgml-insert-begend nil "enum")))
      (itemize	"Itemize" .
		(lambda () (interactive) (yasgml-insert-begend nil "itemize")))
      )))
  (setq yasgml-menu-map-form (make-sparse-keymap "label & ref"))
  (YaTeX-define-menu
   'yasgml-menu-map-form
   (nreverse
    '((url	"Url" .
		(lambda () (interactive) (yasgml-insert-form "url")))
      (htmlurl	"HTMLurl" .
		(lambda () (interactive) (yasgml-insert-form "htmlurl")))
      (label	"Label" .
		(lambda () (interactive) (yasgml-insert-form "label")))
      (ref	"Ref" .
		(lambda () (interactive) (yasgml-insert-form "ref")))
      )))
  (setq yasgml-menu-map-item (make-sparse-keymap "item"))
  (YaTeX-define-menu
   'yasgml-menu-map-item
   (nreverse
    '((item	"item" .
		(lambda () (interactive) (yasgml-insert-single "item")))
      (p	"P)aragraph" .
		(lambda () (interactive) (yasgml-insert-p)))
      (toc	"Table of Contents" .
		(lambda () (interactive) (yasgml-insert-single "toc")))
      ;;These tags are ignored by all backends except LaTeX.
      ;;But LaTeX-backend is not support Japanese yet.
;      (idx	"index" .
;		(lambda () (interactive) (yasgml-insert-single "idx")))
;      (cdx	"index(constant-width font)" .
;		(lambda () (interactive) (yasgml-insert-single "cdx")))
;      (nidx	"index(no appearance)" .
;		(lambda () (interactive) (yasgml-insert-single "nidx")))
;      (ncdx	"index(constant-width font, no appearance)" .
;		(lambda () (interactive) (yasgml-insert-single "ncdx")))
;      (file	"filename" .
;		(lambda () (interactive) (yasgml-insert-single "file")))
      )))
  (define-key yasgml-mode-map [menu-bar yasgml]
    (cons "yasgml" yasgml-menu-map))
  (let ((keys (where-is-internal 'fill-paragraph global-map)))
    (while keys
      (define-key yasgml-mode-map (car keys) 'yahtml-fill-paragraph)
      (setq keys (cdr keys))))
  (YaTeX-define-menu
   'yasgml-menu-map
   (nreverse
    (list
     (cons (list 'sect "Sectioning")
	   (cons "sectioning" yasgml-menu-map-sectioning))
     (cons (list 'env  "Environment")
           (cons "environment" yasgml-menu-map-environment))
     (cons (list 'list "Listing")
	   (cons "Listing" yasgml-menu-map-listing))
     (cons (list 'item "Item")
	   (cons "Itemizing" yasgml-menu-map-item));;;
     (cons (list 'type "Typeface tags")
	   (cons "typeface" yasgml-menu-map-typeface))
     (cons (list 'form "URL, label, ref")
           (cons "url, label, ref" yasgml-menu-map-form))
     )))
  (if (featurep 'xemacs)
      (add-hook 'yasgml-mode-hook
		'(lambda ()
		   (or (assoc "yasgml" current-menubar)
		       (progn
			 (set-buffer-menubar (copy-sequence current-menubar))
			 (add-submenu nil yasgml-menu-map))))))
  ))


;;; ------ insertion -------
(defalias 'yasgml-insert-single 'yahtml-insert-single)
(defalias 'yasgml-insert-tag 'yahtml-insert-tag)

(defun yasgml-make-document (style title author date)
  "Set up a SGML document."
  (interactive (let* ((result (progn (message "A)rticle B)ook R)eport M)anpage (default: %s)" yasgml-default-style)
				     (read-char)))
                      (title  (read-string "Title: "))
                      (author (read-string "Author: " yasgml-author))
                      (date   (read-string "Date: " (current-time-string))))
		 (list (downcase result) title author date)))
  (let ((style-string
	 (cond
	  ((eq style ?a)"article")
	  ((eq style ?b)"book")
	  ((eq style ?r)"report")
	  ((eq style ?m)"manpage")
	  (t yasgml-default-style))))
    (insert "<!doctype " yasgml-default-dtd " system>\n\n")
    (insert "<" style-string ">\n\n<title>" title
            "\n<author>" author "\n<date>" date "\n\n")
    (save-excursion (insert "\n</" style-string ">\n"))))

(defun yasgml-in-verb-p ()
  "Return t if the (point) is between <verb> and </verb>."
  (save-excursion
    (if (re-search-backward "<\\(/?\\)\\(verb\\|code\\|quote\\)>" nil t)
        (cond ((string= (YaTeX-match-string 1) "") t)
              ((string= (YaTeX-match-string 1) "/") nil))
      nil)))

(defun yasgml-ampersand ()
  "When you press '&' key, execute this function."
  (interactive)
  (cond ((eq yasgml-special-characters-input-method 'auto)
         (insert "&amp;"))
        ((eq yasgml-special-characters-input-method 'bell)
         (insert "&")
         (ding)
         (message (substitute-command-keys
                   "Warning: Bare '&' is an error in SGML. Use '\\[yasgml-complete-mark]' to input special-char.")))
        ))

(defun yasgml-complete-mark ()
  "Complete &amp, &lt, &gt, &dollar, &num, &percnt, &dquot, &tilde, &etago."
  (interactive)
  (message "%s" "a:& s:</ d:< f:> g:$ h:\# j:% k:\" l:~")
  (let ((c (downcase(read-char))) s del)
    (if (eq (char-before (point)) ?&) (progn
                                        (delete-backward-char 1)
                                        (setq del t)))
    (setq c (cond ((eq c ?a) ?&)
                  ((eq c ?s) ?/)
                  ((eq c ?d) ?<)
                  ((eq c ?f) ?>)
                  ((eq c ?g) ?$)
                  ((eq c ?h) ?#)
                  ((eq c ?j) ?%)
                  ((eq c ?k) ?\")
                  ((eq c ?l) ?~)
                  (t c)))
    (cond ((eq c ?/) (insert "&etago;"))
          ((eq c ?&) (insert "&amp;"))
          (t (if (setq s (cdr (assoc (char-to-string c)
                                     yasgml-special-characters-alist)))
                 (insert s)
               (if del (insert "&")))))))




(defun yasgml-insert-etago ()
  "Intelligent insertion of &etago;. The binding of this command is \"/\".
If previous char is \"<\", replace it with \"&etago;\".
Otherwise insert \"/\"."
  (interactive)
  (cond ((eq (char-before) ?<)
         (delete-backward-char 1) (insert "&etago;"))
        (t
         (insert "/"))))

(defun yasgml-insert-p ()
  "Insert <p>."
  (interactive)
  (yahtml-insert-single "p") (newline))

(defvar yasgml-last-begend "verb")
(defun yasgml-insert-begend (&optional region env)
  "Insert <cmd> ... </cmd>."
  (interactive "P")
  (let*((completion-ignore-case t)
	(cmd
	 (or env
	     (YaTeX-cplread-with-learning
	      (format "Environment(default %s): " yasgml-last-begend)
	      'yahtml-env-table 'yahtml-user-env-table 'yahtml-tmp-env-table)))
	(bolp (save-excursion
		(skip-chars-backward " \t" (point-beginning-of-line)) (bolp)))
	(cc (current-column)))
    (if (string< "" cmd) (setq yasgml-last-begend cmd))
    (setq yahtml-last-begend
	  (or (cdr (assoc yasgml-last-begend yahtml-env-table))
	      yasgml-last-begend))
    (setq cmd yasgml-last-begend)
    (if yahtml-prefer-upcases (setq cmd (upcase cmd)))
    (if region
	(let ((beg (region-beginning))
	      (end (region-end))
	      (addin (yasgml-addin cmd)))
	  (goto-char end)
	  (insert (format "</%s>%s" cmd (if bolp "\n" "")))
	  (goto-char beg)
	  (insert (format "<%s%s>%s" cmd addin (if bolp "\n" ""))))
      (insert (format "<%s%s>" cmd (yasgml-addin cmd)))
      (save-excursion
	(insert "\n")
	(indent-to-column cc)
	(insert (format "</%s>" cmd)))
      (yasgml-intelligent-newline nil)
      )))


(defun yasgml-insert-form (&optional form)
  "Insert <FORM option=\"argument\">."
   (interactive)
   (or form
       (let ((completion-ignore-case t))
	 (setq form
	       (YaTeX-cplread-with-learning
		(format "Form(default %s): " yahtml-last-form)
		'yahtml-form-table 'yahtml-user-form-table
		'yahtml-tmp-form-table))))
   (let ((p (point)) q)
     (if (string= form "") (setq form yahtml-last-form))
     (setq yahtml-last-form form)
     (if (assoc form yasgml-sect-table)
         (yasgml-insert-sect form)
       (insert (format "<%s%s>" form (yasgml-addin form))))
     ;;(indent-relative-maybe)
     (if (cdr (assoc form yahtml-form-table))
	 (save-excursion (insert (format "</%s>" form))))
     (if (search-backward "\"\"" p t) (forward-char 1))))

(defun yasgml-insert-sect (cmd)
  "Insert <sect> .. <sect4>."
  (interactive
   (list (completing-read "<sect>, <sect1> .. <sect4>: " yasgml-sect-table nil t "sect")))
  (insert "<" cmd ">" (read-string "title: ") "\n<p>\n"))


;;; ---------- Add-in ----------
(defun yasgml-addin (form)
  "Check add-in function's existence and call it if exists."
   (let ((addin (concat "yasgml:" (downcase form))) s)
     (if (and (intern-soft addin) (fboundp (intern-soft addin))
	      (stringp (setq s (funcall (intern addin))))
	      (string< "" s))
	 (if (eq (aref s 0) ? ) s (concat " " s))
       "")))

(defun yasgml:url ()
  "Add-in function for url."
  (let ((href ""))
    (setq yahtml-completing-buffer (current-buffer)
	  href (read-from-minibuffer "url: " "" yahtml-url-completion-map)
	  ;; yahtml-urls-local is buffer-local, so we must put
	  ;; that into yahtml-urls here
	  yahtml-urls (append yahtml-urls-private yahtml-urls-local))
    (prog1
	(concat (yahtml-make-optional-argument
		 "url" href)
		(yahtml-make-optional-argument
		 "name" (read-string "name: ")))
      (if (and (string-match "^http://" href)
	       (null (assoc href yahtml-urls)))
	  (YaTeX-update-table
	   (list href)
	   'yahtml-urls-private 'yahtml-urls-private 'yahtml-urls-local))
      )))

(fset 'yasgml:htmlurl 'yasgml:url)

(defun yasgml:label ()
  "Add-in function for label."
  (let ()
    (concat "id=\"" (read-string "Label-id: " nil) "\"")))

(defun yasgml:ref ()
  "Add-in function for ref."
  (let ()
    (concat "id=\"" (read-string "Ref-id: " nil) "\" "
            "name=\"" (read-string "Name: " nil) "\"")))


;;; ---------- Jump ----------

(defun yasgml-forward-word (count)
  "Forward-word for yasgml."
  (interactive "p")
  (if (zerop count) t
    (if (eq (char-after (point)) ?<)
        (search-forward ">" nil t)
      (forward-word 1))
    (yasgml-forward-word (1- count))))

(defun yasgml-on-url-p ()
  ""
  (let ((p (point)) e cmd)
    (save-excursion
      (and (goto-char (1+ (search-backward "<" nil t)))
           (looking-at "\\(htmlurl\\|url\\)")
           (save-excursion
             (search-forward ">" nil t)
             (setq e (point)))
           (search-forward "url" e t)
           (search-forward "=" e t)
	   (progn
	     (skip-chars-forward " \t\n")
	     (looking-at "\"?\\([^\"> \t\n]+\\)\"?"))
	   (< p (match-end 0))
	   (YaTeX-match-string 1)
	   ))))

(defun yasgml-goto-corresponding-url (&optional other)
  "Go to corresponding name."
  (let ((href (yasgml-on-url-p)) file name (parent buffer-file-name))
    (if href
	(cond
	 ((string-match "^\\(ht\\|f\\)tp:" href)
	  (yahtml-browse-html href))
	 (t (setq file (substring href 0 (string-match "#" href)))
	    (if (string-match "#" href)
		(setq name (substring href (1+ (string-match "#" href)))))
	    (if (string< "" file)
		(progn
		  (if (string-match "/$" file)
		      (setq file (concat file yahtml-directory-index)))
		  (if (string-match "^/" file)
		      (setq file (yahtml-url-to-path file)))
		  (if other (YaTeX-switch-to-buffer-other-window file)
		    (YaTeX-switch-to-buffer file))
		  (or YaTeX-parent-file (setq YaTeX-parent-file parent))))
	    (if name
		(progn (set-mark-command nil) (yasgml-jump-to-name name)))
	    t)))))

(defun yasgml-jump-to-name (name)
  "Jump to html's named tag."
  (setq name (format "<label[^>]+name\\s *=\\s *\"?%s\"?" name))
  (or (and (re-search-forward name nil t) (goto-char (match-beginning 0)))
      (and (re-search-backward name nil t) (goto-char (match-beginning 0)))
      (message "Named tag `%s' not found" (substring href 1))))


(defun yasgml-goto-corresponding-* (&optional other)
  "Go to corresponding object."
  (interactive)
  (let ((major-mode 'yahtml-mode))
  (cond
   ((yasgml-goto-corresponding-url other))
   ;((yasgml-goto-corresponding-label))
   ((yahtml-goto-corresponding-begend))
   (t (message "I don't know where to go."))
   )))

(defun yasgml-beginning-of-environment (&optional limit-search-bound end)
  "YaTeX-beginning-of-environment for yasgml-mode."
  (interactive)
  (let ((major-mode 'yahtml-mode))
    (YaTeX-beginning-of-environment limit-search-bound end)))

(defun yasgml-end-of-environment (&optional limit-search-bound)
  "YaTeX-end-of-environment for yasgml-mode."
  (interactive)
  (yasgml-beginning-of-environment limit-search-bound t))

;;;
;;; ---------- Typesetting and Previewing ----------
;;;
(defun yasgml-typeset-menu ()
  "Browsing menu"
  (interactive)
  (message "J)typeset P)review T)ext H)tml I)nfo L)atex D)vi y)lYx R)tf X)ml C)heck...")
  (let ((c (char-to-string (read-char))))
    (cond
     ((string-match "j" c)
      (yasgml-typeset-this-buffer))
     ((string-match "p" c)
      (yasgml-preview-this-buffer))
     ((string-match "t" c)
      (yasgml-format-this-buffer 'text))
     ((string-match "h" c)
      (yasgml-format-this-buffer 'html))
     ((string-match "i" c)
      (yasgml-format-this-buffer 'info))
     ((string-match "l" c)
      (yasgml-format-this-buffer 'latex))
     ((string-match "d" c)
      (yasgml-format-this-buffer 'dvi))
     ((string-match "y" c)
      (yasgml-format-this-buffer 'lyx))
     ((string-match "r" c)
      (yasgml-format-this-buffer 'rtf))
     ((string-match "x" c)
      (yasgml-format-this-buffer 'xml))
     ((string-match "c" c)
      (yasgml-lint-this-buffer))
     )))

(defvar yasgml-typeset-buffer "*yasgml-typesetting*")

(defun yasgml-typeset-this-buffer ()
  "Validate this buffer"
  (interactive)
  (if yasgml-do-special-characters-check
      (yasgml-special-characters-check))
  (require 'yatexprc)
  (YaTeX-save-buffers)
  (YaTeX-typeset
   (concat yasgml-typeset-program " "
	   (file-name-nondirectory (buffer-file-name (current-buffer))))
   yasgml-typeset-buffer  "typeset" "typeset")
  )

(defun yasgml-special-characters-check ()
  "Check for special characters. "
  (interactive)
  (let ((pt (point))
        amp colon)
    (goto-char (point-min))
    (while (prog1
             (setq amp (search-forward "&" nil t))
             (setq colon (save-excursion
                           (re-search-forward "[&;]" nil t)
                           (if (eq (char-before (point)) ?&) nil (point)))))
      (if (or (null colon)
              (> (-  colon amp) 7 ))
          (if (progn
                (momentary-string-display "=>" (1- (point)) 0
                                          "This '&' should be '&amp;', FIX? ")
                (y-or-n-p "This '&' should be '&amp;', FIX? "))
              (insert "amp;")
            (ding)
            (message "WARNING: THIS FILE MAY BE ERROR.")
            (sit-for 2))))
    (goto-char pt)
    (message "")
    (sit-for 0)))

(defun yasgml-format-this-buffer (to)
  "Make various formats from SGML file."
  (let ((cmd (cdr (assq to yasgml-format-program-alist)))
        )
    (if cmd
        (YaTeX-typeset
         (concat cmd " "
                 (file-name-nondirectory (buffer-file-name (current-buffer))))
         "*yasgml-formatting*"  "format" "format")
      (error "Command for %s is not specified" (symbol-name to)))))

(defun yasgml-preview-this-buffer ()
  "Preview. Executing function specified in yasgml-preview-function."
  (interactive)
  (funcall yasgml-preview-function))

(defun yasgml-get-filename (&optional ext)
  "Get current-file's basename if EXT is nil, otherwise (concat basename EXT)."
  (let* ((fname (file-name-nondirectory (buffer-file-name)))
         (period (rindex fname ?.)))
    (concat (substring fname 0 period) (if ext ext ""))
    ))

(defvar yasgml-preview-point 1)

(defvar yasgml-preview-window-configuration nil)

(if yasgml-preview-buffer-map nil
  (setq yasgml-preview-buffer-map (make-keymap))
  (define-key yasgml-preview-buffer-map "q" 'yasgml-close-preview-window)
  (define-key yasgml-preview-buffer-map "h" 'forward-char)
  (define-key yasgml-preview-buffer-map "j" 'next-line)
  (define-key yasgml-preview-buffer-map "k" 'previous-line)
  (define-key yasgml-preview-buffer-map "l" 'backward-char)
  (define-key yasgml-preview-buffer-map "^" 'beginning-of-line)
  (define-key yasgml-preview-buffer-map "$" 'end-of-line)
  (define-key yasgml-preview-buffer-map "/" 'isearch-forward)
  (define-key yasgml-preview-buffer-map "?" 'isearch-backward)
  (define-key yasgml-preview-buffer-map "b" 'scroll-down)
  (define-key yasgml-preview-buffer-map " " 'scroll-up)
  (define-key yasgml-preview-buffer-map "o" 'other-window)
  (define-key yasgml-preview-buffer-map [backspace] 'scroll-down)
  (define-key yasgml-preview-buffer-map [delete] 'scroll-down)
  (define-key yasgml-preview-buffer-map "\C-?" 'scroll-down)
  (define-key yasgml-preview-buffer-map "\C-m" 'next-line)
  (define-key yasgml-preview-buffer-map ">" 'end-of-buffer)
  (define-key yasgml-preview-buffer-map "<" 'beginning-of-buffer)
  )

(defun yasgml-show-txt (display-function)
  "Show txt version (for preview)."
  (let* ((dir default-directory)
         (txtfile (yasgml-get-filename ".txt"))
         (buf (get-buffer "*yasgml-preview*")))
    (if buf
        (save-excursion
          (set-buffer buf)
          (setq yasgml-preview-point (point)))
      (setq buf (get-buffer-create "*yasgml-preview*")
            yasgml-preview-point 1))
    (save-excursion
      (set-buffer buf)
      (use-local-map yasgml-preview-buffer-map)
      (setq buffer-read-only nil)
      (erase-buffer)
      (cd dir)
      (insert-file txtfile)
      (if yasgml-fontify-preview-buffer
	  (yasgml-fontify-this-buffer))
      (set-buffer-modified-p nil)
      (setq buffer-read-only t)
      )
    (setq yasgml-preview-window-configuration (current-window-configuration))
    (funcall display-function buf)
    (goto-char yasgml-preview-point)
    ))

(defun yasgml-fontify-this-buffer ()
  "Convert overstriking and underlining to the correct fonts."
  (goto-char (point-min))
  (while (search-forward "\e[1m" nil t)
    (delete-backward-char 4)
    (put-text-property (point)
		       (progn (if (search-forward "\e[0m" nil 'move)
				  (delete-backward-char 4))
			      (point))
		       'face yasgml-overstrike-face))
  (goto-char (point-min))
  (while (search-forward "_\b" nil t)
    (backward-delete-char 2)
    (put-text-property (point) (1+ (point)) 'face yasgml-underline-face))
  (goto-char (point-min))
  (while (search-forward "\b_" nil t)
    (backward-delete-char 2)
    (put-text-property (1- (point)) (point) 'face yasgml-underline-face))
  (goto-char (point-min))
  (while (re-search-forward "\\(.\\)\\(\b\\1\\)+" nil t)
    (replace-match "\\1")
    (put-text-property (1- (point)) (point) 'face yasgml-overstrike-face))
  (goto-char (point-min))
  (while (re-search-forward "\\(..?.?\\)\\(\b\b\\1\\)+" nil t)
    (replace-match "\\1")
    (put-text-property (1- (point)) (point) 'face yasgml-overstrike-face))
  (goto-char (point-min))
  (while (re-search-forward "o\b\\+\\|\\+\bo" nil t)
    (replace-match "o")
    (put-text-property (1- (point)) (point) 'face 'bold))
  (goto-char (point-min))
  (while (re-search-forward "[-|]\\(\b[-|]\\)+" nil t)
    (replace-match "+")
    (put-text-property (1- (point)) (point) 'face 'bold))
  ;; \255 is some kind of dash in Latin-1.
  (goto-char (point-min))
  (if enable-multibyte-characters
      (while (search-forward "\255" nil t)
	(if (= (preceding-char) ?\255)
	    (replace-match "-")))
    (while (search-forward "\255" nil t) (replace-match "-"))))

(defun yasgml-show-txt-other-window ()
  "Show txt version other window (for preview)."
  (interactive)
  (yasgml-show-txt 'switch-to-buffer-other-window))

(defun yasgml-show-txt-other-frame ()
  "Show txt version other frame (for preview)."
  (interactive)
  (yasgml-show-txt 'switch-to-buffer-other-frame))

(defun yasgml-show-txt-bully ()
  "Show txt version in one window (for preview)."
  (interactive)
  (delete-other-windows)
  (yasgml-show-txt 'switch-to-buffer))

(defun yasgml-close-preview-window ()
  (interactive)
  (let ((buf (get-buffer "*yasgml-preview*")))
    (if (bufferp buf) (bury-buffer buf)))
  (set-window-configuration yasgml-preview-window-configuration))

;;; ------- Lint -------
(if yasgml-lint-buffer-map nil
  (setq yasgml-lint-buffer-map (make-keymap))
  (define-key yasgml-lint-buffer-map " " 'yasgml-jump-to-error-line))

(defvar yasgml-lint-buffer "*sgmlcheck*")

(defun yasgml-lint-this-buffer ()
  "Call lint on buffer BUF."
  (require 'yatexprc)
  (let ((buf (current-buffer)))
    (YaTeX-save-buffers)
    (YaTeX-typeset (concat yasgml-lint-program " "
			   (file-name-nondirectory (buffer-file-name buf)))
		   yasgml-lint-buffer  "lint" "lint")))

(defun yasgml-prev-error ()
  "Jump to previous error seeing lint buffer."
  (interactive)
  (or (get-buffer yasgml-lint-buffer)
      (error "No lint program ran."))
  (YaTeX-showup-buffer yasgml-lint-buffer nil t)
  (yasgml-jump-to-error-line t))

(defun yasgml-jump-to-error-line (&optional sit)
  (interactive "P")
  (let ((p (point)) (e (point-end-of-line)))
    (end-of-line)
    (if (re-search-backward yasgml-error-line-regexp nil t)
	(let ((f (YaTeX-match-string 1))
	      (l (string-to-int (YaTeX-match-string 2))))
	  (if sit (sit-for 1))
	  (forward-line -1)
	  (YaTeX-showup-buffer (YaTeX-switch-to-buffer f t) nil t)
	  (goto-line l))
      (message "No line number usage"))))

;;; ------- Intelligent newline -------
(defun yasgml-inner-environment ()
  (let ((major-mode 'yahtml-mode))
    (YaTeX-inner-environment)))

(defun yasgml-intelligent-newline (&optional arg)
  "Intelligent newline for SGML"
  (interactive "P")
  (let (env func)
    (end-of-line)
    (setq env (downcase (yasgml-inner-environment)))
    (setq func (intern-soft (concat "yasgml-intelligent-newline-" env)))
    (newline)
    (if (and env func (fboundp func))
	;; if intelligent line function is defined, call that
	(funcall func)
      ;; else do the default action
      )))
(defun yasgml-intelligent-newline-descrip ()
  (interactive)
  (yasgml-insert-tag nil "tag")
  (setq yahtml-last-typeface-cmd "tag"))

(defun yasgml-intelligent-newline-enum ()
  (interactive)
  (yasgml-insert-single "item")
  (setq yahtml-last-single-cmd "item"))
(defalias 'yasgml-intelligent-newline-itemize 'yasgml-intelligent-newline-enum)

;;; ------- for debug ---------
(defun yasgml-reload ()
  (interactive)
  (setq yasgml-mode-map nil)
  (load "yasgml")
  (yasgml-mode))

;;; ------- for outline-minor ---------
(make-local-variable 'outline-level)
(make-local-variable 'outline-regexp)

(defun yasgml-outline-level ()
  (save-excursion
    (looking-at outline-regexp)
    (let ((title (buffer-substring (match-beginning 1) (match-end 1))))
      (cond ((equal title "sect") 1)
            ((equal title "sect1") 2)
            ((equal title "sect2") 3)
            ((equal title "sect3") 4)
            ((equal title "sect4") 5)
           ))))

;;; yasgml.el ends here
