/**
 * @file  WorkerThreadPool.cpp
 * @brief [J[Xbhv[NX.
 *
 * @author JIN
 *
 * Copyright (C) 2009- JIN All rights reserved.
 */
#include "stdafx.h"
#include "MathUtility.h"
using namespace Math;
#include "WorkerThreadPool.h"
#include "CpuMap.h"

namespace GenericUtility {

/////////////////////////////////////////////////////////////////////////////
// CWorkerThread

CWorkerThread::CWorkerThread()
	: m_Status(S_Waiting)
{
}

CWorkerThread::~CWorkerThread()
{
	// Xbh~
	Stop();
}

bool CWorkerThread::Initialize(const OnCompleteJobFunc& onCompleteJob)
{
	// WuR[obN
	m_OnCompleteJob = onCompleteJob;
	// Xe[^XύXCxgZbg
	VERIFY(m_evStatus.Reset());

	// Xbh̏ƋNs
	m_ThreadRunner.reset(new ThreadRunner);
	if (!m_ThreadRunner) {
		return false;
	}
	m_ThreadRunner->SetThread(shared_from_this());
	return m_Thread.Create(m_ThreadRunner);
}

CWorkerThread::Status CWorkerThread::GetStatus() const
{
	// Xe[^X擾
	Status status;
	{
		CAutoCriticalSection acs(m_cs);
		status = m_Status;
	}
	return status;
}

void CWorkerThread::SetCompleteStatus_()
{
	CAutoCriticalSection acs(m_cs);

	m_Status = S_Complete;
	m_evStatus.Reset();
}

bool CWorkerThread::IsReady() const
{
	CAutoCriticalSection acs(m_cs);

	// ҋ@邢͊
	return m_Status == S_Waiting || m_Status == S_Complete;
}

bool CWorkerThread::StartJob(const IJobPtr& job)
{
	CAutoCriticalSection acs(m_cs);

	// ҋ@邢͊̂
	ASSERT(IsReady());
	if (!IsReady()) {
		return false;
	}

	// Wu
	m_job = job;

	// Xe[^XύXăCxgZbg
	m_Status = S_Starting;
	m_evStatus.Set();

	return true;
}

#if 0
CWorkerThread::IJobPtr CWorkerThread::GetJob() const
{
	IJobPtr job;
	{
		CAutoCriticalSection acs(m_cs);

		// ҋ@邢͊̂
		ASSERT(IsReady());
		if (!IsReady()) {
			return IJobPtr();
		}

		job = m_job;
	}
	return job;
}
#endif

bool CWorkerThread::Stop()
{
	{
		CAutoCriticalSection acs(m_cs);

		// ҋ@邢͊̂
		ASSERT(IsReady());
		if (!IsReady())
			return false;

		// Xe[^XύXăCxgZbg
		m_Status = S_Ending;
		m_evStatus.Set();
	}

	return m_Thread.Stop();
}

bool CWorkerThread::ThreadFunc()
{
	IJobPtr job;

	while (true) {
		// ̃Wu΁Â܂܏𑱍s
		if (job) {
			CAutoCriticalSection acs(m_cs);

			m_job = job;
#if 0	// (̃WuƂ S_Running ̂܂ܕςĂȂ)
			m_Status = S_Running;
#endif
		}
		// ̃WuȂ΁AXe[^XύX҂
		else {
			// Xe[^XύX҂
			if (!m_evStatus.Wait()) {
				// G[
				m_evStatus.Reset();
				return false;
			}
			m_evStatus.Reset();

			// Xe[^XXV
			{
				CAutoCriticalSection acs(m_cs);

				switch (m_Status) {
				// WuJn
				case S_Starting:
					m_Status = S_Running;
					job = m_job;
					break;
				// XbhI
				case S_Ending:
					return true;
				// G[
				default:
					// XbhI
					m_Status = S_Ending;
					return false;
				}
			}
		}

		// Wus
		ASSERT(job);
		if (job) {
			// Wus
			bool keepJob = job->Execute();

			// WuR[obNĂяo
			ASSERT(m_OnCompleteJob);
			if (m_OnCompleteJob) {
				// (΁ÃWuԂ)
				job = m_OnCompleteJob(shared_from_this(), job, keepJob);
			}
			else {
				job.reset();
			}
		}
	}
}

/////////////////////////////////////////////////////////////////////////////
// CWorkerThread::ThreadRunner

CWorkerThread::ThreadRunner::ThreadRunner()
{
}

void CWorkerThread::ThreadRunner::SetThread(const CWorkerThreadPtr& thread)
{
	ASSERT(!m_thread.lock());
	m_thread = thread;
}

bool CWorkerThread::ThreadRunner::Initialize()
{
	// (Ȃ)
	ASSERT(m_thread.lock());
	return m_thread.lock();
}

DWORD CWorkerThread::ThreadRunner::Run()
{
	// CWorkerThread ̃Xbhnhs
	CWorkerThreadPtr thread = m_thread.lock();
	ASSERT(thread);
	return thread && thread->ThreadFunc() ? 0 : ~0;
}

bool CWorkerThread::ThreadRunner::Stop()
{
	// (Ȃ)
	return true;
}



/////////////////////////////////////////////////////////////////////////////
// CWorkerThreadPool

namespace {

	typedef boost::shared_ptr<CEvent>	CEventPtr;
	typedef std::vector<CEventPtr>		Events;

	/**
	 * Cxg҂.
	 */
	DWORD WaitForEvents(Events& events, BOOL bWaitAll)
	{
		// Cxgnh
		typedef std::vector<HANDLE> Handles;
		Handles handles;
		handles.reserve(events.size());
		for (Events::iterator it = events.begin(); it != events.end(); ++it) {
			ASSERT(*it && **it);
			handles.push_back(**it);
		}

		// Cxg҂
		return ::WaitForMultipleObjects(
				static_cast<DWORD>(handles.size()), &handles[0], bWaitAll, INFINITE);
	}

}	// anonymous namespace

CWorkerThreadPool::CWorkerThreadPool()
{
}

CWorkerThreadPool::~CWorkerThreadPool()
{
}

bool CWorkerThreadPool::Initialize(int nThreads)
{
	// TODO: ͈̂

	if (nThreads <= 0) {
		// gp\ȃvZbTRAɑ
		// (<= 0 Ȃ̂ŁAȂȂ. ŏ 1)
		nThreads = Max(1, CCpuMap::Instance().GetCurrentProcessAffinityBits() + nThreads);
	}

	// [J[Xbh
	m_AllThreads.clear();
	m_AllThreads.reserve(nThreads);
	for (int i = 0; i < nThreads; ++i) {
		// [J[Xbh쐬
		CWorkerThreadPtr thread(new CWorkerThread);
		// [J[Xbh
		if (!thread ||
			!thread->Initialize(
				boost::bind(&CWorkerThreadPool::OnCompleteJob, this, _1, _2, _3)))
		{
			return false;
		}

		m_AllThreads.push_back(thread);
	}

	return true;
}

size_t CWorkerThreadPool::GetThreadNum() const
{
	return m_AllThreads.size();
}

bool CWorkerThreadPool::Reset(int nGroupID)
{
	ASSERT(!m_AllThreads.empty());
	if (m_AllThreads.empty())
		return false;

	{
		CAutoCriticalSection acs(m_cs);

		// ȑÕWu܂sꂸɎcĂȂH
		bool bNotExecuted =
			GetFirstJobInfo(m_WaitingJobInfoQueue, nGroupID) != m_WaitingJobInfoQueue.end();
		// ȑÕWu񂪊҂ꂸɎcĂȂH
		bool bNotWaited =
			GetFirstJobInfo(m_RunningJobInfoQueue, nGroupID) != m_RunningJobInfoQueue.end();

		ASSERT(!bNotExecuted && !bNotWaited);

		// ܂ꂸɎcĂ
		if (bNotExecuted || bNotWaited) {
			acs.Leave();

			// ׂĂ̊҂
			VERIFY(WaitAll(nGroupID));
		}
	}

	return true;
}

bool CWorkerThreadPool::AddJob(const IJobPtr& job)
{
	if (!job) {
		return false;
	}

	CAutoCriticalSection acs(m_cs);

	// AChXbh擾
	CWorkerThreadPtr thread = GetIdleThread();
	// AChXbh
	if (thread) {
		// sL[ɓ
		m_RunningJobInfoQueue.push_back(JobInfo(job, true));

#if 0	// HACK: WusOɑ̃XbhɎ\̂ŁALeave() Ă͂ȂȂB
	// HACK: Ȃ Leave() Ăł̂H
		acs.Leave();
#endif

		// Wus
		VERIFY(thread->StartJob(job));
	}
	// AChXbhȂ
	else {
		// sҋ@L[ɓ
		// (Cxg͍쐬Ȃ)
		m_WaitingJobInfoQueue.push_back(JobInfo(job, false));
	}

	return true;
}

#if 0
bool CWorkerThreadPool::CancelJob(int nGroupID)
{
	CAutoCriticalSection acs(m_cs);

	JobInfoQueue::iterator it;
	// ҋ@L[̊Y邷ׂẴWuAҋ@L[폜
	while ((it = GetFirstJobInfo(m_WaitingJobInfoQueue, nGroupID)) != m_WaitingJobInfoQueue.end()) {
		m_WaitingJobInfoQueue.erase(it);
	}

	return true;
}

bool CWorkerThreadPool::WaitSingle(int nGroupID, IJobPtr& job)
{
	CAutoCriticalSection acs(m_cs);

	// Wu񂪂Ȃ
	if (m_RunningJobInfoQueue.empty()) {
		// ҋ@WuL[ɂ܂ĂȂ͂
		ASSERT(m_WaitingJobInfoQueue.empty());

		// (I)
		job.reset();
		return true;
	}

	// ܂ĂȂWũWuCxg
	// (Cxg폜Ȃ悤ɁACEventPtr ێĂKv)
	Events events;
	// HACK: ܂ĂȂWu̐́A[J[Xbh̐ȉ̂͂
	events.reserve(m_AllThreads.size());
	// s/WuL[犮̂
	for (JobInfoQueue::iterator it = m_RunningJobInfoQueue.begin(); it != m_RunningJobInfoQueue.end(); ++it) {
		// Ώۂ̃WuO[vǂ
		ASSERT(it->m_job);
		if (nGroupID != it->m_job->GetGroupID()) {
			continue;
		}

		CEventPtr& ev = it->m_Event;

		// Ă
		if (!ev) {
			// WuԂ
			job = it->m_job;

			// Wu폜
			m_RunningJobInfoQueue.erase(it);

			return true;
		}
		// ĂȂ
		else {
			// ҂߂̃Cxg擾
			events.push_back(ev);
		}
	}

	// [J[Xbh̐菭Ȃ
	// (őŃ[J[Xbh̐܂œɎs\̂ŁAҋ@̂̂҂ΏۂɊ܂߂)
	if (events.size() < m_AllThreads.size()) {
		// ҋ@WuL[
		for (JobInfoQueue::iterator it = m_WaitingJobInfoQueue.begin(); it != m_WaitingJobInfoQueue.end(); ++it) {
			// Ώۂ̃WuO[vǂ
			ASSERT(it->m_job);
			if (nGroupID != it->m_job->GetGroupID()) {
				continue;
			}

			CEventPtr& ev = it->m_Event;

			// Cxg܂쐬ĂȂ΍쐬
			if (!ev) {
				ev.reset(new CEvent);
			}

			// ҂߂̃Cxg擾
			events.push_back(ev);

			// [J[Xbh̐ɒBA҂Ώۂ͂ŏ\
			if (events.size() >= m_AllThreads.size())
				break;
		}
	}

	// Ώۂ̃WuO[v̂̂Aҋ@WuL[ɂs/WuL[ɂȂ
	if (events.empty()) {
		// (I)
		job.reset();
		return true;
	}

	// ̂Ȃ
	{
		CAutoLeaveCriticalSection alcs(m_cs);

		// Cxgnĥꂩ҂
		DWORD ret = WaitForEvents(events, FALSE);
		ASSERT(WAIT_OBJECT_0 <= ret && ret < WAIT_OBJECT_0 + events.size());
	}

	// s/WuL[犮̂ēx (K͂)
	for (JobInfoQueue::iterator it = m_RunningJobInfoQueue.begin(); it != m_RunningJobInfoQueue.end(); ++it) {
		// Ώۂ̃WuO[vǂ
		ASSERT(it->m_job);
		if (nGroupID != it->m_job->GetGroupID())
			continue;

		CEventPtr& ev = it->m_Event;

		// Ă
		if (!ev) {
			// WuԂ
			job = it->m_job;

			// Wu폜
			m_RunningJobInfoQueue.erase(it);

			return true;
		}
	}

	// Ȃ
	ASSERT(0);
	job.reset();
	return false;
}

bool CWorkerThreadPool::WaitFirst(int nGroupID, IJobPtr& job)
{
	CAutoCriticalSection acs(m_cs);

	// Wu񂪂Ȃ
	if (m_RunningJobInfoQueue.empty()) {
		// ҋ@WuL[ɂ܂ĂȂ͂
		ASSERT(m_WaitingJobInfoQueue.empty());

		// (I)
		job.reset();
		return true;
	}

	{
		// s/WuL[AΏۂ̃WuO[v̈ԏ߂̂̂
		JobInfoQueue::iterator it = GetFirstJobInfo(m_RunningJobInfoQueue, nGroupID);

		// Ώۂ̃WuO[v̂̂Ȃ
		if (it == m_RunningJobInfoQueue.end()) {
			// ҋ@WuL[ŏ̂̂
			it = GetFirstJobInfo(m_WaitingJobInfoQueue, nGroupID);

			// ҋ@WuL[ɂȂ
			if (it == m_WaitingJobInfoQueue.end()) {
				// (I)
				job.reset();
				return true;
			}

			// Cxg܂쐬ĂȂ΍쐬
			if (!it->m_Event) {
				it->m_Event.reset(new CEvent);
			}
		}

		// ԏ߂̃Wu
		JobInfo& info = *it;

		// ܂ĂȂ
		if (info.m_Event) {
			// NOTE: QƂ NG
			CEventPtr ev = info.m_Event;

			{
				CAutoLeaveCriticalSection alcs(m_cs);

				// ҂
				VERIFY(ev->Wait());
			}
		}
	}

	// Ô߂x擾Ȃ
	// (CriticalSection OĂԂ info ɂȂƂ̂)
	{
		// s/WuL[AΏۂ̃WuO[v̈ԏ߂̂̂
		JobInfoQueue::iterator it = GetFirstJobInfo(m_RunningJobInfoQueue, nGroupID);

		// Ώۂ̃WuO[v̂̂͂
		ASSERT(it != m_RunningJobInfoQueue.end());
		if (it == m_RunningJobInfoQueue.end()) {
			job.reset();
			return false;
		}

		// ԏ߂̃Wu
		JobInfo& info = *it;

		// ͂
		ASSERT(!info.m_Event);

		// WuԂ
		job = info.m_job;

		// Wu폜
		m_RunningJobInfoQueue.erase(it);
	}

	return true;
}
#endif

bool CWorkerThreadPool::WaitAll(int nGroupID, Jobs* pJobs)
{
	CAutoCriticalSection acs(m_cs);

	// Wu񂪂Ȃ
	if (m_RunningJobInfoQueue.empty()) {
		// ҋ@WuL[ɂ܂ĂȂ͂
		ASSERT(m_WaitingJobInfoQueue.empty());

		// (I)
		if (pJobs) {
			pJobs->clear();
		}
		return true;
	}

	while (true) {
		// s҂̍Ō̃Wu
		JobInfoQueue::reverse_iterator it = GetLastJobInfo(m_WaitingJobInfoQueue, nGroupID);
		// s҂̂̂
		if (it != m_WaitingJobInfoQueue.rend()) {
			JobInfo& info = *it;

			// Cxg܂쐬ĂȂ΍쐬
			if (!info.m_Event) {
				info.m_Event.reset(new CEvent);
				ASSERT(info.m_Event);
			}
			// NOTE: QƂ NG
			CEventPtr ev = info.m_Event;

			{
				CAutoLeaveCriticalSection alcs(m_cs);

				// ҂
				VERIFY(ev->Wait());
			}
		}

		// (܂łs҂̃Wu͂Ȃ͂Ȃ̂ŁAݎŝ̂҂ł悢)
		ASSERT(GetFirstJobInfo(m_WaitingJobInfoQueue, nGroupID) == m_WaitingJobInfoQueue.end());

		// ܂ĂȂWũWuCxg
		// (Cxg폜Ȃ悤ɁACEventPtr ێĂKv)
		Events events;
		// HACK: ܂ĂȂWu̐́A[J[Xbh̐ȉ̂͂
		events.reserve(m_AllThreads.size());
		for (JobInfoQueue::iterator it = m_RunningJobInfoQueue.begin(); it != m_RunningJobInfoQueue.end(); ++it) {
			// Ώۂ̃WuO[vǂ
			ASSERT(it->m_job);
			if (nGroupID != it->m_job->GetGroupID())
				continue;

			CEventPtr& ev = it->m_Event;

			// ĂȂ
			if (ev) {
				// ҂߂̃Cxg擾
				events.push_back(ev);
			}
		}

		// ĂȂWuȂ (ׂĊ)
		if (events.empty()) {
			break;
		}
		// ĂȂWu
		else {
			CAutoLeaveCriticalSection alcs(m_cs);

			// ׂĂ̊Cxgnh҂
			DWORD ret = WaitForEvents(events, TRUE);
			ASSERT(WAIT_OBJECT_0 <= ret && ret < WAIT_OBJECT_0 + events.size());
		}
	}

	// WuԂ
	if (pJobs) {
		Jobs& jobs = *pJobs;

		jobs.clear();
		jobs.reserve(m_RunningJobInfoQueue.size());
		for (JobInfoQueue::iterator it = m_RunningJobInfoQueue.begin(); it != m_RunningJobInfoQueue.end(); ++it) {
			// Ώۂ̃WuO[vǂ
			ASSERT(it->m_job);
			if (nGroupID != it->m_job->GetGroupID()) {
				continue;
			}

			// ς݂̂͂
			ASSERT(!it->m_Event);
			// WuԂ
			jobs.push_back(it->m_job);
		}
	}

	/**
	 * JobInfo ƃWuO[v ID r.
	 */
	class CCompareJobInfo : public std::binary_function<const JobInfo&, int, bool>
	{
	public:
		bool operator()(const JobInfo& info, int nGroupID) const
		{
			ASSERT(info.m_job);
			return nGroupID == info.m_job->GetGroupID();
		}
	};

	// Ώۂ̃WuO[v̂̂폜
	m_RunningJobInfoQueue.erase(
		std::remove_if(
			m_RunningJobInfoQueue.begin(), m_RunningJobInfoQueue.end(),
			boost::bind(CCompareJobInfo(), _1, nGroupID)),
		m_RunningJobInfoQueue.end());

	return true;
}

CWorkerThreadPtr CWorkerThreadPool::GetIdleThread()
{
	CAutoCriticalSection acs(m_cs);

	for (WorkerThreadVec::iterator it = m_AllThreads.begin(); it != m_AllThreads.end(); ++it) {
		CWorkerThreadPtr& thread = *it;
		ASSERT(thread);
		// AChXbh
		if (thread && thread->IsReady()) {
			return thread;
		}
	}

	// AChXbhȂ
	return CWorkerThreadPtr();
}

CWorkerThreadPool::JobInfoQueue::iterator
CWorkerThreadPool::GetFirstJobInfo(JobInfoQueue& queue, int nGroupID)
{
	JobInfoQueue::iterator it;
	for (it = queue.begin(); it != queue.end(); ++it) {
		// Ώۂ̃WuO[vǂ
		ASSERT(it->m_job);
		if (nGroupID == it->m_job->GetGroupID()) {
			// 
			break;
		}
	}

	return it;
}

CWorkerThreadPool::JobInfoQueue::reverse_iterator
CWorkerThreadPool::GetLastJobInfo(JobInfoQueue& queue, int nGroupID)
{
	JobInfoQueue::reverse_iterator it;
	for (it = queue.rbegin(); it != queue.rend(); ++it) {
		// Ώۂ̃WuO[vǂ
		ASSERT(it->m_job);
		if (nGroupID == it->m_job->GetGroupID()) {
			// 
			break;
		}
	}

	return it;
}

CWorkerThreadPool::IJobPtr CWorkerThreadPool::OnCompleteJob(
	const CWorkerThreadPtr& thread, const IJobPtr& job, bool keepJob)
{
	CAutoCriticalSection acs(m_cs);

	// s/WuL[猟
	JobInfoQueue::iterator it = m_RunningJobInfoQueue.end();
	for (it = m_RunningJobInfoQueue.begin(); it != m_RunningJobInfoQueue.end(); ++it) {
		if (it->m_job == job) {
			break;
		}
	}

	// K͂
	ASSERT(it != m_RunningJobInfoQueue.end());
	if (it != m_RunningJobInfoQueue.end()) {
		CEventPtr& ev = it->m_Event;
		ASSERT(ev);
		// ̂ŃCxgZbgĊJ
		if (ev) {
			VERIFY(ev->Set());
			ev.reset();
		}

		// ̌ WaitSingle() Ȃǂő҂\肪Ȃ̂ŁAL[폜
		if (!keepJob) {
			// Wu폜
			m_RunningJobInfoQueue.erase(it);
		}
	}

	// ҋ@WuL[ɃWu΁ÃWuƂĕԂ
	IJobPtr nextJob;
	// ̃Wu
	if (!m_WaitingJobInfoQueue.empty()) {
		// ̃Wu
		JobInfo& info = m_WaitingJobInfoQueue.front();
		// Cxg܂쐬ĂȂ΍쐬
		if (!info.m_Event) {
			info.m_Event.reset(new CEvent);
		}

		// ̃WuƂĕԂ
		nextJob = info.m_job;

		// sL[ɓ
		m_RunningJobInfoQueue.push_back(info);

		// (info ͎QƂȂ̂ŁAsL[ɈڂĂ폜Kv)
		m_WaitingJobInfoQueue.pop_front();

		// (̃WuƂ́AS_Complete ̏uԂȂ悤ɂ)
	}
	// ̃WuȂ
	else {
		// CriticalSection 𔲂OɁAXe[^XύX
		// (҂Ă AddJob()  IdleThread mɌoł悤ɂ邽)
		ASSERT(thread);
		thread->SetCompleteStatus_();
	}
	return nextJob;
}

}	// namespace GenericUtility
