/*
 * graph2D
 * Copyright (c) 2009 Shun Moriya <shun126@users.sourceforge.jp>
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 *  1. The origin of this software must not be misrepresented; you must not
 *     claim that you wrote the original software. If you use this software
 *     in a product, an acknowledgment in the product documentation would be
 *     appreciated but is not required.
 *
 *  2. Altered source versions must be plainly marked as such, and must not be
 *     misrepresented as being the original software.
 *
 *  3. This notice may not be removed or altered from any source
 *     distribution.
 */

#include "button.h"
#include "component.h"
#include "container.h"
#include "desktop.h"
#include "dialog.h"
#include "graphicDevice.h"
#include "menu.h"
#include "message.h"
#include "messageWindow.h"
#include "scene.h"
#include "text.h"
#include "textWindow.h"
#include "window.h"
#include <float.h>
#include <math.h>
#include <stdlib.h>

#define COMPONENT_DISABLE_COLOR_LEVEL	(0.58f)

#define COMPONENT_FLAGS_OPAQUE_BIT		(0x01)	//!< 不透明？
#define COMPONENT_FLAGS_CLIPPING_BIT	(0x02)	//!< クリッピングする？
#define COMPONENT_FLAGS_TOUCHABLE_BIT	(0x04)	//!< タッチ可能フラグ
#define COMPONENT_FLAGS_VISIBLE_BIT		(0x08)	//!< 表示するのか？
#define COMPONENT_FLAGS_ENABLE_BIT		(0x10)	//!< 有効？
#define COMPONENT_FLAGS_HIGHLIGHT_BIT	(0x20)	//!< FRAME_STYLE_PUSHEDまたはCOMPONENT_FLAGS_ENABLE_BITによる強調表示
#define COMPONENT_FLAGS_OWNERDRAW_BIT	(0x40)	//!< 自身で描画する？

#if !defined(M_PI)
#define M_PI 3.14159265358979323846f
#endif

namespace Graph2D
{
	enum AnimationPositionEnable
	{
		ANIMATION_POSITION_ENABLE_NONE = 0,
		ANIMATION_POSITION_ENABLE_MOVE,
		ANIMATION_POSITION_ENABLE_CHASE
	};
	std::map<unsigned int, Component*> Component::components;
	std::map<std::string, Texture*> Component::textures;
	std::stack<Component*> Component::mouseCaptureComponent;
	Component* Component::lastFoundComponent = NULL;
	unsigned int Component::componentCounter = 1;
	unsigned int Component::lastTouchedComponentID = 0;
	bool Component::drawDebugInfomation = false;
	bool Component::dragging;

	Component::Component()
		: type(TYPE_COMPONENT)
		, flags(COMPONENT_FLAGS_VISIBLE_BIT|COMPONENT_FLAGS_ENABLE_BIT|COMPONENT_FLAGS_HIGHLIGHT_BIT)
		, frameStyle(FRAME_STYLE_NOLINE|FRAME_STYLE_FLAT)
		, parent(NULL)
		, texture(NULL)
		, actionEvent(NULL)
		, scaleTime(0.0f)
		, scaleEnable(false)
		, fadeColorEnable(false)
		, animationPositionEnable(ANIMATION_POSITION_ENABLE_NONE)
		, jumpTime(0.0f)
		, shakeTime(0.0f)
		, data(0)
		, enableColor(M_PI / 2)
	{
		identification = componentCounter++;

		setBlendMode(GraphicDevice::BLEND_MODE_NORMAL);

		textureScrollOffset.zero();
		textureScrollSpeed.zero();

		scale.set(1.0f, 1.0f);

		color.set(1, 1, 1, 1);

		clearCellAnimation();

		// コンポーネントリストに追加
		components.insert(std::pair<unsigned int, Component*>(identification, this));
	}

	Component::~Component()
	{
		// コンポーネントリストから解放
		removeFromComponents();

		// アニメーションリストを開放
		for(std::map<std::string, CellAnimation*>::iterator i = animations.begin(); i != animations.end(); i++)
		{
			i->second->release();
		}
		animations.clear();

		// テクスチャーを開放
		unloadTexture();
	}

	void Component::removeFromComponents()
	{
		// マウスキャプチャーコンポーネントを消去
		if(isMouseCaptured())
		{
			mouseCaptureComponent.pop();
		}

		// 最後に検索したコンポーネントを消去
		if(lastFoundComponent && lastFoundComponent->getID() == identification)
		{
			lastFoundComponent->release();
			lastFoundComponent = NULL;
		}

		// コンポーネントリストから削除
		std::map<unsigned int, Component*>::iterator i = components.find(identification);
		if(i != components.end())
		{
			components.erase(i);
		}
	}

	void Component::release()
	{
		if(retainCount() == 1)
		{
			// 削除イベントを実行
			onDestroy(this);
		}

		super::release();
	}

	/*!
	 * 親からの登録を削除します
	 */
	void Component::removeFromParent()
	{
		if(parent)
		{
			parent->remove(this);
			parent = NULL;
		}
	}

	Scene* Component::getParentScene() const
	{
		Component* component = getParent();
		while(component)
		{
			if(component->isKindOfA(TYPE_SCENE))
				break;
			component = component->getParent();
		}
		return static_cast<Scene*>(component);
	}

	/*
	 * スクロール可能なコンテナにタッチ開始イベントを発行します
	 * @param[in]	localPosition	ローカル座標
	 * @return		開始したコンテナオブジェクトポインター
	 */
	Container* Component::forcusScrollableContainer(const Vector2& localPosition)
	{
		// 自分自身を確認
		if(isKindOfA(TYPE_CONTAINER))
		{
			Container* container = static_cast<Container*>(this);
			if(container->getScrollable())
			{
				if(container->isInPosition(localPosition))
				{
					onTouchesCancelled(localPosition);
					onTouchesBegan(localPosition);
					return container;
				}
			}
		}

		// 親を再帰しながら確認
		Vector2 currentPosition = getPosition() + localPosition;
		for(Container* container = getParent(); container; container = container->getParent())
		{
 			if(container->getScrollable())
			{
				if(container->isInPosition(currentPosition))
				{
					onTouchesCancelled(localPosition);
					container->onTouchesBegan(currentPosition);
					return container;
				}
			}
			// 親の座標系に変換
			currentPosition += container->getPosition();
		}

		// 発見できず
		return NULL;
	}
	
	/*
	 * スクロール可能なコンテナを取得します
	 * @return	開始したコンテナオブジェクトポインター
	 */
	Container* Component::getScrollableContainer()
	{
		for(Container* container = getParent(); container; container = container->getParent())
		{
			if(container->getScrollableContainer())
				return container;
		}

		// 発見できず
		return NULL;
	}

	/*
	 * スクリーン座標を取得します
	 * @return	スクリーン座標
	 */
	Vector2 Component::getScreenPosition() const
	{
		Vector2 position;
		
		for(const Component* component = this; component; component = component->getParent())
		{
			position += component->getLeftTopPosition();
		}
		
		return position;
	}

	/*
	 * スクリーン座標をローカル座標に変換します
	 * @param[in]	screenPosition	スクリーン座標
	 * @return		ローカル座標
	 */
	Vector2 Component::translatePosition(const Vector2& screenPosition) const
	{
		Vector2 position = screenPosition;
		
		for(const Component* component = this; component; component = component->getParent())
		{
			position -= component->getPosition();
		}
		
		return position;
	}

#if 0
	Vector2 Component::getBottomPosition() const
	{
		Vector2 bottomPosition;
		bottomPosition = getCenterPosition();
		bottomPosition.y += getSize().y * getScale().y;

		if(jumpTime > 0.0f)
		{
			bottomPosition.y += jumpHeight * sinf(jumpTime * jumpAngle);
		}

		return bottomPosition;
	}
#endif

	//! サイズを設定します
	void Component::setSize(const Vector2& size)
	{
		// イベント呼び出し
		if(onSize(size))
		{
			// サイズ設定
			this->size = size;
			// イベント呼び出し
			onSizeChanged();
		}
	}

	float Component::getWidth() const
	{
		return size.x;
	}

	void Component::setWidth(const float w)
	{
		// イベント呼び出し
		if(onSize(Vector2(w, size.y)))
		{
			// サイズ設定
			size.x = w;
			// イベント呼び出し
			onSizeChanged();
		}
	}

	float Component::getHeight() const
	{
		return size.y;
	}

	void Component::setHeight(const float h)
	{
		// イベント呼び出し
		if(onSize(Vector2(size.x, h)))
		{
			// サイズ設定
			size.y = h;
			// イベント呼び出し
			onSizeChanged();
		}
	}

	void Component::setScale(const Vector2& scale, const float second)
	{
		if(second <= 0)
		{
			this->scale = scale;
			scaleEnable = false;
		}
		else
		{
			animationTargetScale = scale;
			animationDeltaScale = (scale - this->scale) * (1.0f / second);
			scaleTime = second;
			scaleEnable = true;
		}
	}

	void Component::setColor(const Color& color, const float second)
	{
		if(second <= 0)
		{
			this->color = color;
			fadeColorEnable = false;
		}
		else
		{
			fadeTargetColor = color;
			fadeDeltaColor = (color - this->color) * (1.0f / second);
			fadeColorTime = second;
			fadeColorEnable = true;
		}
	}

	bool Component::loadTextureCore(Texture** texture, const std::string& filename)
	{
		// 同じテクスチャを読み込んでいるか調べます
		if(*texture && (*texture)->getFileName() == filename)
		{
			return true;
		}

		// テクスチャを開放
		unloadTextureCore(texture);

		// テクスチャが読み込み済みか調べます
		std::map<std::string, Texture*>::iterator i = textures.find(filename);
		if(i != textures.end())
		{
			(*texture) = i->second;
			(*texture)->retain();
			return true;
		}
		else
		{
			*texture = new Texture();

			if((*texture)->load(filename, false))
			{
				// テクスチャリストに追加
				textures.insert(std::pair<std::string, Texture*>(filename, *texture));
				return true;
			}
			else
			{
				(*texture)->release();
				(*texture) = NULL;
				return false;
			}
		}
	}

	void Component::unloadTextureCore(Texture** texture)
	{
		if(*texture)
		{
			textures.erase((*texture)->getFileName());
			(*texture)->release();
			(*texture) = NULL;
		}
	}

	void Component::setTexture(Texture* texture)
	{
		if(this->texture != texture)
		{
			unloadTexture();

			this->texture = texture;
		}

		if(this->texture)
		{
			this->texture->retain();
		}
	}

	void Component::setCellAnimation(const Vector2& position, const Vector2& size, const float wait, const unsigned short count, const short loopCount)
	{
		animationCellStartPosition = position;
		animationCellSize = size;
		animationCellTime = 0;
		animationCellWait = wait;
		animationCellCount = 0;
		animationCellMaxCount = count;
		animationCellLoopCountOrigin = loopCount;
		animationCellLoopCount = loopCount;
		animationCellEnable = true;
		animationCellRunning = false;
	}

	float Component::getCellAnimationTime() const
	{
		// 無限ループ？
		if(animationCellLoopCount < 0)
			return FLT_MAX;
		// セルアニメーションの時間を計算
		return animationCellWait * animationCellMaxCount * (animationCellLoopCount + 1);
	}

	void Component::setCell(const Vector2& position, const Vector2& size)
	{
		setCellAnimation(position, size, FLT_MAX, 1, -1);
	}

	void Component::removeAnimation(const std::string& name)
	{
		std::map<std::string, CellAnimation*>::iterator i = animations.find(name);
		if(i != animations.end())
		{
			i->second->release();

			animations.erase(i);
		}
	}

	bool Component::setAnimation(const std::string& name)
	{
		std::map<std::string, CellAnimation*>::iterator i = animations.find(name);
		if(i == animations.end())
		{
			return false;
		}
		else
		{
			CellAnimation* animation = i->second;
			setCellAnimation(animation->getPosition(), animation->getSize(), animation->getWait(), animation->getCount(), animation->getLoop());
			startCellAnimation();
			return true;
		}
	}

	CellAnimation* Component::getAnimation(const std::string& name)
	{
		const std::map<std::string, CellAnimation*>::iterator i = animations.find(name);
		if(i != animations.end())
		{
			return i->second;
		}else{
			return NULL;
		}
	}

	void Component::move(const Vector2& position, const float second)
	{
		if(second <= 0)
		{
			this->position = position;
		}
		else
		{
			animationTargetPosition = position;
			animationDeltaPosition = (position - this->position) * (1.0f / second);
			animationPositionTime = second;
			animationPositionEnable = ANIMATION_POSITION_ENABLE_MOVE;
		}
	}

	void Component::chase(const Vector2& position, const float rate)
	{
		if(rate >= 1.0f)
		{
			this->position = position;
		}
		else
		{
			animationTargetPosition = position;
			animationPositionTime = rate;
			animationPositionEnable = ANIMATION_POSITION_ENABLE_CHASE;
		}
	}

	//! 到着したか調べます
	bool Component::arrived() const
	{
		return animationPositionEnable == ANIMATION_POSITION_ENABLE_NONE;
	}

	//! 不透明状態を取得します
	bool Component::getOpaque() const
	{
		return flags & COMPONENT_FLAGS_OPAQUE_BIT;
	}

	//! 不透明状態を設定します
	void Component::setOpaque(const bool opaque)
	{
		if(opaque)
			flags |= COMPONENT_FLAGS_OPAQUE_BIT;
		else
			flags &= ~COMPONENT_FLAGS_OPAQUE_BIT;
	}

	//! シザリング状態を取得します
	bool Component::getClipping() const
	{
		return flags & COMPONENT_FLAGS_CLIPPING_BIT;
	}

	//! シザリング状態を設定します
	void Component::setClipping(const bool clipping)
	{
		if(clipping)
			flags |= COMPONENT_FLAGS_CLIPPING_BIT;
		else
			flags &= ~COMPONENT_FLAGS_CLIPPING_BIT;
	}

	//! タッチ可能状態を取得します
	bool Component::getTouchable() const
	{
		return flags & COMPONENT_FLAGS_TOUCHABLE_BIT;
	}

	//! タッチ可能状態を設定します
	void Component::setTouchable(const bool touchable)
	{
		if(touchable)
			flags |= COMPONENT_FLAGS_TOUCHABLE_BIT;
		else
			flags &= ~COMPONENT_FLAGS_TOUCHABLE_BIT;
	}

	//! 表示するか取得
	bool Component::getVisible() const
	{
		return flags & COMPONENT_FLAGS_VISIBLE_BIT;
	}

	//! 表示するか設定
	void Component::setVisible(const bool visible)
	{
		if(visible)
			flags |= COMPONENT_FLAGS_VISIBLE_BIT;
		else
			flags &= ~COMPONENT_FLAGS_VISIBLE_BIT;
	}

	//! 有効・無効設定
	bool Component::getEnable() const
	{
		return flags & COMPONENT_FLAGS_ENABLE_BIT;
	}

	//! 有効か取得
	void Component::setEnable(const bool enable)
	{
		if(enable)
			flags |= COMPONENT_FLAGS_ENABLE_BIT;
		else
			flags &= ~COMPONENT_FLAGS_ENABLE_BIT;
	}

	//! ハイライトドローフラグ取得
	bool Component::getHighlightFlag() const
	{
		return flags & COMPONENT_FLAGS_HIGHLIGHT_BIT;
	}

	//! ハイライトドローフラグ設定
	void Component::setHighlightFlag(const bool enable)
	{
		if(enable)
			flags |= COMPONENT_FLAGS_HIGHLIGHT_BIT;
		else
			flags &= ~COMPONENT_FLAGS_HIGHLIGHT_BIT;
	}

	//! オーナードローフラグ取得
	bool Component::getOwnerDraw() const
	{
		return flags & COMPONENT_FLAGS_OWNERDRAW_BIT;
	}

	//! オーナードローフラグ設定
	void Component::setOwnerDraw(const bool ownerDraw)
	{
		if(ownerDraw)
			flags |= COMPONENT_FLAGS_OWNERDRAW_BIT;
		else
			flags &= ~COMPONENT_FLAGS_OWNERDRAW_BIT;
	}

	int Component::getUserData() const							//!< ユーザーデータを取得します
	{
		return data;
	}

	//!< ユーザーデータを設定します
	void Component::setUserData(const int data)
	{
		this->data = data;
	}

	//! マウスキャプチャー中？
	bool Component::isMouseCaptured() const
	{
		return mouseCaptureComponent.empty() ? false : (mouseCaptureComponent.top() == this);
	}

	//! マウスキャプチャー状態にする
	Component* Component::setMouseCapture(Component* component)
	{
		Component* oldMouseCapturedComponent = Component::getMouseCapture();
		mouseCaptureComponent.push(component);
		return oldMouseCapturedComponent;
	}

	//! マウスキャプチャー状態を解除する
	bool Component::releaseMouseCapture(Component* component)
	{
		if(!mouseCaptureComponent.empty() && mouseCaptureComponent.top() == component)
		{
			mouseCaptureComponent.pop();
			return true;
		}
		else
		{
			return false;
		}

	}

	//! マウスキャプチャーしているコンポーネントを取得する
	Component* Component::getMouseCapture()
	{
		return mouseCaptureComponent.empty() ? NULL : mouseCaptureComponent.top();
	}

	Component* Component::deserialize(mana_stream* stream)
	{
		Component* component;

		switch(mana_stream_get_unsigned_integer(stream))
		{
		case TYPE_COMPONENT:
			component = new Component();
			break;

		case TYPE_SCROLL_BAR:
			break;

		case TYPE_TEXT:
			component = new Text();
			break;
			
		case TYPE_CONTAINER:
			component = new Container();
			break;
				
		case TYPE_IMAGE_BUTTON:
			component = new ImageButton();
			break;

		case TYPE_BUTTON:
			component = new Button();
			break;

		case TYPE_MESSAGE:
			component = new Message();
			break;
			
		case TYPE_SCENE:
			component = new Scene();
			break;

		case TYPE_WINDOW:
			component = new Window();
			break;

		case TYPE_MENU:
			component = new Menu();
			break;
			
		case TYPE_DIALOG:
			component = new Dialog();
			break;

		case TYPE_MESSAGE_WINDOW:
			component = new MessageWindow();
			break;
			
		case TYPE_TEXT_WINDOW:
			component = new TextWindow();
			break;
			
		default:
			abort(ERROR_CODE_INVALID, "unknown component type detect.");
			component = NULL;
			break;
		}

		component->onDeserialize(stream);

		return component;
	}

	void Component::allTestSerialize()
	{
		{
			Component* component = new Component();
			component->testSerialize();
			component->release();
		}
		{
			Button* component = new Button();
			component->testSerialize();
			component->release();
		}
		{
			Text* component = new Text();
			component->testSerialize();
			component->release();
		}
		{
			Container* component = new Container();
			component->testSerialize();
			component->release();
		}
		{
			Window* component = new Window();
			component->testSerialize();
			component->release();
		}
		{
			Dialog* component = new Dialog();
			component->testSerialize();
			component->release();
		}
		{
			Menu* component = new Menu();
			component->testSerialize();
			component->release();
		}
		{
			Message* component = new Message();
			component->testSerialize();
			component->release();
		}
	}

	/*!
	 * @param[in]	stream	mana_stream オブジェクト
	 */
	void Component::onSerialize(mana_stream* stream) const
	{
		mana_stream_push_unsigned_integer(stream, type);
		mana_stream_push_unsigned_integer(stream, identification);
		mana_stream_push_unsigned_integer(stream, parent ? parent->getID() : 0);
		mana_stream_push_unsigned_char(stream, frameStyle);

		position.serialize(stream);
		size.serialize(stream);
		offset.serialize(stream);

		scale.serialize(stream);
		animationDeltaScale.serialize(stream);
		animationTargetScale.serialize(stream);
		mana_stream_push_float(stream, scaleTime);
		mana_stream_push_unsigned_char(stream, scaleEnable);

		color.serialize(stream);

		mana_stream_push_unsigned_char(stream, texture != NULL);
		if(texture)
			texture->serialize(stream);

		animationCellStartPosition.serialize(stream);
		animationCellSize.serialize(stream);
		mana_stream_push_float(stream, animationCellTime);
		mana_stream_push_float(stream, animationCellWait);
		mana_stream_push_unsigned_short(stream, animationCellCount);
		mana_stream_push_unsigned_short(stream, animationCellMaxCount);
		mana_stream_push_short(stream, animationCellLoopCount);
		mana_stream_push_unsigned_char(stream, animationCellEnable);
		mana_stream_push_unsigned_char(stream, animationCellRunning);

		mana_stream_mark(stream);

		fadeDeltaColor.serialize(stream);
		fadeTargetColor.serialize(stream);
		mana_stream_push_float(stream, fadeColorTime);
		mana_stream_push_unsigned_char(stream, fadeColorEnable);

		animationDeltaPosition.serialize(stream);
		animationTargetPosition.serialize(stream);
		mana_stream_push_float(stream, animationPositionTime);
		mana_stream_push_unsigned_char(stream, animationPositionEnable);

		mana_stream_push_float(stream, jumpTime);
		mana_stream_push_float(stream, jumpAngle);
		mana_stream_push_float(stream, jumpHeight);

		mana_stream_push_float(stream, shakeTime);
		mana_stream_push_float(stream, shakeLength);
		mana_stream_push_float(stream, shakeAmplitude);

		mana_stream_push_integer(stream, data);

		mana_stream_push_unsigned_char(stream, flags);
		mana_stream_push_unsigned_char(stream, blendMode);

		mana_stream_mark(stream);
	}

	/*!
	 * @param[in]	stream	mana_stream オブジェクト
	 */
	void Component::onDeserialize(mana_stream* stream)
	{
		removeFromComponents();

		type = mana_stream_pop_unsigned_integer(stream);
		identification = mana_stream_pop_unsigned_integer(stream);
		parent = reinterpret_cast<Container*>(Component::find(mana_stream_pop_unsigned_integer(stream)));
		frameStyle = mana_stream_pop_unsigned_char(stream);

		position.deserialize(stream);
		size.deserialize(stream);
		offset.deserialize(stream);

		scale.deserialize(stream);
		animationDeltaScale.deserialize(stream);
		animationTargetScale.deserialize(stream);
		scaleTime = mana_stream_pop_float(stream);
		scaleEnable = mana_stream_pop_unsigned_char(stream) ? true : false;

		color.deserialize(stream);

		if(mana_stream_pop_unsigned_char(stream))
		{
			texture = new Texture();
			texture->deserialize(stream);
		}

		animationCellStartPosition.deserialize(stream);
		animationCellSize.deserialize(stream);
		animationCellTime = mana_stream_pop_float(stream);
		animationCellWait = mana_stream_pop_float(stream);
		animationCellCount = mana_stream_pop_unsigned_short(stream);
		animationCellMaxCount = mana_stream_pop_unsigned_short(stream);
		animationCellLoopCount = mana_stream_pop_short(stream);
		animationCellEnable = mana_stream_pop_unsigned_char(stream) ? true : false;
		animationCellRunning = mana_stream_pop_unsigned_char(stream) ? true : false;

		mana_stream_check(stream);

		fadeDeltaColor.deserialize(stream);
		fadeTargetColor.deserialize(stream);
		fadeColorTime = mana_stream_pop_float(stream);
		fadeColorEnable = mana_stream_pop_unsigned_char(stream) ? true : false;

		animationDeltaPosition.deserialize(stream);
		animationTargetPosition.deserialize(stream);
		animationPositionTime = mana_stream_pop_float(stream);
		animationPositionEnable = mana_stream_pop_unsigned_char(stream);

		jumpTime = mana_stream_pop_float(stream);
		jumpAngle = mana_stream_pop_float(stream);
		jumpHeight = mana_stream_pop_float(stream);

		shakeTime = mana_stream_pop_float(stream);
		shakeLength = mana_stream_pop_float(stream);
		shakeAmplitude = mana_stream_pop_float(stream);

		data = mana_stream_pop_integer(stream);

		flags = mana_stream_pop_unsigned_char(stream);
		blendMode = mana_stream_pop_unsigned_char(stream);

		mana_stream_check(stream);

		// コンポーネントリストに追加
		components.insert(std::pair<unsigned int, Component*>(identification, static_cast<Component*>(retain())));

		// コンポーネントカウンターの調整
		if(componentCounter <= identification)
			componentCounter = identification + 1;
	}

	void Component::onUpdate(const UpdateInfomation& updateInfomation)
	{
		// スケールアニメーションの更新
		if(scaleEnable)
		{
			scale += animationDeltaScale * updateInfomation.deltaTime;

			scaleTime -= updateInfomation.deltaTime;
			if(scaleTime <= 0)
			{
				scale = animationTargetScale;
				scaleEnable = false;
			}
		}

		// セルアニメーションの更新
		if(animationCellEnable)
		{
			if(animationCellRunning)
			{
				animationCellTime += updateInfomation.deltaTime;
				while(animationCellTime >= animationCellWait)
				{
					animationCellCount++;
					if(animationCellCount >= animationCellMaxCount)
					{
						if(animationCellLoopCount >= 0)
						{
							animationCellLoopCount--;
							if(animationCellLoopCount <= 0)
							{
#if 1
								setVisible(false);
#else
								animationCellCount = animationCellMaxCount - 1;
#endif
								animationCellRunning = false;
								break;
							}
						}
						animationCellCount = 0;
					}
					animationCellTime -= animationCellTime;
				}
			}
		}
		else
		{
			// テクスチャースクロールの更新
			// (セルアニメーション中はスクロール禁止)
			if(!textureScrollSpeed.empty())
			{
				textureScrollOffset += textureScrollSpeed * updateInfomation.deltaTime;
			}
		}

		// カラーアニメーションの更新
		if(fadeColorEnable)
		{
			color += fadeDeltaColor * updateInfomation.deltaTime;

			fadeColorTime -= updateInfomation.deltaTime;
			if(fadeColorTime <= 0)
			{
				color = fadeTargetColor;
				fadeColorEnable = false;
			}
		}

		// 位置アニメーションの更新
		switch(animationPositionEnable)
		{
		case ANIMATION_POSITION_ENABLE_NONE:
			break;

		case ANIMATION_POSITION_ENABLE_MOVE:
			position += animationDeltaPosition * updateInfomation.deltaTime;

			animationPositionTime -= updateInfomation.deltaTime;
			if(animationPositionTime <= 0)
			{
				position = animationTargetPosition;
				animationPositionEnable = ANIMATION_POSITION_ENABLE_NONE;
			}
			break;

		case ANIMATION_POSITION_ENABLE_CHASE:
			{
				const Vector2 delta = animationTargetPosition - position;

				position += delta * pow(animationPositionTime, updateInfomation.stepCount);

				if(delta.squareLength() <= 1.0f * 1.0f)
				{
					position = animationTargetPosition;
					animationPositionEnable = ANIMATION_POSITION_ENABLE_NONE;
				}
			}
			break;
		}

		// オフセットで操作する位置アニメーションの更新
		{
			offset.zero();

			// ジャンプ
			if(jumpTime > 0.0f)
			{
				offset.y -= jumpHeight * sinf(jumpTime * jumpAngle);
				jumpTime -= updateInfomation.deltaTime;
			}

			// 振動
			if(shakeTime > 0.0f)
			{
				offset.x += (float)(rand()) / (float)(RAND_MAX);
				offset.y += (float)(rand()) / (float)(RAND_MAX);

				float amplitude = shakeAmplitude * (shakeTime / shakeLength);
				offset.x *= amplitude;
				offset.y *= amplitude;

				shakeTime -= updateInfomation.deltaTime;
			}
		}

		// enableによる輝度の変化
		if(getEnable())
		{
			if(enableColor >= M_PI / 2)
			{
				enableColor = M_PI / 2;
			}
			else
			{
				enableColor += M_PI / 2 * 2 * updateInfomation.deltaTime;
			}
		}
		else
		{
			if(enableColor <= 0)
			{
				enableColor = 0;
			}
			else
			{
				enableColor -= M_PI / 2 * 2 * updateInfomation.deltaTime;
			}
		}
	}

	Color Component::getDrawColor(const DrawRect& drawRect) const
	{
		return getDrawColor(drawRect, color);
	}

	Color Component::getDrawColor(const DrawRect& drawRect, const Color& color) const
	{
		Color baseColor = drawRect.getDrawColor(color);

		if(getHighlightFlag() && !getOwnerDraw())
        {
			if(frameStyle & FRAME_STYLE_PUSHED)
			{
                baseColor.setMulRGB(COMPONENT_DISABLE_COLOR_LEVEL);
			}
			else
			{
				baseColor.setMulRGB(COMPONENT_DISABLE_COLOR_LEVEL + (1.0f - COMPONENT_DISABLE_COLOR_LEVEL) * sin(enableColor));
			}
        }

		return baseColor;
	}

	/*!
	 * @param[in]	drawRect	自身を描画するための環境
	 */
	void Component::onDraw(const DrawRect& drawRect)
	{
		if(getVisible() == false)
			return;
		if(getOwnerDraw() == true)
			return;
		if(frameStyle & FRAME_STYLE_CUSTOM)
			return;

		Color baseColor = getDrawColor(drawRect);
		if(baseColor.a <= 0.0f)
			return;

		const Vector2 p1 = drawRect.getDrawLeftTopPosition();
		const Vector2 p2 = drawRect.getDrawRightBottomPosition();
		if(!Desktop::getInstance().rangeOfVisibility(p1, p2))
			return;

		const Vector2 p3(p2.x, p1.y);
		const Vector2 p4(p1.x, p2.y);

		if(texture)
			texture->bind();

		unsigned char primitiveType = 0;

		Vector2 lastScissorPosition;
		Vector2 lastScissorSize;

		if(getClipping())
		{
			GraphicDevice::getScissor(lastScissorPosition, lastScissorSize);
			GraphicDevice::setScissor(drawRect.clippingLeftTop, drawRect.getClippingSize());
			primitiveType = GraphicDevice::TYPE_SCISSOR;
		}
		if(getOpaque())
			primitiveType |= GraphicDevice::TYPE_OPAQUE;
		if(texture)
			primitiveType |= GraphicDevice::TYPE_TEXTURE_COORDINATE;
		if(!getEnable())
			baseColor.setMulRGB(COMPONENT_DISABLE_COLOR_LEVEL + (1.0f - COMPONENT_DISABLE_COLOR_LEVEL) * sin(enableColor));
		if(frameStyle & FRAME_STYLE_FLAT)
		{
			GraphicDevice::setColor(baseColor);
		}else{
			primitiveType |= GraphicDevice::TYPE_COLOR;
			GraphicDevice::setColor(Color::WHITE);
		}
		GraphicDevice::setBlendMode(getBlendMode());

		GraphicDevice::begin(primitiveType, GraphicDevice::TRIANGLES, baseColor);

		if(!(frameStyle & FRAME_STYLE_FLAT))
		{
			GraphicDevice::addColor(baseColor);
			GraphicDevice::addColor(baseColor.mulRGB(0.50f));
			GraphicDevice::addColor(baseColor.mulRGB(0.25f));
			GraphicDevice::addColor(baseColor.mulRGB(0.50f));
			GraphicDevice::addColor(baseColor.mulRGB(0.25f));
			GraphicDevice::addColor(baseColor.mulRGB(0.10f));
		}

		if(texture)
		{
			const Vector2& textureSize = texture->getMaxSize();
			const Vector2& texSize = texture->getSize();
			if(animationCellEnable && (texSize.x != 0 && texSize.y != 0))
			{
				const Vector2& textureSize = texture->getTextureSize();
				const int horizonCount = static_cast<int>(texSize.x / animationCellSize.x);
				const float x = animationCellStartPosition.x + static_cast<float>(animationCellCount % horizonCount) * animationCellSize.x;
				const float y = animationCellStartPosition.y + static_cast<float>(animationCellCount / horizonCount) * animationCellSize.y;
				const float u = x / textureSize.x;
				const float v = y / textureSize.y;
				const float w = animationCellSize.x / textureSize.x;
				const float h = animationCellSize.y / textureSize.y;

				GraphicDevice::addTextureCoord(Vector2(u    , v    ));
				GraphicDevice::addTextureCoord(Vector2(u + w, v    ));
				GraphicDevice::addTextureCoord(Vector2(u    , v + h));
				GraphicDevice::addTextureCoord(Vector2(u + w, v    ));
				GraphicDevice::addTextureCoord(Vector2(u    , v + h));
				GraphicDevice::addTextureCoord(Vector2(u + w, v + h));
			}
			else
			{
				GraphicDevice::addTextureCoord(textureScrollOffset);
				GraphicDevice::addTextureCoord(Vector2(textureScrollOffset.x + textureSize.x, textureScrollOffset.y));
				GraphicDevice::addTextureCoord(Vector2(textureScrollOffset.x, textureScrollOffset.y + textureSize.y));
				GraphicDevice::addTextureCoord(Vector2(textureScrollOffset.x + textureSize.x, textureScrollOffset.y));
				GraphicDevice::addTextureCoord(Vector2(textureScrollOffset.x, textureScrollOffset.y + textureSize.y));
				GraphicDevice::addTextureCoord(textureScrollOffset + textureSize);
			}
		}

		GraphicDevice::addVertex(p1);
		GraphicDevice::addVertex(p3);
		GraphicDevice::addVertex(p4);
		GraphicDevice::addVertex(p3);
		GraphicDevice::addVertex(p4);
		GraphicDevice::addVertex(p2);

		GraphicDevice::end();

		primitiveType &= GraphicDevice::TYPE_SCISSOR;

		if(!(frameStyle & FRAME_STYLE_NOLINE))
		{
			GraphicDevice::begin(primitiveType, GraphicDevice::LINE_LOOP, baseColor.mulRGB((frameStyle & FRAME_STYLE_PUSHED) ? 2.0f : 0.1f));
			GraphicDevice::addVertex(p1);
			GraphicDevice::addVertex(p3);
			GraphicDevice::addVertex(p2);
			GraphicDevice::addVertex(p4);
			GraphicDevice::end();
		}

		if(drawDebugInfomation)
		{
			GraphicDevice::begin(primitiveType, GraphicDevice::TRIANGLES, Color(1.0f, 0.0f, 0.0f, 0.1f));
			GraphicDevice::addVertex(p1);
			GraphicDevice::addVertex(p3);
			GraphicDevice::addVertex(p4);
			GraphicDevice::addVertex(p3);
			GraphicDevice::addVertex(p4);
			GraphicDevice::addVertex(p2);
			GraphicDevice::end();

			GraphicDevice::begin(primitiveType|GraphicDevice::TYPE_OPAQUE, GraphicDevice::LINE_LOOP, Color(1.0f, 0.0f, 0.0f, 1.0f));
			GraphicDevice::addVertex(p1);
			GraphicDevice::addVertex(p3);
			GraphicDevice::addVertex(p2);
			GraphicDevice::addVertex(p4);
			GraphicDevice::end();
		}

		if(getClipping())
		{
			GraphicDevice::setScissor(lastScissorPosition, lastScissorSize);
		}
	}

	//! タッチ開始イベント
	bool Component::onTouchesBegan(const Vector2& localPosition)
	{
#if GRAPH2D_LOG_DETAIL_LEVEL >= 1
		MANA_TRACE("%s::onTouchesBegan:(%f,%f)\n", getTypeName(), localPosition.x, localPosition.y);
#endif

		// タッチ開始したローカル座標を記録
		toucheBeganPosition = localPosition;
		// ドラッギング中フラグを初期化
		dragging = false;

		return getTouchable() || isMouseCaptured();
	}

	bool Component::onTouchesMoved(const Vector2& localPosition)
	{
#if GRAPH2D_LOG_DETAIL_LEVEL >= 2
		MANA_TRACE("%s::onTouchesMoved:(%f,%f)\n", getTypeName(), localPosition.x, localPosition.y);
#endif

		// ドラッギング開始？
		if(!dragging)
		{
			if(toucheBeganPosition.squareDistance(localPosition) >= 5 * 5)
			{
				Container* component = forcusScrollableContainer(localPosition);
				if(component)
				{
					component->setMouseCapture();
					component->dragging = true;
				}
			}
		}

		return getTouchable() || isMouseCaptured();
	}

	bool Component::onTouchesEnded(const Vector2& localPosition)
	{
#if GRAPH2D_LOG_DETAIL_LEVEL >= 1
		MANA_TRACE("%s::onTouchesEnded:(%f,%f)\n", getTypeName(), localPosition.x, localPosition.y);
#endif

		if(getTouchable())
		{
			// 最後に触れたIDを記録します
			lastTouchedComponentID = identification;
			
			if(actionEvent)
			{
				actionEvent->onAction(this, getUserData());
			}
		}

		// マウスキャプチャー中？
		const bool captured = isMouseCaptured();

		// ドラッギング中？
		if(dragging)
		{
			releaseMouseCapture();
			dragging = false;
		}

		return getTouchable() || captured;
	}
	
	bool Component::onTouchesCancelled(const Vector2& localPosition)
	{
#if GRAPH2D_LOG_DETAIL_LEVEL >= 1
		MANA_TRACE("%s::onTouchesCancelled:(%f,%f)\n", getTypeName(), localPosition.x, localPosition.y);
#endif
		
		(void)localPosition;

		// マウスキャプチャー中？
		const bool captured = isMouseCaptured();

		// ドラッギング中？
		if(dragging)
		{
			releaseMouseCapture();
			dragging = false;
		}

		return getTouchable() || captured;
	}

	//! サイズ変更イベント
	bool Component::onSize(const Vector2& size)
	{
		return size.x >= 0 && size.y >= 0;
	}

	//! サイズ変更イベント
	void Component::onSizeChanged()
	{
	}

	bool Component::touchesBegan(const Vector2& localPosition)
	{
		if(!getEnable())
			return false;
		if(!getVisible())
			return false;
		if(!isMouseCaptured() && !isInPosition(localPosition))
			return false;
		return onTouchesBegan(localPosition);
	}
	
	bool Component::touchesMoved(const Vector2& localPosition)		//!< タッチ移動
	{
		if(!getEnable())
			return false;
		if(!getVisible())
			return false;
		if(!isMouseCaptured() && !isInPosition(localPosition))
			return false;
		return onTouchesMoved(localPosition);
	}
	
	bool Component::touchesEnded(const Vector2& localPosition)		//!< タッチ終了
	{
		if(!getEnable())
			return false;
		if(!getVisible())
			return false;
		if(!isMouseCaptured() && !isInPosition(localPosition))
			return false;
		return onTouchesEnded(localPosition);
	}
	
	bool Component::touchesCancelled(const Vector2& localPosition)	//!< タッチキャンセル
	{
		if(!getEnable())
			return false;
		if(!getVisible())
			return false;
		if(!isMouseCaptured() && !isInPosition(localPosition))
			return false;
		return onTouchesCancelled(localPosition);
	}

	bool Component::compare(const Component& window) const
	{
		(void)window;
/*
 if(toucheBeganPosition != other.toucheBeganPosition)
 return false;
 */
		abort();
		return true;
	}

	void Component::panel(const Vector2& p1, const Vector2& size, const Color& color, const unsigned int frameStyle)
	{
		const Vector2 p2 = p1 + size;
		if(!Desktop::getInstance().rangeOfVisibility(p1, p2))
			return;
		const Vector2 p3(p2.x, p1.y);
		const Vector2 p4(p1.x, p2.y);

		Color baseColor(color);

		unsigned char primitiveType = 0;
		if(baseColor.a >= 1.0f)
			primitiveType |= GraphicDevice::TYPE_OPAQUE;
		if(frameStyle & FRAME_STYLE_PUSHED)
			baseColor.setMulRGB(COMPONENT_DISABLE_COLOR_LEVEL);
		if(frameStyle & FRAME_STYLE_FLAT)
		{
			GraphicDevice::setColor(baseColor);
		}
		else
		{
			primitiveType |= GraphicDevice::TYPE_COLOR;
			GraphicDevice::setColor(Color::WHITE);
		}

		GraphicDevice::begin(primitiveType, GraphicDevice::TRIANGLES, baseColor);

		if(!(frameStyle & FRAME_STYLE_FLAT))
		{
			GraphicDevice::addColor(baseColor);
			GraphicDevice::addColor(baseColor.mulRGB(0.50f));
			GraphicDevice::addColor(baseColor.mulRGB(0.25f));
			GraphicDevice::addColor(baseColor.mulRGB(0.50f));
			GraphicDevice::addColor(baseColor.mulRGB(0.25f));
			GraphicDevice::addColor(baseColor.mulRGB(0.10f));
		}

		GraphicDevice::addVertex(p1);
		GraphicDevice::addVertex(p3);
		GraphicDevice::addVertex(p4);
		GraphicDevice::addVertex(p3);
		GraphicDevice::addVertex(p4);
		GraphicDevice::addVertex(p2);

		GraphicDevice::end();

		if(!(frameStyle & FRAME_STYLE_NOLINE))
		{
			GraphicDevice::begin(primitiveType, GraphicDevice::LINE_LOOP, baseColor.mulRGB((frameStyle & FRAME_STYLE_PUSHED) ? 2.0f : 0.1f));
			GraphicDevice::addVertex(p1);
			GraphicDevice::addVertex(p3);
			GraphicDevice::addVertex(p2);
			GraphicDevice::addVertex(p4);
			GraphicDevice::end();
		}
	}

	void Component::clearAll()
	{
		// マウスキャプチャーしたコンポーネントを消去
		while(!mouseCaptureComponent.empty())
			mouseCaptureComponent.pop();

		// 最後に検索したコンポーネントを開放
		if(lastFoundComponent)
		{
			lastFoundComponent->release();
			lastFoundComponent = NULL;
		}

		// 登録されている全てのコンポーネントを開放
		for(std::map<unsigned int, Component*>::iterator i = components.begin(); i != components.end(); i++)
		{
			(i->second)->release();
		}
		components.clear();

		clearComponentCounter();

		// componentsとの参照を切っているので、解放エラーになる
		abort();
	}

	void Component::dumpAll()
	{
		mana_print_debug("--------------------------------------------------\n");
		for(std::map<unsigned int, Component*>::iterator i = components.begin(); i != components.end(); i++)
		{
			const unsigned int identification = i->first;
			const Component* component = i->second;
			if(component->isKindOfA(TYPE_TEXT))
			{
				const Text* text = reinterpret_cast<const Text*>(component);
				mana_print_debug("%d:%s(%d) %s\n", identification, text->getTypeName(), text->retainCount(), text->getString().c_str());
			}
			else
			{
				mana_print_debug("%d:%s(%d)\n", identification, component->getTypeName(), component->retainCount());
			}
		}
		for(std::map<std::string, Texture*>::iterator i = textures.begin(); i != textures.end(); i++)
		{
			const Texture* texture = i->second;
			mana_print_debug("%s(%d)\n", texture->getFileName().c_str(), texture->retainCount());
		}
	}

	// @todo:スクリプトから何度も呼び出される。可能な限り高速化してください。
	Component* Component::find(const unsigned int identification)
	{
		// 最後に検索したコンポーネントと一致した？
		if(lastFoundComponent && lastFoundComponent->getID() == identification)
		{
			return lastFoundComponent;
		}

		std::map<unsigned int, Component*>::iterator i = components.find(identification);
		if(i == components.end())
			return NULL;

		Component* foundComponent = i->second;
		if(foundComponent)
		{
			Component* component = lastFoundComponent;
			lastFoundComponent = NULL;
			if(component)
				component->release();

			lastFoundComponent = foundComponent;
			if(lastFoundComponent)
				lastFoundComponent->retain();
		}
		return foundComponent;
	}

	void Component::testSerialize() const
	{
		mana_stream* stream = mana_stream_create();

		onSerialize(stream);

		Component* other = Component::deserialize(stream);
		assert(compare(*other));
		other->release();

		mana_stream_destroy(stream);
	}
}
