/*
 * graph2D
 * Copyright (c) 2009 Shun Moriya <shun126@users.sourceforge.jp>
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 *  1. The origin of this software must not be misrepresented; you must not
 *     claim that you wrote the original software. If you use this software
 *     in a product, an acknowledgment in the product documentation would be
 *     appreciated but is not required.
 *
 *  2. Altered source versions must be plainly marked as such, and must not be
 *     misrepresented as being the original software.
 *
 *  3. This notice may not be removed or altered from any source
 *     distribution.
 */

/*!
 * @file
 * @brief	複数の Component に関するソースファイル
 *
 * このファイルは複数の Component に関するソースファイルです。
 *
 * @author	Shun Moriya <shun126@users.sourceforge.jp>
 */

#include "abort.h"
#include "container.h"
#include "graphicDevice.h"
#include <libmana.h>
#include <stdlib.h>
#include <string.h>

namespace Graph2D
{
	static const int DEFAULT_PAGE_SIZE = 10;

	Vector2 Container::draggingVelocity;

	Container::Container()
		: children(NULL)
		, allocatedSize(0)
		, pageSize(DEFAULT_PAGE_SIZE)
		, leftFrameSize(0)
		, topFrameSize(0)
		, rightFrameSize(0)
		, bottomFrameSize(0)
		, flags(GRAPH2D_CONTAINER_FLAG_COLOR_AFFECTED)
    {
		type = TYPE_CONTAINER;
		memset(frameTexture, NULL, sizeof(frameTexture));

		setClipping(false);
	}

	Container::~Container()
	{
		scrollOffset.zero();
		scrollTargetOffset.zero();
		scrollOffsetLimit.zero();

		for(int i = 0; i <= LAYOUT_SOUTH_WEST; i++)
		{
			unloadFrameTexture(static_cast<Layout>(i));
		}

		clear();
	}

	void Container::allocate(const size_t count)
	{
		for(int i = 0; i < allocatedSize; i++)
		{
			if(children[i])
				children[i]->release();
		}

		children = (Component**)g2d_realloc(children, count * sizeof(Component*));
		if(children == NULL)
			abort(ERROR_CODE_NOT_ENOUGH_MEMORY);

		memset(children, 0, count * sizeof(Component*));

		allocatedSize = count;
	}

	Component* Container::get(const unsigned int index) const
	{
		return (index < allocatedSize) ? children[index] : NULL;
	}

	void Container::set(const unsigned int index, Component* component)
	{
		if(index >= allocatedSize)
		{
			children = (Component**)g2d_realloc(children, (index + pageSize) * sizeof(Component*));
			if(children == NULL)
				abort(ERROR_CODE_NOT_ENOUGH_MEMORY);

			memset(&children[allocatedSize], 0, (index + pageSize - allocatedSize) * sizeof(Component*));
			allocatedSize = index + pageSize;
		}

		if(children[index])
		{
			children[index]->release();
		}

		children[index] = component;
		if(children[index])
		{
			children[index]->retain();
			children[index]->parent = this;
		}
	}

	void Container::append(Component* component)
	{
#if 0
		for(long i = (long)allocatedSize - 1; i >= 0; i--)
#else
		for(size_t i = 0; i < allocatedSize; i++)
#endif
		{
			Component* child = children[i];
			if(child == NULL)
			{
				set(i, component);
				return;
			}
		}

		set(allocatedSize, component);
	}

	void Container::remove(Component* component)
	{
		for(size_t i = 0; i < allocatedSize; i++)
		{
			if(children[i] == component)
			{
				if(children[i])
				{
					children[i]->release();
					children[i] = NULL;
				}
				break;
			}
		}
	}

	bool Container::find(Component* component) const
	{
		for(size_t i = 0; i < allocatedSize; i++)
		{
			if(children[i] == component)
				return true;
		}
		return false;
	}

	void Container::swap(const unsigned int to, const unsigned int from)
	{
		Component* component = get(to);
		if(component)
			component->retain();
		set(to, get(from));
		set(from, component);
		if(component)
			component->release();
	}

	void Container::clear()
	{
		for(unsigned int i = 0; i < allocatedSize; i++)
		{
			if(children[i])
			{
				children[i]->release();
				children[i] = NULL;
			}
		}

		g2d_free(children);
		children = NULL;
		allocatedSize = 0;
	}

	void Container::compact()
	{
		long index = (long)allocatedSize - 1;
		while(index > 0)
		{
			if(children[index])
				break;
			--index;
		}
		if(index < 0)
		{
			clear();
		}
		else if(allocatedSize < static_cast<size_t>(index + 1))
		{
			allocatedSize = index + 1;
			children = (Component**)g2d_realloc(children, allocatedSize * sizeof(Component*));
		}
	}

	bool Container::loadFrameTexture(const Container::Layout direction, const std::string& filename/*, const bool repeat_h, const bool repeat_v*/)
	{
		/*
		const unsigned int shift = direction * 2;
		if(repeat_h)
			flags |= (GRAPH2D_CONTAINER_FLAG_FRAME_TEXTURE_REPEAT_H << shift);
		else
			flags &= ~(GRAPH2D_CONTAINER_FLAG_FRAME_TEXTURE_REPEAT_H << shift);

		if(repeat_v)
			flags |= (GRAPH2D_CONTAINER_FLAG_FRAME_TEXTURE_REPEAT_V << shift);
		else
			flags &= ~(GRAPH2D_CONTAINER_FLAG_FRAME_TEXTURE_REPEAT_V << shift);
		*/
		return loadTextureCore(&frameTexture[direction], filename);
	}

	void Container::unloadFrameTexture(const Container::Layout direction)
	{
		unloadTextureCore(&frameTexture[direction]);
	}

	const Vector2& Container::getFrameTexCoord(const Layout direction) const
	{
		return frameTexCoord[direction];
	}

	void Container::setFrameTexCoord(const Container::Layout direction, const Vector2& frameTexCoord)
	{
		this->frameTexCoord[direction] = frameTexCoord;
	}

	void Container::setFrameTexCoord(const Container::Layout direction, const float u, const float v)
	{
		setFrameTexCoord(direction, Vector2(u, v));
	}

	/*
	void Container::applyLayout(Layout* layout)
	{
		if(layout)
			layout->apply(this);
	}
	*/

	void Container::onSerialize(mana_stream* stream) const
	{
		super::onSerialize(stream);

		size_t count = 0;
		for(unsigned int i = 0; i < allocatedSize; i++)
		{
			if(children[i])
				count++;
		}
		mana_stream_push_unsigned_integer(stream, count);
		mana_stream_push_unsigned_integer(stream, pageSize);
		mana_stream_push_unsigned_integer(stream, flags);

		for(unsigned int i = 0; i < allocatedSize; i++)
		{
			if(children[i])
			{
				mana_stream_mark(stream);
				mana_stream_push_unsigned_integer(stream, i);
				children[i]->onSerialize(stream);
			}
		}
		mana_stream_mark(stream);
	}

	void Container::onDeserialize(mana_stream* stream)
	{
		super::onDeserialize(stream);

		const unsigned int count = mana_stream_pop_unsigned_integer(stream);
		pageSize = mana_stream_pop_unsigned_integer(stream);
		flags = mana_stream_pop_unsigned_integer(stream);

		for(unsigned int i = 0; i < count; i++)
		{
			mana_stream_check(stream);
			const unsigned int index = mana_stream_pop_unsigned_integer(stream);
			set(index, Component::deserialize(stream));
		}
		mana_stream_check(stream);
	}

	void Container::onUpdate(const UpdateInfomation& updateInfomation)
	{
		for(unsigned int i = 0; i < allocatedSize; i++)
		{
			Component* child = children[i];
			if(child)
				child->onUpdate(updateInfomation);
		}

		if(getScrollable())
		{
			const Vector2 delta = (scrollTargetOffset - scrollOffset);
			scrollOffset += delta * (1.0f / 0.18f) * updateInfomation.deltaTime;

			if(!dragging)
			{
				scrollTargetOffset += draggingVelocity;				
				correctScrollTargetOffset();
			}

			draggingVelocity *= 0.8f;
		}

		super::onUpdate(updateInfomation);
	}

	/*!
	 * シェルソート
	 * Donald L. Shellが開発した改良挿入ソートのアルゴリズム。高速だが、安定ソートではない。
	 */
	void Container::shellSort(Component** array, const unsigned int length) const
	{
		unsigned int h;

		/* 交換処理の間隔を決めるループ */
		for(h = 1; h < length; h = h * 2 + 1)
			;

		while(h > 1)
		{
			/* hは交換処理の間隔を表す */
			h /= 2;

			for(unsigned int i = h; i < length; i++)
			{
				Component* work = array[i];
				const Vector2& workDepth  = work->getBottomPosition();

				int j = i - h;
				while(workDepth.y > array[j]->getBottomPosition().y)
				{
					array[j + h] = array[j];
					j = j - h;
					if(j < 0)
						break;
				}
				array[j + h] = work;
			}
		}
	}

	void Container::drawChild(const DrawRect& drawRect, Component* component)
	{
		if(component && component->getVisible())
		{
			const Vector2 drawLeftTop = drawRect.getDrawLeftTopPosition();
			const Vector2 drawRightBottom = drawLeftTop + drawRect.getDrawSize();
			const Vector2 drawPosition = drawLeftTop + scrollOffset * drawRect.scale;

			DrawRect newDrawRect(drawRect);
			newDrawRect.position = drawPosition + component->getPosition() * drawRect.scale;
			newDrawRect.size = component->getSize();
			newDrawRect.scale = component->getScale() * drawRect.scale;
			newDrawRect.updateClippingInfomation(drawRect);
			newDrawRect.setScrollOffset(getScrollPosition());

			if(flags & GRAPH2D_CONTAINER_FLAG_COLOR_AFFECTED)
			{
				newDrawRect.color *= color;
			}

			const Vector2 rb = newDrawRect.position + newDrawRect.size * drawRect.scale;
			if(newDrawRect.position.x < drawRightBottom.x && rb.x >= drawLeftTop.x && newDrawRect.position.y < drawRightBottom.y && rb.y >= drawLeftTop.y)
			{
				Vector2 lastScissorPosition;
				Vector2 lastScissorSize;
				
				if(getClipping())
				{
					GraphicDevice::getScissor(lastScissorPosition, lastScissorSize);
					GraphicDevice::setScissor(drawRect.clippingLeftTop, drawRect.getClippingSize());
				}

				component->onDraw(newDrawRect);

				if(getClipping())
				{
					GraphicDevice::setScissor(lastScissorPosition, lastScissorSize);
				}
			}
		}
	}

	void Container::drawChildren(const DrawRect& drawRect, Component** array, const unsigned int length)
	{
		Vector2 lastScissorPosition;
		Vector2 lastScissorSize;
		
		if(getClipping())
		{
			GraphicDevice::getScissor(lastScissorPosition, lastScissorSize);
			GraphicDevice::setScissor(drawRect.clippingLeftTop, drawRect.getClippingSize());
		}

		const Vector2 drawLeftTop = drawRect.getDrawLeftTopPosition();
		const Vector2 drawRightBottom = drawLeftTop + drawRect.getDrawSize();
		const Vector2 drawPosition = drawLeftTop + scrollOffset * drawRect.scale;

		for(long i = (long)length - 1; i >= 0; i--)
		{
			Component* child = array[i];
			if(child && child->getVisible())
			{
				DrawRect newDrawRect(drawRect);
				newDrawRect.position = drawPosition + child->getPosition() * drawRect.getScale();
				newDrawRect.size = child->getSize();
				newDrawRect.scale = child->getScale() * drawRect.getScale();
				newDrawRect.updateClippingInfomation(drawRect);
				if(flags & GRAPH2D_CONTAINER_FLAG_COLOR_AFFECTED)
					newDrawRect.color *= color;

#if 0
				const Vector2 rightBottom = newDrawRect.position + newDrawRect.size * drawRect.scale;
				if(newDrawRect.position.x < drawRightBottom.x)
				{
					if(rightBottom.x >= drawLeftTop.x)
					{
						if(newDrawRect.position.y < drawRightBottom.y)
						{
							if(rightBottom.y >= drawLeftTop.y)
								child->onDraw(newDrawRect);
						}
					}
				}
#else
				child->onDraw(newDrawRect);
#endif
			}
		}

		if(getClipping())
		{
			GraphicDevice::setScissor(lastScissorPosition, lastScissorSize);
		}
	}

	void Container::drawFrameParts(const Layout direction, const Vector2& v0, const Vector2& v1, const Vector2& v2, const Vector2& v3, const Color& color, const float width, const float height)
	{
		if(frameTexture[direction])
		{
			frameTexture[direction]->bind();

			const Vector2 textureSize = frameTexture[direction]->getTextureSize();
			const Vector2 uv0((frameTexCoord[direction].x        ) / textureSize.x, (frameTexCoord[direction].y         ) / textureSize.y);
			const Vector2 uv1((frameTexCoord[direction].x + width) / textureSize.x, (frameTexCoord[direction].y         ) / textureSize.y);
			const Vector2 uv2((frameTexCoord[direction].x        ) / textureSize.x, (frameTexCoord[direction].y + height) / textureSize.y);
			const Vector2 uv3((frameTexCoord[direction].x + width) / textureSize.x, (frameTexCoord[direction].y + height) / textureSize.y);

			GraphicDevice::begin(GraphicDevice::TYPE_TEXTURE_COORDINATE, GraphicDevice::TRIANGLES, color);
			GraphicDevice::addTextureCoord(uv0);
			GraphicDevice::addTextureCoord(uv1);
			GraphicDevice::addTextureCoord(uv2);
			GraphicDevice::addTextureCoord(uv1);
			GraphicDevice::addTextureCoord(uv2);
			GraphicDevice::addTextureCoord(uv3);
			GraphicDevice::addVertex(v0);
			GraphicDevice::addVertex(v1);
			GraphicDevice::addVertex(v2);
			GraphicDevice::addVertex(v1);
			GraphicDevice::addVertex(v2);
			GraphicDevice::addVertex(v3);
			GraphicDevice::end();
		}
	}

	void Container::drawFrame(const DrawRect& drawRect)
	{
		if(getVisible() == false || getOwnerDraw() == true)
			return;

		Color baseColor = getDrawColor(drawRect);
		if(baseColor.a <= 0.0f)
			return;

		GraphicDevice::setBlendMode(static_cast<GraphicDevice::BlendMode>(blendMode));
		GraphicDevice::enableTexture(true);

		//////////////////////////////////////////////////////////////////////
		const Vector2 lt = drawRect.getDrawLeftTopPosition();
		const Vector2 rb = drawRect.getDrawRightBottomPosition();

		const float leftFrameSize = this->leftFrameSize * drawRect.scale.x;
		const float topFrameSize = this->topFrameSize * drawRect.scale.y;
		const float rightFrameSize = this->rightFrameSize * drawRect.scale.x;
		const float bottomFrameSize = this->bottomFrameSize * drawRect.scale.y;

		// 左上頂点
		const Vector2 vltlt(lt.x                 , lt.y                  );
		const Vector2 vltrt(lt.x + leftFrameSize , lt.y                  );
		const Vector2 vltlb(lt.x                 , lt.y + topFrameSize   );
		const Vector2 vltrb(lt.x + leftFrameSize , lt.y + topFrameSize   );
		// 右上頂点
		const Vector2 vrtlt(rb.x - rightFrameSize, lt.y                  );
		const Vector2 vrtrt(rb.x                 , lt.y                  );
		const Vector2 vrtlb(rb.x - rightFrameSize, lt.y + topFrameSize   );
		const Vector2 vrtrb(rb.x                 , lt.y + topFrameSize   );
		// 左下頂点
		const Vector2 vlblt(lt.x                 , rb.y - bottomFrameSize);
		const Vector2 vlbrt(lt.x + leftFrameSize , rb.y - bottomFrameSize);
		const Vector2 vlblb(lt.x                 , rb.y                  );
		const Vector2 vlbrb(lt.x + leftFrameSize , rb.y                  );
		// 右下頂点
		const Vector2 vrblt(rb.x - rightFrameSize, rb.y - bottomFrameSize);
		const Vector2 vrbrt(rb.x                 , rb.y - bottomFrameSize);
		const Vector2 vrblb(rb.x - rightFrameSize, rb.y                  );
		const Vector2 vrbrb(rb.x                 , rb.y                  );
		// フレーム内部のサイズ
		const float widthFrameSize = vrtlb.x - vltrb.x;
		const float heightFrameSize = vlbrt.y - vltrb.y;

		//////////////////////////////////////////////////////////////////////
		// 描画
		drawFrameParts(LAYOUT_CENTER    , vltrb, vrtlb, vlbrt, vrblt, baseColor, widthFrameSize, heightFrameSize);
		drawFrameParts(LAYOUT_NORTH     , vltrt, vrtlt, vltrb, vrtlb, baseColor, widthFrameSize, topFrameSize);
		drawFrameParts(LAYOUT_SOUTH     , vlbrt, vrblt, vlbrb, vrblb, baseColor, leftFrameSize, topFrameSize);
		drawFrameParts(LAYOUT_EAST      , vrtlb, vrtrb, vrblt, vrbrt, baseColor, rightFrameSize, heightFrameSize);
		drawFrameParts(LAYOUT_WEST      , vltlb, vltrb, vlblt, vlbrt, baseColor, leftFrameSize, heightFrameSize);
		drawFrameParts(LAYOUT_NORTH_EAST, vrtlt, vrtrt, vrtlb, vrtrb, baseColor, rightFrameSize, topFrameSize);
		drawFrameParts(LAYOUT_NORTH_WEST, vltlt, vltrt, vltlb, vltrb, baseColor, rightFrameSize, topFrameSize);
		drawFrameParts(LAYOUT_SOUTH_EAST, vrblt, vrbrt, vrblb, vrbrb, baseColor, leftFrameSize, bottomFrameSize);
		drawFrameParts(LAYOUT_SOUTH_WEST, vlblt, vlbrt, vlblb, vlbrb, baseColor, leftFrameSize, bottomFrameSize);
	}

	void Container::onDraw(const DrawRect& drawRect)
	{
		super::onDraw(drawRect);

		drawFrame(drawRect);

		DrawRect newDrawRect(drawRect);
		newDrawRect.position += Vector2(leftFrameSize, topFrameSize) * drawRect.scale;
		newDrawRect.size -= Vector2(leftFrameSize + rightFrameSize, topFrameSize + bottomFrameSize) * drawRect.scale;
		newDrawRect.updateClippingInfomation(drawRect);

		if(allocatedSize > 0)
		{
			if(flags & GRAPH2D_CONTAINER_FLAG_PERSPECTIVE_MODE)
			{
				Component** orderingTable = (Component**)g2d_malloc(allocatedSize * sizeof(Component*));
				if(orderingTable)
				{
					unsigned int j = 0;
					for(unsigned int i = 0; i < allocatedSize; i++)
					{
						Component* child = children[i];
						if(child && child->getVisible())
						{
							orderingTable[j] = child;
							j++;
						}
					}
					shellSort(orderingTable, j);
					drawChildren(newDrawRect, orderingTable, j);
				}
				g2d_free(orderingTable);
			}
			else
			{
				drawChildren(newDrawRect, children, allocatedSize);
			}
		}
	}

	/*
	 * スクロール先ターゲットをスクロール範囲に収めます
	 */
	void Container::correctScrollTargetOffset()
	{
		// スクロール範囲する必要が無いならスクロール範囲を無効にする
		Vector2 limit = size - scrollOffsetLimit;
		if(limit.x > 0.0f)
			limit.x = 0.0f;
		if(limit.y > 0.0f)
			limit.y = 0.0f;
		
		// スクロール先がスクロール範囲外なら、範囲内に収める
		if(scrollTargetOffset.x > 0.0f)
			scrollTargetOffset.x = 0.0f;
		else if(scrollTargetOffset.x < limit.x)
			scrollTargetOffset.x = limit.x;
		
		if(scrollTargetOffset.y > 0.0f)
			scrollTargetOffset.y = 0.0f;
		else if(scrollTargetOffset.y < limit.y)
			scrollTargetOffset.y = limit.y;
	}

	bool Container::onTouchesBegan(const Vector2& localPosition)
	{
		if(getScrollable())
		{
			// 現在のスクロール値を記録
			toucheBeganScrollOffset = scrollTargetOffset = scrollOffset;
			// ドラッギング用速度を初期化
			draggingVelocity.zero();
		}

		return super::onTouchesBegan(localPosition) | getScrollable();
	}

	bool Container::onTouchesMoved(const Vector2& localPosition)
	{
		if(getScrollable() && dragging)
		{
			// 目標スクロール値を設定
			scrollTargetOffset = toucheBeganScrollOffset + (localPosition - toucheBeganPosition);
			// スクロール範囲する必要が無いならスクロール範囲を無効にする
			const Vector2 limit = size - scrollOffsetLimit;
			if(limit.x >= 0.0f)
				scrollOffset.x = scrollTargetOffset.x = 0.0f;
			if(limit.y >= 0.0f)
				scrollOffset.y = scrollTargetOffset.y = 0.0f;
			// スクロール速度を計算
			draggingVelocity += scrollTargetOffset - scrollOffset;
			// タッチ中はすぐ反応する
			scrollOffset = scrollTargetOffset;
		}

		return super::onTouchesMoved(localPosition) | getScrollable();
	}

	bool Container::onTouchesEnded(const Vector2& localPosition)
	{
		if(getScrollable())
		{
			// 目標スクロール値を設定
			scrollTargetOffset = toucheBeganScrollOffset + (localPosition - toucheBeganPosition);
			// スクロール範囲を計算
			correctScrollTargetOffset();
		}

		return super::onTouchesEnded(localPosition) | getScrollable();
	}

	bool Container::onTouchesCancelled(const Vector2& localPosition)
	{
		if(getScrollable())
		{
			// 目標スクロール値を設定
			scrollTargetOffset = toucheBeganScrollOffset + (localPosition - toucheBeganPosition);
			// スクロール範囲を計算
			correctScrollTargetOffset();
		}

		return super::onTouchesCancelled(localPosition) | getScrollable();
	}

	//! サイズ変更イベント
	bool Container::onSize(const Vector2& size)
	{
		const float w = size.x - getLeftFrameSize() + getRightFrameSize();
		const float h = size.y - getTopFrameSize() + getBottomFrameSize();
		return w >= 0 && h >= 0;
	}

	bool Container::touchesBegan(const Vector2& localPosition)
	{
		if(!getEnable())
			return false;
		if(!getVisible())
			return false;
		if(!isMouseCaptured() && !isInPosition(localPosition))
			return false;

		// フレームを考慮したローカル座標を計算
		const Vector2 positionInFrame = localPosition - scrollOffset - Vector2(leftFrameSize, topFrameSize);

		for(unsigned int i = 0; i < allocatedSize; i++)
		{
			Component* child = children[i];
			if(child == NULL)
				continue;
			if(!child->getEnable())
				continue;
			if(!child->getVisible())
				continue;

			const Vector2 childLocalPosition = positionInFrame - child->getPosition();
			if(child->touchesBegan(childLocalPosition))
				return true;
		}

		return onTouchesBegan(localPosition);
	}

	bool Container::touchesMoved(const Vector2& localPosition)
	{
		if(!getEnable())
			return false;
		if(!getVisible())
			return false;
		if(!isMouseCaptured() && !isInPosition(localPosition))
			return false;
		
		// ドラッギング中は子コンポーネントにイベントを通知しない
		if(!dragging)
		{
			// フレームを考慮したローカル座標を計算
			const Vector2 positionInFrame = localPosition - scrollOffset - Vector2(leftFrameSize, topFrameSize);

			for(unsigned int i = 0; i < allocatedSize; i++)
			{
				Component* child = children[i];
				if(child == NULL)
					continue;
				if(!child->getEnable())
					continue;
				if(!child->getVisible())
					continue;

				const Vector2 childLocalPosition = positionInFrame - child->getPosition();
				if(child->isInPosition(childLocalPosition))
				{
					if(child->touchesMoved(childLocalPosition))
						return true;
				}
			}
		}

		return onTouchesMoved(localPosition);
	}

	bool Container::touchesEnded(const Vector2& localPosition)
	{
		if(!getEnable())
			return false;
		if(!getVisible())
			return false;
		if(!isMouseCaptured() && !isInPosition(localPosition))
			return false;

		// ドラッギング中は子コンポーネントにイベントを通知しない
		if(!dragging)
		{
			// フレームを考慮したローカル座標を計算
			const Vector2 positionInFrame = localPosition - scrollOffset - Vector2(leftFrameSize, topFrameSize);

			for(unsigned int i = 0; i < allocatedSize; i++)
			{
				Component* child = children[i];
				if(child == NULL)
					continue;
				if(!child->getEnable())
					continue;
				if(!child->getVisible())
					continue;

				const Vector2 childLocalPosition = positionInFrame - child->getPosition();
				if(child->isInPosition(childLocalPosition))
				{
					if(child->touchesEnded(childLocalPosition))
						return true;
				}
			}
		}

		return onTouchesEnded(localPosition);
	}

	bool Container::touchesCancelled(const Vector2& localPosition)
	{
		if(!getEnable())
			return false;
		if(!getVisible())
			return false;
		if(!isMouseCaptured() && !isInPosition(localPosition))
			return false;

		// ドラッギング中は子コンポーネントにイベントを通知しない
		if(!dragging)
		{
			// フレームを考慮したローカル座標を計算
			const Vector2 positionInFrame = localPosition - scrollOffset - Vector2(leftFrameSize, topFrameSize);

			for(unsigned int i = 0; i < allocatedSize; i++)
			{
				Component* child = children[i];
				if(child == NULL)
					continue;
				if(!child->getEnable())
					continue;
				if(!child->getVisible())
					continue;

				const Vector2 childLocalPosition = positionInFrame - child->getPosition();
				if(child->isInPosition(childLocalPosition))
				{
					if(child->touchesCancelled(childLocalPosition))
						return true;
				}
			}
		}

		return onTouchesCancelled(localPosition);
	}

	bool Container::compare(const Container& other) const
	{
		if(!super::compare(other))
			return false;

		if(allocatedSize != other.allocatedSize)
			return false;

		for(size_t i = 0; i < allocatedSize; i++)
		{
			if(children[i] == NULL && other.children[i] == NULL)
				continue;
			if(children[i] == NULL || other.children[i] == NULL)
				return false;

			if(children[i]->compare(*other.children[i]) == false)
				return false;
		}

		if(flags != other.flags)
			return false;
		if(toucheBeganScrollOffset != other.toucheBeganScrollOffset)
			return false;
		if(scrollOffset != other.scrollOffset)
			return false;
		if(scrollTargetOffset != other.scrollTargetOffset)
			return false;
		if(scrollOffsetLimit != other.scrollOffsetLimit)
			return false;

		return true;
	}
}
