/*
 * graph2D
 * Copyright (c) 2009 Shun Moriya <shun126@users.sourceforge.jp>
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 *  1. The origin of this software must not be misrepresented; you must not
 *     claim that you wrote the original software. If you use this software
 *     in a product, an acknowledgment in the product documentation would be
 *     appreciated but is not required.
 *
 *  2. Altered source versions must be plainly marked as such, and must not be
 *     misrepresented as being the original software.
 *
 *  3. This notice may not be removed or altered from any source
 *     distribution.
 */

#include "message.h"
#include "text.h"

namespace Graph2D
{
	Message::Message()
		: delayTimer(0)
		, length(0)
		, timer(1.0f / 18.0f)
	{
		type = TYPE_MESSAGE;

		setColor(0, 0, 0, 0, 0);
		setFrameStyle(FRAME_STYLE_NOLINE|FRAME_STYLE_FLAT);

		text = new Text();
		text->setColor(0, 0, 0, 0, 0.0f);
		text->setPosition(0, 0);
		set(0, text);

		compact();
	}

	Message::~Message()
	{
		clear();

		if(text)
		{
			text->release();
		}
	}

	void Message::clear()
	{
		text->clear();

		message.clear();

		length = 0;
	}

	const std::string& Message::getString() const
	{
		return message;
	}

	void Message::setString(const std::string& message)
	{
		clear();

		this->message = message;
		this->wait = 0;
		this->delayWait = 0;
		this->clicked = false;
	}

	void Message::format(const char* format, ...)
	{
		va_list ap;
		va_start(ap, format);
		text->vformat(format, ap);
		va_end(ap);
	}

	void Message::vformat(const char* format, va_list ap)
	{
		text->vformat(format, ap);
	}

	void Message::replace(const char* key, const char* word)
	{
		text->replace(key, word);
	}

	const Color& Message::getTextColor() const
	{
		return text->getTextColor();
	}

	void Message::setTextColor(const Color& color)
	{
		text->setTextColor(color);
	}

	void Message::setTextColor(const float r, const float g, const float b, const float a)
	{
		setTextColor(Color(r, g, b, a));
	}

	Text& Message::getText()
	{
		return *text;
	}

	const Text& Message::getText() const
	{
		return getText();
	}

	float Message::getTimer() const
	{
		return timer;
	}

	void Message::setTimer(const float timer)
	{
		this->timer = timer;
	}

	float Message::getDelayTimer() const
	{
		return delayTimer;
	}

	/*!
	 * ディレイタイマーを設定します
	 * @param[in]	delayTimer	0ならタッチを待ちます。正の値なら指定秒数後閉じます。負の値ならば表示したままです。
	 */
	void Message::setDelayTimer(const float delayTimer)
	{
		this->delayTimer = delayTimer;
	}

	bool Message::finished() const
	{
		return clicked && length >= text->getString().length();
	}

	void Message::onSerialize(mana_stream* stream) const
	{
		super::onSerialize(stream);

		mana_stream_mark(stream);
		mana_stream_push_string(stream, const_cast<char*>(message.c_str()));
		mana_stream_push_unsigned_integer(stream, text->getID());
		mana_stream_push_float(stream, wait);
		mana_stream_push_float(stream, timer);
		mana_stream_push_float(stream, delayWait);
		mana_stream_push_float(stream, delayTimer);
		mana_stream_push_unsigned_integer(stream, length);
		mana_stream_push_unsigned_char(stream, clicked);
		mana_stream_mark(stream);
	}

	void Message::onDeserialize(mana_stream* stream)
	{
		super::onDeserialize(stream);

		mana_stream_check(stream);
		message = mana_stream_get_string_pointer(stream);
		mana_steram_seek(stream, mana_stream_get_string_length(stream) + 1);
		SAFE_ASSIGN(text, static_cast<Text*>(Component::find(mana_stream_pop_unsigned_integer(stream))));
		wait = mana_stream_pop_float(stream);
		timer = mana_stream_pop_float(stream);
		delayWait = mana_stream_pop_float(stream);
		delayTimer = mana_stream_pop_float(stream);
		length = mana_stream_pop_unsigned_integer(stream);
		clicked = mana_stream_pop_unsigned_char(stream) ? true : false;
		mana_stream_check(stream);
	}

	void Message::onUpdate(const UpdateInfomation& updateInfomation)
	{
		if(!message.empty())
		{
			if(length < message.length())
			{
				wait += updateInfomation.deltaTime;
				while(wait >= timer)
				{
					// UTF-8のマルチバイト分スキップ
					unsigned int count = 0;
					unsigned char flag = 0x80;
					const unsigned char code = message.at(length);
					while(flag & code)
					{
						flag >>= 1;
						count++;
					}
					if(count <= 0)
						count = 1;
					length += count;

					// メッセージを全て表示した？
					if(length >= message.length())
					{
						length = message.length();
						break;
					}

					wait -= timer;
				}
			}
			else
			{
				if(delayTimer > 0)
				{
					delayWait += updateInfomation.deltaTime;
					if(delayWait >= delayTimer)
						clicked = true;
				}
			}

			if(text->getString().length() != length)
			{
				text->setString(message.substr(0, length));
			}
		}

		super::onUpdate(updateInfomation);
	}
	
#if defined(DEBUG) || defined(_DEBUG) || defined(_DEBUG_) || !defined(NDEBUG)
	void Message::onDraw(const DrawRect& drawRect)
	{
		super::onDraw(drawRect);
	}
#endif

	bool Message::onTouchesEnded(const Vector2& localPosition)
	{
		if(delayTimer == 0)
		{
			if(length >= message.length())
			{
				clicked = true;
			}
			else
			{
				length = message.length();
			}
		}

		return super::onTouchesEnded(localPosition);
	}

	bool Message::compare(const Message& other) const
	{
		if(!super::compare(other))
			return false;

		if(message != other.message)
			return false;
		if(text && text->compare(*other.text) == false)
			return false;
		if(wait != other.wait)
			return false;
		if(timer != other.timer)
			return false;
		if(delayWait != other.delayWait)
			return false;
		if(delayTimer != other.delayTimer)
			return false;
		if(length != other.length)
			return false;
		if(clicked != other.clicked)
			return false;

		return true;
	}
}
