/*
 * mdr32f9qx_spl.c
 *
 *  Created on: 1 нояб. 2018 г.
 *      Author: alexrayne <alexraynepe196@gmail.com>
  ------------------------------------------------------------------------
    Copyright (c) alexrayne

   All rights reserved.
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:
   - Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   - Neither the name of ARM nor the names of its contributors may be used
     to endorse or promote products derived from this software without
     specific prior written permission.
   *
   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS AND CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. *
 */

// this header declares HAL type for other headers
#include "stm32f4xx.h"

#include "cslr.h"
#include "mcu_rcc.h"
#include <hal_device.h>



#include "mcu_nvic.h"

void NVIC_Init(const NVIC_InitTypeDef* x ){
    uint32_t grouping = NVIC_GetPriorityGrouping();
    uint32_t prio = NVIC_EncodePriority(grouping
                        , x->NVIC_IRQChannelPreemptionPriority
                        , x->NVIC_IRQChannelSubPriority
                        );
    NVIC_SetPriority((IRQn_Type)x->NVIC_IRQChannel, prio);
    if (x->NVIC_IRQChannelCmd == ENABLE){
        NVIC_EnableIRQ((IRQn_Type)x->NVIC_IRQChannel);
    }
    else
        NVIC_DisableIRQ((IRQn_Type)x->NVIC_IRQChannel);
}



#include "mcu_gpio.h"
#include "gpio.h"

void gpio_conf_mode(GPIO_TypeDef* port, unsigned pins, unsigned mode){
    uint32_t tmpreg_FUNC   = port->MODER;
    uint32_t mask_l = 0x00000003;
    mode = mode & 3;
    for (int pinid = 0; pins > 0; pinid++, pins= pins>>1, mode = mode << 2, mask_l = mask_l << 2)
    {
        if ( (pins&1) != 0){
            tmpreg_FUNC   = (tmpreg_FUNC   & ~mask_l) | mode;
        }
    }
    port->MODER = tmpreg_FUNC;
}

void gpio_conf_pin(GPIO_TypeDef* port, unsigned pin){
    RCC_EnableClock_GPIO( port );
    gpio_conf_mode(port, pin, LL_GPIO_MODE_INPUT);
}

void gpio_conf_pin(const GPIOFUNC_INIT* x){
    gpio_conf_pin(x->port, x->pins);
}

void gpio_conf_pout(GPIO_TypeDef* port, unsigned pins){
    gpio_conf_pout_oc(port, pins);
    LL_GPIO_SetPinOutputType(port, pins, LL_GPIO_OUTPUT_PUSHPULL);
}

void gpio_conf_pout_oc(GPIO_TypeDef* port, unsigned pins){
    RCC_EnableClock_GPIO( port );
    LL_GPIO_SetPinOutputType(port, pins, LL_GPIO_OUTPUT_OPENDRAIN);
    gpio_conf_mode(port, pins, LL_GPIO_MODE_OUTPUT);
}

void gpio_conf_port_style(GPIO_TypeDef* port, unsigned pins, unsigned style ){
    if ((style& psOD) == 0)
        LL_GPIO_SetPinOutputType(port, pins, LL_GPIO_OUTPUT_PUSHPULL);
    else
        LL_GPIO_SetPinOutputType(port, pins, LL_GPIO_OUTPUT_OPENDRAIN);

    unsigned mode = LL_GPIO_PULL_NO;
    if ((style& psPU) != 0)
        mode |= LL_GPIO_PULL_UP;
    if ((style& psPD) != 0)
        mode |= LL_GPIO_PULL_DOWN;

    for (int pinid = 0; pins > 0; pinid++, pins= pins>>1){
        if ( (pins&1) != 0){
            LL_GPIO_SetPinPull(port, pinid, mode);
        }
    }
}

void gpio_conf_out(const PIN_INIT* x){
    RCC_EnableClock_GPIO( x->port );
    unsigned pins = 1 << x->pin;

    if ( (x->style& psINV) != 0)
        gpio_on(x->port, pins);

    gpio_conf_port_style(x->port, pins, x->style);
    gpio_conf_mode(x->port, pins, LL_GPIO_MODE_OUTPUT);
}


void gpio_conf_in(const PIN_INIT* x){
    gpio_conf_in(x, 1 << x->pin);
}

void gpio_conf_in(const PIN_INIT* x, unsigned pins) {
    gpio_conf_pin(x->port, pins);
    gpio_conf_port_style(x->port, pins, x->style);
}

void gpio_set_func(GPIO_TypeDef* port, unsigned pinset, PORT_FUNC_TypeDef f){
    unsigned mode = LL_GPIO_MODE_ALTERNATE;
    if (f  == PORT_FUNC_ANALOG){
        mode = LL_GPIO_MODE_ANALOG;
    }
    else {
        uint32_t tmpreg_FUNC   = port->AFR[0];
        /* Form new values */
        uint32_t mask_l = 0x0000000f;
        unsigned pins   = (pinset & 0xff);
        unsigned fmode  = f;
        for (; pins > 0; pins= pins>>1){
            if ( (pins&1) != 0){
                tmpreg_FUNC   = (tmpreg_FUNC   & ~mask_l) | fmode;
            }
            mask_l <<= 4;
            fmode  <<= 4;
        }
        port->AFR[0]   = tmpreg_FUNC;

        tmpreg_FUNC   = port->AFR[1];
        /* Form new values */
        mask_l = 0x0000000f;
        fmode  = f;
        pins   = (pinset>>8) & 0xff;
        for (; pins > 0; pins= pins>>1){
            if ( (pins&1) != 0){
                tmpreg_FUNC   = (tmpreg_FUNC   & ~mask_l) | fmode;
            }
            mask_l <<= 4;
            fmode  <<= 4;
        }
        port->AFR[1]   = tmpreg_FUNC;
    }
    gpio_conf_mode(port, pinset, mode);
}

void gpio_set_func(GPIO_TypeDef* port, unsigned pinset, unsigned f){
    gpio_set_func(port, pinset, (PORT_FUNC_TypeDef)(f&PORT_FUNC_MASK) );
    gpio_conf_port_style(port, pinset, (f &~PORT_FUNC_MASK ) );
}


void gpio_conf_speed(GPIO_TypeDef* port, unsigned pins, unsigned speed){
    for (unsigned pin = 1; pins > 0; pin = pin << 1, pins = pins>>1){
        if ((pins&1) != 0)
            gpio_conf1pin_speed(port, pin, speed);
    }
}


void gpio_conf_func(const GPIOFUNC_INIT* x){
    RCC_EnableClock_GPIO( RCC_GPIO(x->port) );
    gpio_set_func(x->port, x->pins, x->func);
}

void gpio_set_func(const GPIOFUNC_INIT* x){
    gpio_set_func(x->port, x->pins, x->func);
}



void gpio_on(const PIN_INIT* x){
    HAL_ASSERT(x != NULL);
    uint32_t pins = 1 << x->pin;
    if ((x->style & psINV) == 0)
        gpio_on(x->port, pins);
    else
        gpio_off(x->port, pins);
}

void gpio_off(const PIN_INIT* x){
    HAL_ASSERT(x != NULL);
    uint32_t pins = 1 << x->pin;
    if ((x->style & psINV) == 0)
        gpio_off(x->port, pins);
    else
        gpio_on(x->port, pins);
}

uint32_t gpio_pins(const PIN_INIT* x){
    HAL_ASSERT(x != NULL);
    uint32_t pins = GPIO_ReadInputData(x->port);
    if ((x->style & psINV) != 0)
        pins = ~pins;
    pins &= 1 << x->pin;
    return pins;
}

#if 0
uint32_t gpio_styled_pins(GPIO_TypeDef* port, unsigned pin, unsigned style){
    return GPIO_STYLED_PINS(port, pin, style);
}

uint32_t gpio_styled_pouts(GPIO_TypeDef* port, unsigned pin, unsigned style){
    return GPIO_STYLED_POUTS(port, pin, style);
}
#endif



#include "mcu_system.h"
#include "mcu_isr.h"
#include <arm/arm_system.cch>

RCC_ClocksTypeDef    sys_clocks_value;

//for STM32F446
unsigned RCC_GetTimClockFreq(TIM_TypeDef* tim){
    RCC_ClocksTypeDef* sys_clocks = &sys_clocks_value;
    if (IS_APB1CLK(tim)){
        if (LL_RCC_GetAPB1Prescaler() == LL_RCC_APB1_DIV_1)
            return sys_clocks->PCLK1_Frequency;
        else
            return sys_clocks->PCLK1_Frequency*2;
    }
    else {
        if (LL_RCC_GetAPB2Prescaler() == LL_RCC_APB2_DIV_1)
            return sys_clocks->PCLK2_Frequency;
        else
            return sys_clocks->PCLK2_Frequency*2;
    }
}



//  chip specific perifery
#ifdef MDR1986VE94T

const GPIOFUNC_INIT UART1A_GPIO = PINFUNC_DEF(MDR_PORTA, (PORT_Pin_6 | PORT_Pin_7), PORT_FUNC_OVERRID);
const GPIOFUNC_INIT UART1B_GPIO = PINFUNC_DEF(MDR_PORTB, (PORT_Pin_5 | PORT_Pin_6), PORT_FUNC_ALTER);
const GPIOFUNC_INIT UART1D_GPIO = PINFUNC_DEF(MDR_PORTD, (PORT_Pin_7 | PORT_Pin_8), PORT_FUNC_OVERRID);
const GPIOFUNC_INIT UART1E_GPIO = PINFUNC_DEF(MDR_PORTE, (PORT_Pin_12 | PORT_Pin_13), PORT_FUNC_OVERRID);

const GPIOFUNC_INIT UART2D_GPIO = PINFUNC_DEF(MDR_PORTD, (PORT_Pin_0 | PORT_Pin_1), PORT_FUNC_ALTER);
const GPIOFUNC_INIT UART2F_GPIO = PINFUNC_DEF(MDR_PORTF, (PORT_Pin_0 | PORT_Pin_1), PORT_FUNC_OVERRID);

#endif
