/*
 * hal_uart.c
 *
 *  Created on: 28/04/2022
 *      Author: <alexraynepe196@gmail.com>
 * ----------------------------------------------------------------------
    Copyright (c) alexrayne

   All rights reserved.
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:
   - Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   - Neither the name of ARM nor the names of its contributors may be used
     to endorse or promote products derived from this software without
     specific prior written permission.
   *
   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS AND CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. *
 * ----------------------------------------------------------------------
 * Renesas implementation of HAL UART for proto-threads
 */

#include "hal_uart.h"
#include "r_sci_uart.h"



void UART_rsci_handle(uart_callback_args_t *arg){
    UART_Device* uart = (UART_Device*)arg->p_context;

    if ( arg->event & (UART_EVENT_RX_ANY | UART_EVENT_ERR_ANY) )
    if (uart->rx_ctx){
        process_poll( (struct process *)(uart->rx_ctx) );
    }

    if ( arg->event & UART_EVENT_TX_ANY)
    if (uart->tx_ctx){
        process_poll( (struct process *)(uart->tx_ctx) );
    }

    if (uart->iohandle) {
        int ev = arg->event | (arg->data << 8);
        (uart->iohandle)(ev, uart);
    }
}

UARTError UART_init(UART_Device* uart, uart_ctrl_t*    port){
    uart->port = port;

#if SCI_UART_CFG_DTC_SUPPORTED
    sci_uart_instance_ctrl_t * p_ctrl = (sci_uart_instance_ctrl_t *) port;
    transfer_instance_t const * drdma = p_ctrl->p_cfg->p_transfer_rx;
    if ( drdma != NULL )
        uart->rx_dma = drdma->p_cfg->p_info;
    else
        uart->rx_dma = NULL;
#else
    uart->rx_dma = NULL;
#endif

    uart->rx_ctx = NULL;
    uart->tx_ctx = NULL;

    return R_SCI_UART_CallbackSet(uart->port, (UART_rsci_handle), uart, NULL);
}


UARTError UART_read(UART_Device* uart, void* p_dest, unsigned bytes){
    uart->rx_ctx = PROCESS_CURRENT();
    return R_SCI_UART_Read(uart->port, (uint8_t*)p_dest, bytes);
}

UARTError UART_write(UART_Device* uart, const void* p_src, unsigned bytes){
    uart->tx_ctx = PROCESS_CURRENT();
    return R_SCI_UART_Write(uart->port, (const uint8_t*) p_src, bytes);
}

/// @brief check that R_SCI_UART_Write buffers empty and io sending inactive
bool UART_flush(UART_Device* uart){
    if ( !UART_tx_busy(uart) ){
        sci_uart_instance_ctrl_t* self = (sci_uart_instance_ctrl_t*)(uart->port);
        return ( ( UART_GetFlagStatus( self->p_reg ) & UART_FLAG_TXC) != 0 );
    }
    return false;
}






/// @brief check that R_SCI_UART_Write active
bool R_SCI_UART_tx_busy(uart_ctrl_t * p_api_ctrl){
    sci_uart_instance_ctrl_t* self = (sci_uart_instance_ctrl_t*)p_api_ctrl;
    return  __HAL_USART_GET_IT( self->p_reg, (USART_IT_TX | USART_IT_TC) ) != 0;
}


/// @brief receive data pointer
/// @return != NULL - pointer to receiver buffer head
///         == NULL - read not active
uint8_t*  R_SCI_UART_rx_tail(uart_ctrl_t * const p_api_ctrl){
    sci_uart_instance_ctrl_t * p_ctrl = (sci_uart_instance_ctrl_t *) p_api_ctrl;

#if SCI_UART_CFG_DTC_SUPPORTED
    transfer_instance_t const * drdma = p_ctrl->p_cfg->p_transfer_rx;
    if ( drdma != NULL ){
        transfer_info_t *   rx_dma = drdma->p_cfg->p_info;
        if (rx_dma->length > 0)
            return rx_dma->p_dest;
        return NULL;
    }
#endif

    if (p_ctrl->rx_dest_bytes > 0){
        return (uint8_t*)p_ctrl->p_rx_dest;
    }

    return NULL;
}


#if SCI_UART_CFG_DTC_SUPPORTED
uint8_t*  R_SCI_UART_rx_head(uart_ctrl_t * const p_api_ctrl){
    sci_uart_instance_ctrl_t * p_ctrl = (sci_uart_instance_ctrl_t *) p_api_ctrl;
    return (uint8_t*)(p_ctrl->p_rx_dest);
}
#endif


/// @brief receive data pointer
/// @return != NULL - pointer to receiver buffer head
///         == NULL - read not active
uint8_t*  UART_rx_tail(UART_Device* uart){
    transfer_info_t* rx_dma = (transfer_info_t*)uart->rx_dma;
    if (rx_dma != NULL){
        if (rx_dma->length > 0)
            return rx_dma->p_dest;
        return NULL;
    }
    else {
        sci_uart_instance_ctrl_t* self = (sci_uart_instance_ctrl_t*)(uart->port);
        if (self->rx_dest_bytes > 0){
            return (uint8_t*)self->p_rx_dest;
        }
        return NULL;
    }

}
