/***********************************************************************************************************************
 * Copyright [2020-2022] Renesas Electronics Corporation and/or its affiliates.  All Rights Reserved.
 *
 * This software and documentation are supplied by Renesas Electronics America Inc. and may only be used with products
 * of Renesas Electronics Corp. and its affiliates ("Renesas").  No other uses are authorized.  Renesas products are
 * sold pursuant to Renesas terms and conditions of sale.  Purchasers are solely responsible for the selection and use
 * of Renesas products and Renesas assumes no liability.  No license, express or implied, to any intellectual property
 * right is granted by Renesas. This software is protected under all applicable laws, including copyright laws. Renesas
 * reserves the right to change or discontinue this software and/or this documentation. THE SOFTWARE AND DOCUMENTATION
 * IS DELIVERED TO YOU "AS IS," AND RENESAS MAKES NO REPRESENTATIONS OR WARRANTIES, AND TO THE FULLEST EXTENT
 * PERMISSIBLE UNDER APPLICABLE LAW, DISCLAIMS ALL WARRANTIES, WHETHER EXPLICITLY OR IMPLICITLY, INCLUDING WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND NONINFRINGEMENT, WITH RESPECT TO THE SOFTWARE OR
 * DOCUMENTATION.  RENESAS SHALL HAVE NO LIABILITY ARISING OUT OF ANY SECURITY VULNERABILITY OR BREACH.  TO THE MAXIMUM
 * EXTENT PERMITTED BY LAW, IN NO EVENT WILL RENESAS BE LIABLE TO YOU IN CONNECTION WITH THE SOFTWARE OR DOCUMENTATION
 * (OR ANY PERSON OR ENTITY CLAIMING RIGHTS DERIVED FROM YOU) FOR ANY LOSS, DAMAGES, OR CLAIMS WHATSOEVER, INCLUDING,
 * WITHOUT LIMITATION, ANY DIRECT, CONSEQUENTIAL, SPECIAL, INDIRECT, PUNITIVE, OR INCIDENTAL DAMAGES; ANY LOST PROFITS,
 * OTHER ECONOMIC DAMAGE, PROPERTY DAMAGE, OR PERSONAL INJURY; AND EVEN IF RENESAS HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH LOSS, DAMAGES, CLAIMS OR COSTS.
 **********************************************************************************************************************/

#ifndef R_SCI_SPI_H
#define R_SCI_SPI_H

/***********************************************************************************************************************
 * contributor: 2022 alexraynepe196@gmail.com
 *
 * Features over renesas FSP driver version:
 *
 *  1) TXI/RXI irq are more careful detects transfer complete.
 *      Base driver version breaks transaction occasionaly
 *
 **********************************************************************************************************************/

/***********************************************************************************************************************
 * Includes
 **********************************************************************************************************************/
#include "r_spi_api.h"

/* Common macro for FSP header files. There is also a corresponding FSP_FOOTER macro at the end of this file. */
FSP_HEADER

/*****************************************************************************************************************//**
 * @ingroup SCI_SPI
 * @{
 ********************************************************************************************************************/

/***********************************************************************************************************************
 * Macro definitions
 **********************************************************************************************************************/

/***********************************************************************************************************************
 * Typedef definitions
 **********************************************************************************************************************/

/** Settings for adjusting the SPI CLK. */
typedef struct
{
    uint8_t brr;
    uint8_t cks : 2;
    uint8_t mddr;                      ///< Set to 0 to disable MDDR.
} sci_spi_div_setting_t;

/** SCI SPI extended configuration */
typedef struct st_sci_spi_extended_cfg
{
    sci_spi_div_setting_t clk_div;
} sci_spi_extended_cfg_t;

/** SPI instance control block. DO NOT INITIALIZE. */
typedef struct st_sci_spi_instance_ctrl
{
    uint32_t          open;
    spi_cfg_t const * p_cfg;
    R_SCI0_Type     * p_reg;
    uint8_t         * p_src;
    uint8_t         * p_dest;
    uint32_t          tx_count;
    uint32_t          rx_count;
    uint32_t          count;

    transfer_info_t * rx_dma;
    transfer_info_t * tx_dma;

    /* Pointer to callback and optional working memory */
    void (* p_callback)(spi_callback_args_t *);
    spi_callback_args_t * p_callback_memory;

    /* Pointer to context to be passed into callback function */
    void const * p_context;
} sci_spi_instance_ctrl_t;

/**********************************************************************************************************************
 * Exported global variables
 **********************************************************************************************************************/

/** @cond INC_HEADER_DEFS_SEC */
/** Filled in Interface API structure for this Instance. */
extern const spi_api_t g_spi_on_sci;

/** @endcond */

/**********************************************************************************************************************
 * Public Function Prototypes
 **********************************************************************************************************************/
fsp_err_t R_SCI_SPI_Open(spi_ctrl_t * p_api_ctrl, spi_cfg_t const * const p_cfg);
fsp_err_t R_SCI_SPI_Read(spi_ctrl_t * const    p_api_ctrl,
                         void                * p_dest,
                         uint32_t const        length,
                         spi_bit_width_t const bit_width);
fsp_err_t R_SCI_SPI_Write(spi_ctrl_t * const    p_api_ctrl,
                          void const          * p_src,
                          uint32_t const        length,
                          spi_bit_width_t const bit_width);
fsp_err_t R_SCI_SPI_WriteRead(spi_ctrl_t * const    p_api_ctrl,
                              void const          * p_src,
                              void                * p_dest,
                              uint32_t const        length,
                              spi_bit_width_t const bit_width);
fsp_err_t R_SCI_SPI_Close(spi_ctrl_t * const p_api_ctrl);
fsp_err_t R_SCI_SPI_CalculateBitrate(uint32_t bitrate, sci_spi_div_setting_t * sclk_div, bool use_mddr);
fsp_err_t R_SCI_SPI_CallbackSet(spi_ctrl_t * const          p_api_ctrl,
                                void (                    * p_callback)(spi_callback_args_t *),
                                void const * const          p_context,
                                spi_callback_args_t * const p_callback_memory);


/** I2C status indicators */
typedef struct st_spi_master_status
{
    bool open;                         ///< True if driver is open
    unsigned pended;                    ///< amount of bytes to transfer
} spi_master_status_t;

fsp_err_t R_SCI_SPI_StatusGet(spi_ctrl_t * const p_api_ctrl, spi_master_status_t * p_status);

/// @brief restarts current transaction if it hung for some reason.
///     it cleanup pendede irq, and generates TXI
fsp_err_t R_SCI_SPI_Continue(spi_ctrl_t * p_api_ctrl);

/*******************************************************************************************************************//**
 * @} (end ingroup SCI_SPI)
 **********************************************************************************************************************/

/* Common macro for FSP header files. There is also a corresponding FSP_HEADER macro at the top of this file. */
FSP_FOOTER



#include "r_sci_spi_cfg.h"
#if SCI_SPI_DTC_SUPPORT_ENABLE == 1
#if 1
#include <r_dtc.h>

/// use direct DTC calls on i2c drivers
#define scpi_dma_open( th )     R_DTC_Open
#define scpi_dma_close( th )    R_DTC_Close
#define scpi_dma_reset( th )    R_DTC_Reset
#define scpi_dma_recfg( th )    R_DTC_Reconfigure
#define scpi_dma_enable( th )   R_DTC_Enable
#define scpi_dma_disable( th )  R_DTC_Disable
#else
#define scpi_dma_open( th )     ((th)->api->open)
#define scpi_dma_close( th )    ((th)->api->close)
#define scpi_dma_reset( th)     ((th)->api->reset)
#define scpi_dma_recfg( th )    ((th)->api->reconfigure)
#define scpi_dma_enable( th )   ((th)->api->enable)
#define scpi_dma_disable( th )  ((th)->api->disable)
#endif
#endif




/** @brief relyable TRANSFER_COMPLETE event mark
 *
 *  @describe  current FSP driver implementation unrelyable for high speed vs slow DTC
 *          it accasionaly got TXI iqr, and comes to finish transaction.
 *
 *          this r_sci_spi implementation have solved this bug, and this macro declares about it
 *
 */
#ifndef R_SCISPI_SAFE
#define R_SCISPI_SAFE 1
#endif



#endif
