/*
 * memoryx.c
 *
 *  Created on: 13/08/2021
 *      Author: alexrayne <alexraynepe196@gmail.com>
  ------------------------------------------------------------------------
    Copyright (c) alexrayne

   All rights reserved.
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:
   - Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   - Neither the name of ARM nor the names of its contributors may be used
     to endorse or promote products derived from this software without
     specific prior written permission.
   *
   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS AND CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. *
  ------------------------------------------------------------------------
      memory operations, extends stdlib memoru.h
 */

#include <stdint.h>
#include <stdlib.h>
#include "memoryx.h"



void memseti(int* x, int value, size_t cnt){
    for( ; cnt > 0; --cnt)
        *x++ = value;
}

void memsetl(long* x, long value, size_t cnt){
    for( ; cnt > 0; --cnt)
        *x++ = value;
}



void  memcpyi( int *  __restrict y,  const int*  __restrict x, unsigned cnt){
    for( ; cnt > 0; --cnt)
        *y++ = *x++;
}

void  memcpyl( long *  __restrict y, const long*  __restrict x, unsigned cnt){
    for( ; cnt > 0; --cnt)
        *y++ = *x++;
}




/// @return == 0 same
///         > 0 len of same array, when a[len-1] > b[len-1]
///         < 0 -len of same array, when a[len-1] < b[len-1]
int  memcmp_l( const long*  __restrict pa, const long*  __restrict pb, unsigned cnt){

    for ( unsigned len = 1; len <= cnt; ++len ){
        long a = *pa++;
        long b = *pb++;
        if (a == b) continue;
        if (a > b)
            return len;
        else
            return -len;
    }

    return 0;
}

int  memcmp_b( const void * a, const void* b, unsigned cnt){
    const uint8_t* __restrict pa = (const uint8_t*)a;
    const uint8_t* __restrict pb = (const uint8_t*)b;

    unsigned len = 0;
    if (cnt > 8)
    if ( (((uintptr_t)pa | (uintptr_t)pb) & 3) == 0){
        // aligned 4 bytes
        const uint32_t*  __restrict da = (const uint32_t*)a;
        const uint32_t*  __restrict db = (const uint32_t*)b;

        unsigned wlen = cnt;
        for ( ; wlen >= 4; wlen -= 4, da++, db++ ){
            if (*da == *db) continue;
            break;
        }
        if (wlen <= 0)
            return 0;

        pa = (const uint8_t*)da;
        pb = (const uint8_t*)db;
        len = cnt - wlen;
    }

    for ( ; len < cnt; ++len ){
        long xa = *pa++;
        long xb = *pb++;
        if (xa == xb) continue;

        ++len;
        if (xa > xb)
            return len;
        else
            return -len;
    }

    return 0;
}


unsigned mismatch_b( const void * a, const void* b, unsigned bytes){
    return abs( memcmp_b(a, b, bytes) );
}



__WEAK
size_t   strnlen(const char *s, size_t limit){
    const char* z = (const char*)memchr(s, 0, limit);

    if (z != NULL)
        return ( z-s );

    return limit;
}


/// @return true - if  x -> [buf .. +size]
bool    ptr_in( const void* x, const void* buf, unsigned size){
    const uint8_t* px   = (const uint8_t*) x;
    const uint8_t* pbuf = (const uint8_t*) buf;

    return (px >= pbuf) && ( px < (pbuf+size));
}

/// @return max of a or b - accorded to max abs value
long    max_absl( long a, long b){
    if ( labs(a) >= labs(b) )
        return a;
    else
        return b;
}


char* memnchr(const void* src, int fill, size_t size){
    const char* ps = (const char*)src;
    char    x = (char)fill;

    for (; size > 0; --size)
        if (*ps == x)
            ++ps;
        else
            return (char*)ps;

    return NULL;
}


int  idxmax_u( const unsigned * x, unsigned cnt){
    int idx        = 0;
    unsigned val   = *x++;

    for (unsigned i=1 ;  i < cnt; ++i, ++x){
        if (*x > val){
            val = *x;
            idx = i;
        }
    }
    return idx;
}

/// @brief поиск позиции min/max в массиве
int  idxmax_ul( const unsigned long * x, unsigned cnt){
    int idx             = 0;
    unsigned long val   = *x++;

    for (unsigned i=1 ;  i < cnt; ++i, ++x){
        if (*x > val){
            val = *x;
            idx = i;
        }
    }
    return idx;
}

int  idxmin_u( const unsigned * x, unsigned cnt){
    int idx        = 0;
    unsigned val   = *x++;

    for (unsigned i=1 ;  i < cnt; ++i, ++x){
        if (*x < val){
            val = *x;
            idx = i;
        }
    }
    return idx;
}



int  idxmin_ul( const unsigned long * x, unsigned cnt){
    int idx             = 0;
    unsigned long val   = *x++;

    for (unsigned i=1 ;  i < cnt; ++i, ++x){
        if (*x < val){
            val = *x;
            idx = i;
        }
    }
    return idx;
}

