#
# Cookbook Name:: apache_utils
# Recipe:: mirror
#
# Copyright 2013, whitestar
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

::Chef::Recipe.send(:include, Commons::PackageUtils)

mirror_user = node['apache_utils']['mirror']['user']
base_path = node['apache_utils']['mirror']['base_path']

resources(:group => mirror_user) rescue group mirror_user do
  system true
  action :create
end

resources(:user => mirror_user) rescue user mirror_user do
  gid mirror_user
  system true
  home base_path
  shell '/bin/sh'
  password nil
  action :create
end

httpd_service = get_httpd_pkg_name

case node[:platform_family]
when 'debian'
  httpd_conf_path = "/etc/#{httpd_service}/conf.d/apache-mirror"
when 'rhel'
  httpd_conf_path = "/etc/#{httpd_service}/conf.d/apache-mirror.conf"
end

pkg = 'rsync'
resources(:package => pkg) rescue package pkg do
  action :install
end

pkg = get_cron_pkg_name
resources(:package => pkg) rescue package pkg do
  action :install
end

[
  '',
  'mirror',
  'var'
].each {|subdir|
  directory "#{base_path}/#{subdir}" do
    owner mirror_user
    group mirror_user
    mode '0755'
    recursive true
  end
}

rsync_sources = []
node['apache_utils']['mirror']['rsync_sources'].each {|source|
  local_path = "#{base_path}/mirror/#{source['url'].gsub(/^rsync:\/\//, '').gsub(/:.*$/, '')}"
  if local_path.end_with?('/') then
    local_path.chop!
  end
  directory local_path do
    owner mirror_user
    group mirror_user
    mode '0755'
    recursive true
  end

  rsync_sources.push(
    {
      'http_alias' => source['http_alias'],
      'local_path' => local_path,
      'rsync_command' => "rsync #{source['rsync_opts']} #{source['url']} #{local_path}/",
    }
  )
}

template '/usr/local/bin/apache-mirror' do
  source 'usr/local/bin/apache-mirror'
  owner 'root'
  group 'root'
  mode '0755'
  variables({
    :rsync_sources => rsync_sources
  })
end

template '/etc/cron.d/apache-mirror' do
  source 'etc/cron.d/apache-mirror'
  owner 'root'
  group 'root'
  mode '0644'
end

pkg = httpd_service
resources(:package => pkg) rescue package pkg do
  action :install
end

resources(:service => httpd_service) rescue service httpd_service do
  action [:enable, :start]
  supports :status => true, :restart => true, :reload => true
end

template httpd_conf_path do
  source 'etc/apache2/conf.d/apache-mirror'
  owner 'root'
  group 'root'
  mode '0644'
  variables({
    :rsync_sources => rsync_sources
  })
  notifies :restart, "service[#{httpd_service}]"
end

