#
# Cookbook Name:: hadoop
# Recipe:: default
#
# Copyright 2013, whitestar
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

users = {
  :hadoop => {:name => 'hadoop', :uid => 10001},
  :hdfs   => {:name => 'hdfs',   :uid => 10002},
  :mapred => {:name => 'mapred', :uid => 10003}
}

users.each {|key, user|
if key != :hadoop
  group user[:name] do
    gid user[:uid]
    members []
    action :create
  end

  user user[:name] do
    uid user[:uid]
    gid user[:uid]
    home "/home/#{user[:name]}"
    shell '/bin/sh'
    password nil
    supports :manage_home => false
  end
end
}

group users[:hadoop][:name] do
  gid users[:hadoop][:uid]
  members ['hdfs', 'mapred']
  append true
  action :create
end

user users[:hadoop][:name] do
  uid users[:hadoop][:uid]
  gid users[:hadoop][:uid]
  home "/home/#{users[:hadoop][:name]}"
  shell '/bin/sh'
  password nil
  supports :manage_home => false
end

directory '/grid/usr' do
  owner 'root'
  group 'root'
  mode '0755'
  action :create
  recursive true
end

directory '/grid/vol/0/var' do
  owner 'root'
  group 'root'
  mode '0755'
  action :create
  recursive true
end

%w{lib log run}.each {|dir|
  directory "/grid/vol/0/var/#{dir}" do
    owner 'root'
    group 'hadoop'
    mode '0775'
    action :create
    recursive true
  end
}

directory '/grid/vol/0/var/log/hdfs' do
  owner 'hdfs'
  group 'hdfs'
  mode '0755'
  action :create
  recursive true
end

hadoop_mirror_url = 'http://ftp.riken.jp/net/apache/hadoop/common'
hadoop_ver = '1.1.2'
hadoop_tarball = "hadoop-#{hadoop_ver}.tar.gz"
remote_file "#{Chef::Config[:file_cache_path]}/#{hadoop_tarball}" do
  source "#{hadoop_mirror_url}/hadoop-#{hadoop_ver}/#{hadoop_tarball}"
  action :create_if_missing
end

bash 'install_hadoop' do
  code <<-EOC
    tar xvzf "#{Chef::Config[:file_cache_path]}/#{hadoop_tarball}" -C /grid/usr
  EOC
  creates "/grid/usr/hadoop-#{hadoop_ver}"
end
=begin
link '/grid/usr/hadoop' do
  action :delete
  only_if 'test -L /grid/usr/hadoop'
end
=end
link '/grid/usr/hadoop' do
  to "/grid/usr/hadoop-#{hadoop_ver}"
end

conf_files = [
  'capacity-scheduler.xml',
  'configuration.xsl',
  'core-site.xml',
  'fair-scheduler.xml',
  'hadoop-env.sh',
  'hadoop-metrics2.properties',
  'hadoop-policy.xml',
  'hdfs-site.xml',
  'log4j.properties',
  'mapred-queue-acls.xml',
  'mapred-site.xml',
  'masters',
  'slaves',
  'ssl-client.xml.example',
  'ssl-server.xml.example'
]

conf_files.each {|conf_file|
  template "/grid/usr/hadoop-#{hadoop_ver}/conf/#{conf_file}" do
    source "conf/#{conf_file}"
    owner 'root'
    group 'root'
    mode '0644'
  end
}

# with security

template "/grid/usr/hadoop-#{hadoop_ver}/conf/taskcontroller.cfg" do
  source "conf/taskcontroller.cfg"
  owner 'root'
  group 'root'
  mode '0400'
end

package 'jsvc' do
  action :install
end

