/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx00playlist;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import javax.sound.midi.MetaMessage;
import javax.sound.midi.MidiEvent;
import javax.sound.midi.MidiMessage;
import javax.sound.midi.MidiSystem;
import javax.sound.midi.Receiver;
import javax.sound.midi.Sequence;
import javax.sound.midi.Sequencer;
import javax.sound.midi.ShortMessage;
import javax.sound.midi.Track;
import javax.sound.midi.Transmitter;
import javax.swing.DefaultListModel;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.midi.MXReceiver;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.settings.MXSetting;
import jp.synthtarou.midimixer.libs.midi.port.MXMIDIIn;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.settings.MXSettingNode;
import jp.synthtarou.midimixer.libs.settings.MXSettingTarget;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MX00Process extends MXReceiver implements MXSettingTarget {
    private static final MXDebugConsole _debug = new MXDebugConsole(MX00Process.class);

    public MX00Process() {
        _playListModel = new DefaultListModel();
        _setting = new MXSetting("PlayList");
        _setting.setTarget(this);

        _view = new MX00View(this);
    }

    public void readSettings() {
        _setting.readFile();

        if (_playListModel.size() == 0) {
            _playListModel.addElement(new FileWithId(new File("SynthTAROU000.mid")));
            _playListModel.addElement(new FileWithId(new File("SynthTAROU001.mid")));
            _playListModel.addElement(new FileWithId(new File("SynthTAROU002.mid")));
        }

        _view.settingUpdated();
    }
    
    MX00View _view;
    FileWithId _file;
    DefaultListModel _playListModel;
    MXSetting _setting;
    boolean _playAsChained;
    boolean _playAsRepeated;
    
    public FileWithId getCurrent() {
        return _file;
    }

    public synchronized boolean isSequencerPlaying() {
        if (_threadSeq != null && _threadSeq.isAlive()) {
            return true;
        }
        return false;
    }
    
    public synchronized void stopSequencer() {
        while (_threadSeq != null && _threadSeq.isAlive()) {
            try {
                _gotBreak = true;
                _threadSeq.interrupt();
            }catch(Throwable e) {
                e.printStackTrace();
            }
        }
    }
        
    public void waitSequencerDone() {
        while (_threadSeq != null && _threadSeq.isAlive()) {
            try {
                _gotBreak = true;
                Thread.sleep(3000);
            }catch(InterruptedException e) {
                e.printStackTrace();
            }
        }
    }
    
    public boolean hasGotBreak() {
        return _gotBreak;
    }

    public void openFile(FileWithId file) {
        int _noteLowest;
        int _noteHighest;

        _file = file;
        String fileName = file.toString();

        try {
            if (_sequencer != null) {
               _sequencer.close();;
               _sequencer = null;
            }

            _sequencer = MidiSystem.getSequencer(false);

            Sequence mySeq = MidiSystem.getSequence(file._file);
            Track[] trackList = mySeq.getTracks();
        
            _noteLowest = 200;
            _noteHighest = 0;

            _view.setSongName(fileName);
            Transmitter seqTransmitter = _sequencer.getTransmitter();

            _sequencer.setSequence(mySeq);
            _sequencer.open();
            
            seqTransmitter.setReceiver(new Receiver() {
                @Override
                public void send(MidiMessage midi, long timeStamp) {
                    try {
                        int port = MXMIDIIn.INTERNAL_SEQUENCER.assigned();
                        if (port >= 0) {
                            _view.UIUpdate(port, midi);
                            MXMessage message = MXMessageFactory.fromJavaMessage(port, midi);
                            MXMIDIIn.INTERNAL_SEQUENCER.processMidiIn(message);
                        }
                    }catch(Throwable e) {
                        e.printStackTrace();
                    }
                }

                @Override
                public void close() {
                }
            });

            boolean[] activeChannels = new boolean[16];
            int[] programList = new int[16];
            ArrayList<Integer> drums = new ArrayList<Integer>();

            for (Track t : trackList) {
                for (int i = 0; i < t.size(); ++ i) {
                    MidiEvent e = t.get(i);
                    MidiMessage msg = e.getMessage();

                    if (!(msg instanceof ShortMessage)) {
                        continue;
                    }

                    ShortMessage shortMsg = (ShortMessage)msg;

                    int command = shortMsg.getCommand();
                    int ch = shortMsg.getChannel();
                    int data1 = shortMsg.getData1();
                    int data2 = shortMsg.getData2();
                    
                    if (command == ShortMessage.NOTE_ON) {
                        activeChannels[ch] = true;
                        if (ch == MXStatic.DRUM_CH && data2 >= 1) {
                            drums.add(data1);
                        }else {
                            if (data1 < _noteLowest) _noteLowest = data1;
                            if (data1 > _noteHighest) _noteHighest = data1;
                        }
                    }
                    else if (command == ShortMessage.PROGRAM_CHANGE) {
                        if (programList[ch]  < 0) {
                            programList[ch] = data1;
                        }
                    }
                }
            }

            int lo2 = _noteLowest;
            lo2 = lo2 / 12;
            lo2 = lo2 * 12;

            int hi2 = lo2;

            while(hi2 < _noteHighest) {
                hi2 += 12;
            }

            int x = lo2 / 12;
            int width = hi2 / 12;

            width -= x;
            x *= 12;

            if (width <= 2) {
                if (x >= 12) {
                    x -= 12;
                    width += 1;
                }
                if (width <= 2) {
                    width += 1;
                }
            }
            
            int rows = 0;
            
            int[] drumProgs = new int[drums.size()];
            for (int id = 0; id < drums.size(); ++ id) {
                drumProgs[id] = drums.get(id);
            }
            _view.createPianoControls(lo2, width, activeChannels, programList, drumProgs);
        } catch (Exception e) {
            JOptionPane.showMessageDialog(_view, "Couldn't open as Standard MIDI File");
            e.printStackTrace();
        }
    }

    public static String[] readFileInfo(FileWithId file) {
        int _noteLowest;
        int _noteHighest;

        ArrayList<String> ret = new ArrayList();
        String fileName = file.toString();

        try {
            Sequencer sequencer = MidiSystem.getSequencer(false);

            Sequence mySeq = MidiSystem.getSequence(file._file);
            Track[] trackList = mySeq.getTracks();
        
            for (Track t : trackList) {
                for (int i = 0; i < t.size(); ++ i) {
                    MidiEvent event = t.get(i);
                    MidiMessage msg = event.getMessage();
                    if (!(msg instanceof javax.sound.midi.MetaMessage)) {
                        continue;
                    }
                    
                    MetaMessage metaMsg = (MetaMessage)msg;
                    int type = metaMsg.getType();
                    byte[] data = metaMsg.getData();
                    String text = null;
                    try {
                        text = new String(data, "ASCII");
                        text = new String(data);
                        text = new String(data, "SJIS");
                    }catch(Exception e) {
                        e.printStackTrace();
                    }
                    int number = 0;
                    
                    switch(type) {
                        case 0:
                            if (data.length >= 2) {                                
                                number = data[0] * 128 + data[1];
                                ret.add("Sequence Number : " + number);
                            }
                            break;
                        case 1:
                           ret.add("Text : " + text);
                           break;
                        case 2:
                            ret.add("Copyright : " + text);
                            break;  
                        case 3:
                            ret.add("Track Name : " + text);
                            break;  
                    };
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        String[] list = new String[ret.size()];
        ret.toArray(list);
        return list;
    }
    
    private Sequencer _sequencer = null;
    private Thread _threadSeq = null;
    private boolean _gotBreak = false;

    public synchronized void startSequencer(int start, int length) {
        _gotBreak = false;

        long microSec = _sequencer.getMicrosecondLength();
        long startSec = start * microSec / length;
        long startTiming0 = System.currentTimeMillis();
        
        double realStart = startTiming0 - (double)microSec * start / length / 1000;
        long startTiming = (long)realStart;
        
        _sequencer.setMicrosecondPosition(startSec);
        _threadSeq = new Thread(new Runnable() {
            public void run() {
                try {
                    _sequencer.start();

                    while (_sequencer.isRunning()) {
                        try {
                            long spentTiming = System.currentTimeMillis();
                            double songPercent = (spentTiming - startTiming) * 10000.0 / microSec * 1000;
                            _view.progress((int)songPercent, 10000);

                            Thread.sleep(500);
                        }catch(InterruptedException e) {
                        }catch(Throwable e) {
                            e.printStackTrace();
                            break;
                        }
                        if (_gotBreak) {
                            break;
                        }
                    }
                    _sequencer.stop();
                    MXMIDIIn.INTERNAL_SEQUENCER.allNoteOff();
                    if (_gotBreak == false) {
                        _view.whenTheMusicOver(_file);
                    }
                }catch(Throwable e) {
                    e.printStackTrace();
                }
            }
        });
        _threadSeq.start();
    }
    
    @Override
    public String getReceiverName() {
        return "SMF Player";
    }

    @Override
    public JComponent getReceiverView() {
        return _view;
    }
    @Override
    public void prepareSettingFields(jp.synthtarou.midimixer.libs.settings.MXSetting config) {
        config.register("playAsLooped");
        config.register("playAsChained");
        config.register("song[]");
    }

    @Override
    public void afterReadSettingFile(jp.synthtarou.midimixer.libs.settings.MXSetting config) {
        _playListModel.clear();

        _playAsRepeated = config.getSettingAsBoolean("playAsLooped", false);
        _playAsChained = config.getSettingAsBoolean("playAsChained", false);
        
        List<MXSettingNode> nodeList  = _setting.findByPath("song[]");
        int min = 100000;
        int max = -1;
        for (MXSettingNode node : nodeList) {
            String name = node.getName();
            try {
                int x = Integer.parseInt(name);
                if (x < min) min = x;
                if (x > max) max = x;
            }catch(NumberFormatException e) {
                _debug.printStackTrace(e);
            }
        }
        for (int x = min; x <= max; ++ x) {
            String value = config.getSetting("song[" + x + "]");
            if (value != null && value.length() > 0) {
                FileWithId file = new FileWithId(new File(value));
                _playListModel.addElement(file);
            }
        }
    }

    @Override
    public void beforeWriteSettingFile(jp.synthtarou.midimixer.libs.settings.MXSetting config) {
        config.clearValue();
        config.setSetting("playAsLooped", _playAsRepeated);
        config.setSetting("playAsChained", _playAsChained);

        for (int i = 0; i < _playListModel.getSize(); ++ i) {
            FileWithId f = (FileWithId)_playListModel.get(i);
            config.setSetting("song[" + (i + 1) + "]", f._file.getPath());
        }

        _debug.println("beforeWriteConfigFile");
    }

    @Override
    public void processMXMessage(MXMessage message) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }
}
