/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx12masterkeys;

import jp.synthtarou.midimixer.libs.swing.MXButtonUILabel;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.Window;
import java.util.Hashtable;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JScrollPane;
import javax.swing.SwingUtilities;
import jp.synthtarou.midimixer.MXMain;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.MXWrapList;
import jp.synthtarou.midimixer.libs.swing.MXSliderUIForTablet;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.MXMidiUI;
import jp.synthtarou.midimixer.libs.swing.MXPianoComponent;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MX12MasterkeysPanel extends javax.swing.JPanel {
    public static void main(String[] args) {
        MXDebugConsole.globalSwitchOn();
        Runnable myGUI = new Runnable() {
            @Override
            public void run() {
                JFrame win = new JFrame("Piano");
                win.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
                Insets ins = win.getInsets();
                win.setSize(600 + ins.left + ins.right, 500 + ins.top + ins.bottom);
                win.setLayout(new GridLayout(1, 1));
                MX12Process proc = new MX12Process();
                MX12MasterkeysPanel keys = new MX12MasterkeysPanel(proc);
                win.add(keys);
                win.setVisible(true);
                
            }
        };
        SwingUtilities.invokeLater(myGUI); 
    }

    MX12Process _process;
    MXPianoComponent _piano;
    MXWrapList<Integer> _watchPort = MXMidiUI.createPort(false);
    MXWrapList<Integer> _watchChannel = MXMidiUI.createChannel(false);
    
    int _valuePitch = -1;
    int _valueModulation = -1;
    
    /**
     * Creates new form MX00MasterKey
     */
    public MX12MasterkeysPanel(MX12Process process) {
        initComponents();
        _process = process;

        _piano = new MXPianoComponent();
        
        _piano.setNoteRange(0, 11);
        _piano.setMinimumSize(new Dimension(9 * 200, 1));
        _piano.setPreferredSize(new Dimension(9 * 200, 60));
        
        jScrollPane1.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_NEVER);
        jScrollPane1.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        jScrollPane1.setViewportView(_piano);
       
        _piano.setHandler(new MXPianoComponent.Handler() {
            public void noteOn(int note) {
                MXMessage message = MXMessageFactory.fromShortMessage(_process._mousePort, MXMidi.COMMAND_NOTEON + _process._mouseChannel, note, _process._mouseVelocity);
                _process.mouseMessage(message);
            }

            public void noteOff(int note) {
                MXMessage message = MXMessageFactory.fromShortMessage(_process._mousePort, MXMidi.COMMAND_NOTEOFF + _process._mouseChannel, note, 0);
                _process.mouseMessage(message);
            }
        });
        
        jSliderPitch.setMinimum(0);
        jSliderPitch.setMaximum(16384 - 1);
        jSliderPitch.setValue(8192);
        new MXSliderUIForTablet(jSliderPitch);
        jSliderPitch.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseReleased(java.awt.event.MouseEvent evt) {
                setPitchBend(8192);
            }
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                setPitchBend(8192);
            }
        });
        
        jSliderPitch.setMajorTickSpacing(2048);
        jSliderPitch.setMinorTickSpacing(512);
        jSliderPitch.setPaintTicks(true);
        
        Hashtable table = new Hashtable();
        table.put(0, new JLabel("0"));
        table.put(8192, new JLabel("8192"));
        table.put(16383, new JLabel("16383"));
        jSliderPitch.setLabelTable(table);
        jSliderPitch.setPaintLabels(true);
        
        jSliderModwheel.setMinimum(0);
        jSliderModwheel.setMaximum(127);
        jSliderModwheel.setValue(0);
        
        jSliderModwheel.setMajorTickSpacing(32);
        jSliderModwheel.setPaintTicks(true);
        
        Hashtable table2 = new Hashtable();
        table2.put(0, new JLabel("0"));
        table2.put(127, new JLabel("127"));
        jSliderModwheel.setLabelTable(table2);
        jSliderModwheel.setPaintLabels(true);

        final String title = "Select Output to Connect.";
        new MXSliderUIForTablet(jSliderModwheel);
        new MXButtonUILabel(jLabelEdit, "Edit", new Runnable() {
            public void run() {
                Window w = SwingUtilities.getWindowAncestor(MX12MasterkeysPanel.this);
                JFrame frame = MXMain.getMain().getMainWindow();
                if (w instanceof JFrame) {
                    frame = (JFrame)w;
                }
                MX12MasterPanelEditor.showAsDialog(frame, _process, true);
                updateViewForSettingChange();
            }
        });

        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                updateViewForSettingChange();
            }
        });
    }
    
    public void updateViewForSettingChange() {
        StringBuffer info = new StringBuffer();
        jLabelTextLayer.setText(String.valueOf(_process.getNextReceiver().getReceiverName()));
        jLabelTextPort.setText(MXMidiUI.nameOfPort(_process._mousePort));
        jLabelTextChannel.setText(String.valueOf(_process._mouseChannel + 1));
        jLabelTextMouseVelocity.setText(String.valueOf(_process._mouseVelocity));
        jLabelTextReset.setText(_process._overwriteControllerChannel ? "Overwrite" : "As Input");        
    }
    
    public void setPitchBend(int value) {
        if (SwingUtilities.isEventDispatchThread() == false) {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    setPitchBend(value);
                }
            });
            return;
        }
        if (_valuePitch != value) {
            _valuePitch = value;
            jSliderPitch.setValue(value);
         }
    }
    
    public void setModulatoinWheel(int value) {
        if (SwingUtilities.isEventDispatchThread() == false) {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    setModulatoinWheel(value);
                }
            });
            return;
        }
        if (_valueModulation != value) {
            _valueModulation = value;
            jSliderModwheel.setValue(value);
        }
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        buttonGroup1 = new javax.swing.ButtonGroup();
        buttonGroup2 = new javax.swing.ButtonGroup();
        jSliderPitch = new javax.swing.JSlider();
        jSliderModwheel = new javax.swing.JSlider();
        jScrollPane1 = new javax.swing.JScrollPane();
        jPanel1 = new javax.swing.JPanel();
        jLabelEdit = new javax.swing.JLabel();
        jLabelTextLayer = new javax.swing.JLabel();
        jLabelTextPort = new javax.swing.JLabel();
        jLabelTextChannel = new javax.swing.JLabel();
        jLabelLayer = new javax.swing.JLabel();
        jLabelPort = new javax.swing.JLabel();
        jLabelChannel = new javax.swing.JLabel();
        jLabelMouseVelocity = new javax.swing.JLabel();
        jLabelTextMouseVelocity = new javax.swing.JLabel();
        jLabelReset = new javax.swing.JLabel();
        jLabelTextReset = new javax.swing.JLabel();

        setLayout(new java.awt.GridBagLayout());

        jSliderPitch.setOrientation(javax.swing.JSlider.VERTICAL);
        jSliderPitch.setToolTipText("Pitch");
        jSliderPitch.setBorder(javax.swing.BorderFactory.createBevelBorder(javax.swing.border.BevelBorder.RAISED));
        jSliderPitch.addChangeListener(new javax.swing.event.ChangeListener() {
            public void stateChanged(javax.swing.event.ChangeEvent evt) {
                jSliderPitchStateChanged(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        add(jSliderPitch, gridBagConstraints);

        jSliderModwheel.setOrientation(javax.swing.JSlider.VERTICAL);
        jSliderModwheel.setBorder(javax.swing.BorderFactory.createBevelBorder(javax.swing.border.BevelBorder.RAISED));
        jSliderModwheel.addChangeListener(new javax.swing.event.ChangeListener() {
            public void stateChanged(javax.swing.event.ChangeEvent evt) {
                jSliderModwheelStateChanged(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        add(jSliderModwheel, gridBagConstraints);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        add(jScrollPane1, gridBagConstraints);

        jPanel1.setBorder(javax.swing.BorderFactory.createTitledBorder("Output"));
        jPanel1.setLayout(new java.awt.GridBagLayout());

        jLabelEdit.setText("Edit");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 10;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.weightx = 1.0;
        jPanel1.add(jLabelEdit, gridBagConstraints);

        jLabelTextLayer.setText("-");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        jPanel1.add(jLabelTextLayer, gridBagConstraints);

        jLabelTextPort.setText("-");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        jPanel1.add(jLabelTextPort, gridBagConstraints);

        jLabelTextChannel.setText("-");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        jPanel1.add(jLabelTextChannel, gridBagConstraints);

        jLabelLayer.setText("Receiver:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        jPanel1.add(jLabelLayer, gridBagConstraints);

        jLabelPort.setText(", Port:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 30, 0, 0);
        jPanel1.add(jLabelPort, gridBagConstraints);

        jLabelChannel.setText(", Channel:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 30, 0, 0);
        jPanel1.add(jLabelChannel, gridBagConstraints);

        jLabelMouseVelocity.setText(", Velocity (Mouse):");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 6;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.VERTICAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.SOUTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 30, 0, 0);
        jPanel1.add(jLabelMouseVelocity, gridBagConstraints);

        jLabelTextMouseVelocity.setText("-");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 7;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.VERTICAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.SOUTHWEST;
        jPanel1.add(jLabelTextMouseVelocity, gridBagConstraints);

        jLabelReset.setText(", Port/Ch(External):");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 7;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.VERTICAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.SOUTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 30, 0, 0);
        jPanel1.add(jLabelReset, gridBagConstraints);

        jLabelTextReset.setText("-");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 8;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        jPanel1.add(jLabelTextReset, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        add(jPanel1, gridBagConstraints);
    }// </editor-fold>//GEN-END:initComponents

    private void jSliderModwheelStateChanged(javax.swing.event.ChangeEvent evt) {//GEN-FIRST:event_jSliderModwheelStateChanged
        int value = jSliderModwheel.getValue();
        _valueModulation = value;
        MXMessage msg = MXMessageFactory.fromShortMessage(_process._mousePort, MXMidi.COMMAND_CONTROLCHANGE + _process._mouseChannel, MXMidi.DATA1_CCMODULATION, 0);
        msg.setValue(value);
        _process.mouseMessage(msg);
    }//GEN-LAST:event_jSliderModwheelStateChanged

    private void jSliderPitchStateChanged(javax.swing.event.ChangeEvent evt) {//GEN-FIRST:event_jSliderPitchStateChanged
        int value = jSliderPitch.getValue();
        _valuePitch = value;
        MXMessage msg = MXMessageFactory.fromShortMessage(_process._mousePort, MXMidi.COMMAND_PITCHWHEEL + _process._mouseChannel, 0, 0);
        msg.setValue(value);
        _process.mouseMessage(msg);
    }//GEN-LAST:event_jSliderPitchStateChanged

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.ButtonGroup buttonGroup1;
    private javax.swing.ButtonGroup buttonGroup2;
    private javax.swing.JLabel jLabelChannel;
    private javax.swing.JLabel jLabelEdit;
    private javax.swing.JLabel jLabelLayer;
    private javax.swing.JLabel jLabelMouseVelocity;
    private javax.swing.JLabel jLabelPort;
    private javax.swing.JLabel jLabelReset;
    private javax.swing.JLabel jLabelTextChannel;
    private javax.swing.JLabel jLabelTextLayer;
    private javax.swing.JLabel jLabelTextMouseVelocity;
    private javax.swing.JLabel jLabelTextPort;
    private javax.swing.JLabel jLabelTextReset;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JSlider jSliderModwheel;
    private javax.swing.JSlider jSliderPitch;
    // End of variables declaration//GEN-END:variables
}
