/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.libs.console;

import java.util.ArrayList;
import javax.sound.midi.InvalidMidiDataException;
import javax.sound.midi.MidiMessage;
import javax.sound.midi.ShortMessage;
import jp.synthtarou.midimixer.libs.MXUtil;
import jp.synthtarou.midimixer.libs.MXWrapList;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.MXUtilMidi;
import jp.synthtarou.midimixer.libs.midi.port.MXMIDIOut;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MXConsoleMidiRenderer extends MXConsoleRenderer {
    public static final int MODE_SHORT = 1;
    public static final int MODE_BINARY = 2;
    public static final int MODE_DTEXT = 3;
    public static final int MODE_FORMAT = 4;
    
    public synchronized static MXWrapList<Integer> getModeDataModel() {
        MXWrapList<Integer> model = new MXWrapList();
        model.addNameAndValue("Binary", MODE_BINARY);
        model.addNameAndValue("Short", MODE_SHORT);
        model.addNameAndValue("D-Text", MODE_DTEXT);
        model.addNameAndValue("Binary + Short", MODE_FORMAT);
        return model;
    }
    
    public synchronized void setMode(int mode) {
        _mode = mode;
    }

    String _title;
    int _mode = MODE_SHORT;
    
    public MXConsoleMidiRenderer(String title) {
        _title = title;
    }

    public String format(Object value) {
        String text = "";
        MXMessage message = (MXMessage)value;
        ArrayList darray = message.toDArray();
        String dtext = darray.toString();
        
        if (darray.contains("#GL") || darray.contains("#GH")) {
            dtext = dtext + " gate=" + message.getGate();
        }
        if (darray.contains("#VL") || darray.contains("#VH")) {
            dtext = dtext + " value=" + message.getValue();
        }
        
        String portText = "[" + MXUtilMidi.nameOfPort(message.getPort());
        if (message.isMessageTypeChannel()) {
            portText = portText + message.getChannel();
        }
        portText += "] ";
        
        String title = _title + " ";
        String shortText = message.toString();
        String bit = "";

        /*
        if (message.getDataentryType() != MXMidi.DATAENTRY_TYPE_NONE) {
            if (message.getDataentryType() == MXMidi.DATAENTRY_TYPE_RPN) {
                bit += "RPN [" + MXUtil.toHexFF(message.getDataentryMSB()) + ":" + MXUtil.toHexFF(message.getDataentryLSB()) + "]";
            }else if (message.getDataentryType() == MXMidi.DATAENTRY_TYPE_NRPN) {
                bit += "NRPN [" + MXUtil.toHexFF(message.getDataentryMSB()) + ":" + MXUtil.toHexFF(message.getDataentryLSB()) + "]";
            }
        }*/
        if (message.isValue14bit()) {
            bit += " - 14bit";
        }

        ArrayList<String> dump = new ArrayList();

        boolean isDataentry = false;
        if (message.getCommand() == MXMidi.COMMAND_CONTROLCHANGE) {
            switch (message.getGate()) {
                case MXMidi.DATA1_CC_DATAENTRY:
                case MXMidi.DATA1_CC_DATAINC:
                case MXMidi.DATA1_CC_DATADEC:
                    isDataentry = true;
                    break;
            }
        }
        try {
            if (isDataentry) {
                MidiMessage msg1 = MXMessageFactory.toJavaDataentry1MSB(message);
                MidiMessage msg2 = MXMessageFactory.toJavaDataentry2LSB(message);
                MidiMessage data1 = MXMessageFactory.toDataentryValueMSB(message);
                MidiMessage data2 = MXMessageFactory.toDataentryValueLSB(message);
                synchronized(MXMIDIOut.class) {
                    if (msg1 != null && msg2 != null) {
                        dump.add(MXUtil.dumpHexFF(msg1.getMessage()));
                        dump.add(MXUtil.dumpHexFF(msg2.getMessage()));
                    }
                    dump.add(MXUtil.dumpHexFF(data1.getMessage()));
                    if (data2 != null) {
                        dump.add(MXUtil.dumpHexFF(data2.getMessage()));
                    }
                }
            }else if ((message.getCommand() == MXMidi.COMMAND_CONTROLCHANGE) && message.isValue14bit()) {
                MidiMessage data1 = new ShortMessage(MXMidi.COMMAND_CONTROLCHANGE + message.getChannel(), message.getGate(), (message.getValue() >> 7) & 0x7f);
                MidiMessage data2 = new ShortMessage(MXMidi.COMMAND_CONTROLCHANGE + message.getChannel(), message.getGate() + 0x20, (message.getValue()) & 0x7f);
                synchronized(MXMIDIOut.class) {
                    dump.add(MXUtil.dumpHexFF(data1.getMessage()));
                    dump.add(MXUtil.dumpHexFF(data2.getMessage()));
                }
            }else {
                MidiMessage msg = MXMessageFactory.toJavaMessage(message);
                if (msg != null) {
                    dump.add(MXUtil.dumpHexFF(msg.getMessage()));
                }
            }
        }catch(InvalidMidiDataException e) {
            e.printStackTrace();;
        }
        
        switch(_mode) {
            case MODE_SHORT:
                text = title + shortText + bit;
                break;
            case MODE_BINARY:
                text = title + portText + " " + dump.toString() + " " + bit;
                break;
            case MODE_DTEXT:
                text = title + portText + dtext +bit;
                break;
            case MODE_FORMAT:
                text = title + portText + " '" + dump.toString()+ "' = " + shortText + bit; 
                break;
            default:
                text = "ERROR";
                break;
        }

        return text;
     }
 }
