/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx10input;

import javax.swing.JComponent;
import javax.swing.JPanel;
import jp.synthtarou.midimixer.MXMidiLogger;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.MXReceiver;
import jp.synthtarou.midimixer.libs.settings.MXSetting;
import jp.synthtarou.midimixer.libs.settings.MXSettingTarget;
import jp.synthtarou.midimixer.mx12masterkeys.MX12Process;
import jp.synthtarou.midimixer.mx20patchbay.MX20Process;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MX10Process extends MXReceiver implements MXSettingTarget {
    MX10Data _data;
    MX10View _view;

    MX20Process _patchBay;
    MX12Process _masterKey;
    MXMidiLogger _logger;
    MXSetting _setting;

    public MX10Process() {
        _data = new MX10Data();
        _patchBay = new MX20Process(true);
        _view = new MX10View(this);
        _logger = new MXMidiLogger("In");
        _view.setLogger(_logger);
        _setting = new MXSetting("InputSkip");
        _setting.setTarget(this);
        _patchBay.setNextReceiver(new MXReceiver()  {
            @Override
            public String getReceiverName() {
                return "dummy";
            }

            @Override
            public JComponent getReceiverView() {
                return null;
            }

            @Override
            public void processMXMessage(MXMessage message) {
                if (MX10Process.this.isUsingThisRecipe()) {
                    if (_data.isMessageToSkip(message)) {
                        return;
                    }
                }
                if (isUseLogging()) {
                    _logger.processMXMessage(message);
                }
                MX10Process.this.sendToNext(message);
            }
            
        });
    }
    
    public void readSettings() {
        _setting.readFile();
        _patchBay.readSettings();
    }
    
    public void setMasterKeyProcess(MX12Process master) {
        _masterKey = master;
    }

    public void setUseMesssageFilter(boolean log) {
        setUsingThisRecipe(log);
    }
    
    public boolean isUseMessageFilter() {
        return isUsingThisRecipe();
    }
    
    public void setUsePortFowarding(boolean pf) {
        _patchBay.setUsingThisRecipe(pf);
    }
    
    public boolean isUsePortFowarding() {
        if (_patchBay == null) {
            return true;
        }
        return _patchBay.isUsingThisRecipe();
    }
    
    public void setUseLogging(boolean pf) {
        _logger.setUsingThisRecipe(pf);
    }
    
    public boolean isUseLogging() {
        if (_logger == null) {
            return true;
        }
        return _logger.isUsingThisRecipe();
    }
    
    @Override
    public void processMXMessage(MXMessage message) {
        _patchBay.processMXMessage(message);
    }

    @Override
    public String getReceiverName() {
        return "Input Receiver";
    }

    @Override
    public JPanel getReceiverView() {
        return _view;
    }

    @Override
    public void prepareSettingFields(MXSetting setting) {
        String prefix = "Setting[].";
        for (String text : MX10Data.typeNames) {
            setting.register(prefix + text);
        }
    }

    @Override
    public void afterReadSettingFile(MXSetting setting) {
        for (int port = 0; port < MXStatic.TOTAL_PORT_COUNT; ++ port) {
            String prefix = "Setting[" + port + "].";
            StringBuffer str = new StringBuffer();
            for (int j = 0; j <_data.countOfTypes(); ++ j) {
                String name = _data.typeNames[j];
                boolean set = setting.getSettingAsBoolean(prefix + name, false);
                _data.setSkip(port, j, set);
            }
        }
        _view.resetTableModel();
    }

    @Override
    public void beforeWriteSettingFile(MXSetting setting) {
        for (int port = 0; port < MXStatic.TOTAL_PORT_COUNT; ++ port) {
            String prefix = "Setting[" + port + "].";
            StringBuffer str = new StringBuffer();
            for (int j = 0; j <_data.countOfTypes(); ++ j) {
                boolean set = _data.isSkip(port, j);
                String name = _data.typeNames[j];
                setting.setSetting(prefix + name, set);
            }
        }
    }
}
