/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx12masterkeys;

import javax.swing.JComponent;
import jp.synthtarou.midimixer.MXMain;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.midi.MXReceiver;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.MXNoteOffWatcher;
import jp.synthtarou.midimixer.libs.settings.MXSetting;
import jp.synthtarou.midimixer.libs.settings.MXSettingTarget;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MX12Process extends MXReceiver implements MXSettingTarget {


    private static final MXDebugConsole _debug = new MXDebugConsole(MX12Process.class);

    private MXSetting _setting;
    MX12MasterkeysPanel _view;
    MXNoteOffWatcher _noteOff;
    
    private int _mousePort = 0;
    private int _mouseChannel = 0;
    private boolean _overwriteInputChannel;
    private int _mouseVelocity = 100;

    boolean _construction;
    
    private boolean _acceptThisPageSignal;
    private boolean _acceptInputPanelSignal;
    
    public MX12Process() {
        _construction = true;
        _setting = new MXSetting("MasterKey");
        _setting.setTarget(this);
        _noteOff = new MXNoteOffWatcher();
        _overwriteInputChannel = false;
        _construction = false;
    }

    public void readSettings() {
        _setting.readFile();
        if (_view != null) {
            _view.updateViewForSettingChange();
        }
    }
   
    @Override
    public void prepareSettingFields(MXSetting setting) {
        setting.register("outputReceiver");
        setting.register("outputPort");
        setting.register("outputChannel");
        setting.register("overwriteControllerChannel");
        setting.register("outputVelocity");
        setting.register("acceptThisPanelSignal");
        setting.register("acceptInputPanelSignal");
    }

    @Override
    public void afterReadSettingFile(MXSetting setting) {
        String receiverName = setting.getSetting("outputReceiver");
        if (receiverName != null) {
            int x = MXMain.getMain().getReceiverList().indexOfName(receiverName);
            if (x >= 0) {
                setNextReceiver(MXMain.getMain().getReceiverList().get(x).value);
            }
        }
        setOverwriteInputChannel(setting.getSettingAsInt("overwriteControllerChannel", 0) != 0);        
        setMousePort(setting.getSettingAsInt("outputPort", 0));
        setMouseChannel(setting.getSettingAsInt("outputChannel", 0));
        setMouseVelocity(setting.getSettingAsInt("outputVelocity", 100));
        _acceptThisPageSignal = setting.getSettingAsBoolean("acceptThisPanelSignal", true);
        _acceptInputPanelSignal = setting.getSettingAsBoolean("acceptInputPanelSignal", true);
    }

    @Override
    public void beforeWriteSettingFile(MXSetting setting) {
        setting.setSetting("outputReceiver", getNextReceiver().getReceiverName());
        setting.setSetting("outputPort", getMousePort());
        setting.setSetting("outputChannel", getMouseChannel());
        setting.setSetting("overwriteControllerChannel", isOverwriteInputChannel());
        setting.setSetting("outputVelocity", getMouseVelocity());
        setting.setSetting("acceptThisPanelSignal", _acceptThisPageSignal);
        setting.setSetting("acceptInputPanelSignal", _acceptInputPanelSignal);
    }
   
    public class MyNoteOffHandler implements MXNoteOffWatcher.Handler {
        MXReceiver _receiver;
        
        public MyNoteOffHandler(MXReceiver receiver) {
            _receiver = receiver;
        }

        @Override
        public void onNoteOffEvent(MXMessage target) {
            MXMessage makeOff = MXMessageFactory.fromShortMessage(
                    target.getPort(), 
                    MXMidi.COMMAND_NOTEOFF | target.getChannel(), 
                    target.getNoteNumberFromBytes(),
                    0);
            _receiver.processMXMessage(makeOff);
            if (_view != null) {
                _view._piano.noteOff(target.getNoteNumberFromBytes());
            }
        }
    }

    @Override
    public void processMXMessage(MXMessage message) {
        if (isUsingThisRecipe() == false) { sendToNext(message); return; }
        
        if (message.isMessageTypeChannel()) {
            int port = message.getPort();
            int ch = message.getChannel();
            int command = message.getCommand();
            int data1 = message.getData1FromBytes();
            int data2 = message.getData2FromBytes();

            if (command == MXMidi.COMMAND_NOTEON && data2 == 0) {
                command = MXMidi.COMMAND_NOTEOFF;
            }

            if (command == MXMidi.COMMAND_NOTEOFF) {
               if (_noteOff.notifyNoteOffEvent(port, ch, data1, "@3")) {
                    //done
                    return;
                }
            }

            MXMessage newMessage = null;
            if (isOverwriteInputChannel()) {
                newMessage = MXMessageFactory.fromClone(message);
                newMessage.setPort(getMousePort());
                if (newMessage.isMessageTypeChannel()) {
                    newMessage.setChannel(getMouseChannel());
                }
            }

            if (command == MXMidi.COMMAND_NOTEON) {
                _view._piano.noteOn(data1);
                if (newMessage != null) {
                    _noteOff.addListener(message, newMessage, new MyNoteOffHandler(getNextReceiver()));
                }else {
                    _noteOff.addListener(message, message, new MyNoteOffHandler(getNextReceiver()));
                }
            }else if (command == MXMidi.COMMAND_CONTROLCHANGE && data1 == MXMidi.DATA1_CC_PEDAL) {
                _view._piano.sustain(data2);
            }else if (command == MXMidi.COMMAND_PITCHWHEEL) {
                _view.setPitchBend(message.getValue());
            }else if (command == MXMidi.COMMAND_CONTROLCHANGE && data1 == MXMidi.DATA1_CCMODULATION) {
                _view.setModulatoinWheel(message.getValue());
            }
            if (newMessage != null) {
                sendToNext(newMessage);
            }else {
                sendToNext(message);
            }
            return;
        }
        sendToNext(message);
    }

    public void mouseMessage(MXMessage message) {
        if (_construction) {
            return;
        }
        sendToNext(message);
    }

    /**
     * @return the acceptThisPageSignal
     */
    public boolean isAcceptThisPageSignal() {
        return _acceptThisPageSignal;
    }

    /**
     * @param acceptThisPageSignal the acceptThisPageSignal to set
     */
    public void setAcceptThisPageSignal(boolean acceptThisPageSignal) {
        this._acceptThisPageSignal = acceptThisPageSignal;
    }

    /**
     * @return the acceptInputPanelSignal
     */
    public boolean isAcceptInputPanelSignal() {
        return _acceptInputPanelSignal;
    }

    /**
     * @param acceptInputPanelSignal the acceptInputPanelSignal to set
     */
    public void setAcceptInputPanelSignal(boolean acceptInputPanelSignal) {
        this._acceptInputPanelSignal = acceptInputPanelSignal;
    }

    @Override
    public String getReceiverName() {
        return "Master Keyboard";
    }

    @Override
    public JComponent getReceiverView() {
        return _view;
    }


    /**
     * @return the _mousePort
     */
    public int getMousePort() {
        return _mousePort;
    }

    /**
     * @param _mousePort the _mousePort to set
     */
    public void setMousePort(int _mousePort) {
        this._mousePort = _mousePort;
    }

    /**
     * @return the _mouseChannel
     */
    public int getMouseChannel() {
        return _mouseChannel;
    }

    /**
     * @param _mouseChannel the _mouseChannel to set
     */
    public void setMouseChannel(int _mouseChannel) {
        this._mouseChannel = _mouseChannel;
    }

    /**
     * @return the overwriteInputChannel
     */
    public boolean isOverwriteInputChannel() {
        return _overwriteInputChannel;
    }

    /**
     * @param overwriteInputChannel the overwriteInputChannel to set
     */
    public void setOverwriteInputChannel(boolean overwriteInputChannel) {
        this._overwriteInputChannel = overwriteInputChannel;
    }

    /**
     * @return the _mouseVelocity
     */
    public int getMouseVelocity() {
        return _mouseVelocity;
    }

    /**
     * @param _mouseVelocity the _mouseVelocity to set
     */
    public void setMouseVelocity(int _mouseVelocity) {
        this._mouseVelocity = _mouseVelocity;
    }
    
    public MX12MasterkeysPanel createView() {
        if (_view != null) {
            return _view;
        }
        _view = new MX12MasterkeysPanel(this);
        return _view;
    }
}
