/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx30controller;

import java.awt.event.MouseListener;
import java.util.ArrayList;
import javax.swing.JLabel;
import javax.swing.SwingUtilities;
import jp.synthtarou.midimixer.libs.MXUtil;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.swing.MXSliderUIForTablet;
import jp.synthtarou.midimixer.libs.swing.MXFocusAble;
import jp.synthtarou.midimixer.libs.swing.MXFocusGroupElement;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MGSlider extends javax.swing.JPanel implements MXFocusAble {
    private static final MXDebugConsole _debug = new MXDebugConsole(MGSlider.class);

    MX32MixerProcess _process;
    int _row, _column;

    public MGStatus getStatus() {
        if (_process == null) return null;
        return _process._data.getSliderStatus(_row, _column);
    }
    
    public void setStatus(MGStatus status) {
        _process._data.setSliderStatus(_row, _column, status);
    }

    boolean _disconnectMoment = false;
    
    public MGSlider(MX32MixerProcess process, int row, int column) {
        _row = row;
        _column = column;
        _process = process;
        initComponents();

        updateUI();

        new MXSliderUIForTablet(jSliderValue);
    }

    public void updateUI() {
        super.updateUI();
        MGStatus status = getStatus();
        if (status != null) {
            jSliderValue.setMinimum(status.getRangeMin());
            jSliderValue.setMaximum(status.getRangeMax());
            jSliderValue.setInverted(status.isUiValueInvert());
            jSliderValue.setPaintLabels(true);
            jSliderValue.setValue(status.getValue());
            status.fixRangedValue();
            if (status.getName() == null || status.getName().length() == 0) {
                MXMessage message = status.toMXMessage();
                if (message == null) {
                    jLabel1.setText("??");
                }else {
                    jLabel1.setText(message.toShortString());
                }
            }else {
                jLabel1.setText(status.getName());
            }
            focusStatusChanged(false);
        }
    }

    ArrayList<MouseListener> backupedListener = new ArrayList();
    
    public void setValueChangeable(boolean usable) {
        if (usable == false) {            
            for (MouseListener l : jSliderValue.getMouseListeners()) {
                if (l instanceof MXFocusGroupElement.Listen) {
                    continue;
                }else {
                    backupedListener.add(l);
                }
            }
            for (MouseListener l : backupedListener) {
                jSliderValue.removeMouseListener(l);
            }
        }else {
            for (MouseListener l : backupedListener) {
                jSliderValue.addMouseListener(l);
            }
            backupedListener.clear();
        }
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        jSliderValue = new javax.swing.JSlider();
        jLabel1 = new javax.swing.JLabel();

        setLayout(new java.awt.GridBagLayout());

        jSliderValue.setOrientation(javax.swing.JSlider.VERTICAL);
        jSliderValue.addChangeListener(new javax.swing.event.ChangeListener() {
            public void stateChanged(javax.swing.event.ChangeEvent evt) {
                jSliderValueStateChanged(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        add(jSliderValue, gridBagConstraints);

        jLabel1.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        jLabel1.setText("-");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        add(jLabel1, gridBagConstraints);
    }// </editor-fold>//GEN-END:initComponents

    private void jSliderValueStateChanged(javax.swing.event.ChangeEvent evt) {//GEN-FIRST:event_jSliderValueStateChanged
        int newValue = jSliderValue.getValue();
        if (_updating == newValue) {
            return;
        }
        _process.catchedValue(getStatus(), newValue, null);
    }//GEN-LAST:event_jSliderValueStateChanged

    int _updating = -1;

    public void updateUIOnly(final int newValue) {
        if (SwingUtilities.isEventDispatchThread() == false) {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    updateUIOnly(newValue);
                }
            });
            return;
        }
        
        _updating = newValue;

        if (jSliderValue.getValue() != newValue) {
            jSliderValue.setValue(newValue);
        }

        _updating = -1;
    }

    public JLabel labelFor(int num, int max) {
        String name = "";
        if (max >= 256) {
            int msb = num / 128;
            int cut = msb * 128;
            int lsb = num - cut;
            name = MXUtil.toHexFF(msb) + ":" + MXUtil.toHexFF(lsb);
        }else {
            name = MXUtil.toHexFF(num);
        }
        JLabel label = new JLabel(name);
        //label.setFont(new Font(Font.SANS_SERIF, Font.ITALIC, 10));
        return label;
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JLabel jLabel1;
    private javax.swing.JSlider jSliderValue;
    // End of variables declaration//GEN-END:variables

    @Override
    public void focusStatusChanged(boolean flag) {
        if (flag) {
            _process._parent.showTextForFocus(MGStatus.TYPE_SLIDER,  _process._port, _row, _column);
        }
    }

    @Override
    public void focusedMousePressed() {
        if (_process._parent._editingControl) {
            _process._parent.enterEditMode(false);
            focusStatusChanged(false);
            editContoller();
        }
    }

    @Override
    public void focusChangingValue() {
        _process._parent.showTextForFocus(MGStatus.TYPE_SLIDER,  _process._port, _row, _column);
    }

    public void increment() {
        MGStatus status = getStatus();
        int newValue = status.getValue() + 1;
        if (status.isUiValueInvert()) {
            newValue = status.getValue() - 1;
        }
        if (newValue > status.getRangeMax()) {
            newValue = status.getRangeMax();
        }
        if (newValue < status.getRangeMin()) {
            newValue = status.getRangeMin();
        }
        if (newValue != status.getValue()) {
            _process.catchedValue(status, newValue, null);
        }
    }
    
    public void decriment() {
        MGStatus status = getStatus();
        int newValue = status.getValue() - 1;
        if (status.isUiValueInvert()) {
            newValue = status.getValue() + 1;
        }
        if (newValue > status.getRangeMax()) {
            newValue = status.getRangeMax();
        }
        if (newValue < status.getRangeMin()) {
            newValue = status.getRangeMin();
        }
        if (newValue != status.getValue()) {
            _process.catchedValue(status, newValue, null);
        }
    }
    
    public void beHomePosition() {
        final MGStatus status = getStatus();
        final int current = status.getValue();
        final int value = status.getValueHome();
        Thread t = new Thread(new Runnable() {
            public void run() {
                try {
                    for (int i = 0; i < 5; ++ i) {
                        int x = current * (5 - i) + value * i;
                        x /= 5;
                        Thread.sleep(70);
                        status.setValue(x);
                        updateUIOnly(x);
                        if (x == value) { 
                            break;
                        }
                    }
                }catch(Exception e) {
                    e.printStackTrace();
                }finally {
                    status.setValue(value);
                    updateUIOnly(value);
                }
            }
        });
        t.start();
    }

    public void editContoller() {
        _process._parent.enterEditMode(false);
        MGStatus status = (MGStatus)getStatus().clone();
        MGStatusConfig config = new MGStatusConfig(_process, status);
        MXUtil.showAsDialog(this, config, "Enter Edit Slider {row:" + _row + ", column:" + _column + "}");
        if (config._okOption) {
            setStatus(config._status);
            jLabel1.setText(config._status.getName());
            _process.notifyCacheBroken();
            updateUI();
        }
    }
}
