/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx40layer;

import java.util.ArrayList;
import java.util.LinkedList;
import javax.swing.JComponent;
import javax.swing.SwingUtilities;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableModel;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.MXUtil;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.MXTimer;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.MXUtilMidi;
import jp.synthtarou.midimixer.libs.midi.MXNoteOffWatcher;
import jp.synthtarou.midimixer.libs.midi.MXReceiver;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MXChannelInfoReceiver extends MXReceiver implements TableModel {
    private static final MXDebugConsole _debug = new MXDebugConsole(MXChannelInfoReceiver.class);

    private ArrayList<MXChannelInfo[]> _element;
    public MXNoteOffWatcher _noteoff;

    public MXChannelInfoReceiver() {
        _noteoff = new MXNoteOffWatcher();
        _element = new ArrayList();
        for (int port = 0; port < MXStatic.TOTAL_PORT_COUNT; ++ port) {
            MXChannelInfo[] chinfo = new MXChannelInfo[16];
            for (int i = 0; i < 16; ++ i) {
                chinfo[i] = new MXChannelInfo();
            }
            _element.add(chinfo);
        }
    }
    
    public MXChannelInfo getChannelInfo(int port, int ch) {
        return _element.get(port)[ch];
    }

    public boolean tryPickup(MXMessage message) {
        boolean flag = false;
        if (message.isMessageTypeChannel()) {
            int command = message.getCommand();        
            int port = message.getPort();
            int channel = message.getChannel();
            MXChannelInfo info = getChannelInfo(port, channel);
            if (command == MXMidi.COMMAND_PROGRAMCHANGE) {
                if (message.getGate() >= 0) {
                    info._havingProgram = true;
                    info._program = message.getGate();
                    flag = true;
                }else {
                    info._havingProgram = false;
                    info._program = -1;
                    return true;
                }
                return true;
            }else if (command == MXMidi.COMMAND_CONTROLCHANGE) {
                final int cctype = message.getGate();
                final int value = message.getValue();

                switch(cctype) {
                    case MXMidi.DATA1_CCBANKSELECT:
                        info._bankMSB = value;
                        info._havingBank = true;
                        flag = true;
                        break;
                    case MXMidi.DATA1_CCBANKSELECT + 0x20:
                        info._bankLSB = value;
                        info._havingBank = true;
                        flag = true;
                        break;
                    case MXMidi.DATA1_CCPANPOT:
                        info._infoPan = value;
                        flag = true;
                        break;
                    case MXMidi.DATA1_CC_CHANNEL_VOLUME:
                        info._infoVolume = value;
                        flag = true;
                        break;
                    case MXMidi.DATA1_CC_EXPRESSION:
                        info._infoExpression = value;
                        flag = true;
                        break;
                    case MXMidi.DATA1_CC_DATAENTRY:
                        info._dataentryValue = value;
                        flag = true;
                        break;
                    case MXMidi.DATA1_NRPN_LSB:
                        info._dataentryType = MXMidi.DATAENTRY_TYPE_NRPN;
                        info._dataentryLSB = value;
                        flag = true;
                        break;
                    case MXMidi.DATA1_NRPN_MSB:
                        info._dataentryType = MXMidi.DATAENTRY_TYPE_NRPN;
                        info._dataentryMSB = value;
                        flag = true;
                        break;
                   case MXMidi.DATA1_RPN_LSB:
                        info._dataentryType = MXMidi.DATAENTRY_TYPE_RPN;
                        info._dataentryLSB = value;
                        flag = true;
                        break;
                    case MXMidi.DATA1_RPN_MSB:
                        info._dataentryType = MXMidi.DATAENTRY_TYPE_RPN;
                        info._dataentryMSB = value;
                        flag = true;
                        break;
                }
            }
        }

        if (flag) {
            invokeListener();
        }
        return flag;
    }
    
    public MXMessage incProg(int port, int channel) {
        MXChannelInfo e = getChannelInfo(port, channel);
        if (e._program < 127) {
            e._program ++;
            invokeListener();
        }else {
            return null;
        }
        return MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_PROGRAMCHANGE + channel, e._program, 0);
    }

    public MXMessage decProg(int port, int channel) {
        MXChannelInfo e = getChannelInfo(port, channel);
        if (e._program > 0) {
            e._program --;            
            invokeListener();
        }else {
            return null;
        }
        return MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_PROGRAMCHANGE + channel, e._program, 0);
    }
    
    MXReceiver _nextReciever;

    public String getReceiverName() {
        return "Program List (DEV)";
    }

    public JComponent getReceiverView() {
        return null;
    }

    public void processMXMessage(MXMessage message) {
        //if (usingThis() == false) { sendToNext(message); return; }
        tryPickup(message);
        //sendToNext(message);
    }

    String[] tableColumns =  {
        "Port/Ch", "Bank", "Prog", "Vol",  "Exp", "Pan", "Data"
    };
    @Override
    public int getRowCount() {
        return _element.size() * 16;
    }

    @Override
    public int getColumnCount() {
        return tableColumns.length;
    }

    @Override
    public String getColumnName(int i) {
        return tableColumns[i];
    }

    @Override
    public Class<?> getColumnClass(int i) {
        return String.class;
    }

    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        return false;
    }

    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        int port = rowIndex / 16;
        int channel = rowIndex-  (port * 16);
        MXChannelInfo info = getChannelInfo(port, channel);

        switch(columnIndex) {
            case 0:
                return MXUtilMidi.nameOfPort(port) + (channel+1);
            case 1:
                if (info._havingBank) {
                    return MXUtil.toHexFF(info._bankMSB) + ":" + MXUtil.toHexFF(info._bankLSB);
                }
                break;
            case 2:
                if (info._havingProgram) {
                    return "" + info._program;
                }
                return "";
            case 3:
                return "" + info._infoVolume;
            case 4:
                return "" + info._infoExpression;
            case 5:
                return "" + info._infoPan;
            case 6:
                if (info._dataentryType == MXMidi.DATAENTRY_TYPE_RPN) {
                    return "R(" + MXUtil.toHexFF(info._dataentryMSB) + ":" + MXUtil.toHexFF(info._dataentryLSB) + ")=" + info._dataentryValue;
                }else if (info._dataentryType == MXMidi.DATAENTRY_TYPE_NRPN) {
                    return "N(" + MXUtil.toHexFF(info._dataentryMSB) + ":" + MXUtil.toHexFF(info._dataentryLSB) + ")=" + info._dataentryValue;
                }
                break;
        }
        return "-";
    }

    @Override
    public void setValueAt(Object o, int i, int i1) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    LinkedList<TableModelListener> _listeners = new LinkedList();
    
    @Override
    public void addTableModelListener(TableModelListener l) {
        synchronized(_listeners) {
            _listeners.add(l);
        }
    }

    @Override
    public void removeTableModelListener(TableModelListener l) {
        synchronized(_listeners) {
            _listeners.remove(l);
        }
    }

    boolean updateRequest = false;
    
    public void invokeListener() {
        updateRequest = true;
        MXTimer.letsCountdown(100, new Runnable() { 
            public void run() {
                SwingUtilities.invokeLater(new Runnable() {
                    public void run() {
                        invokeListenerImpl();
                    }
                });
            }
        });
    }
    
    public void invokeListenerImpl() {
        synchronized(_listeners) {
            if (updateRequest) {
                updateRequest = false;
                for (TableModelListener l : _listeners) {
                    TableModelEvent e = new TableModelEvent(this, 0, getRowCount());
                    l.tableChanged(e);
                }
            }
        }
    }
}
