/*
 * Copyright (C) 2022 YOSHIDA Shintarou
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.libs.console;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import javax.swing.event.ListDataListener;
import java.util.ArrayList;
import javax.swing.DefaultListModel;
import javax.swing.JList;
import javax.swing.event.ListDataEvent;
import jp.synthtarou.midimixer.libs.MultiThreadQueue;
import javax.swing.SwingUtilities;

/**
 *
 * @author YOSHIDA Shintarou
 */

public class MXConsoleModel implements javax.swing.ListModel {
    private String[] listLogging = null;
    private int indexFrom = 0;
    private int indexTo = 0;
    private JList bind = null;
    private boolean _pause = false;

    private ArrayList<ListDataListener> listListner = new ArrayList<>();
    private MultiThreadQueue<String> _queue;

    public void setPause(boolean pause) {
        _pause = pause;
    }
    
    public synchronized void clearText() {
        indexFrom = 0;
        indexTo = 0;
    }
    
    public MXConsoleModel(int maxRowCount) {
        listLogging = new String[maxRowCount];

        _queue = new MultiThreadQueue<String>();
        new Thread(new Runnable() {
            public  void run() {
                while(true) {
                    String text = _queue.pop();
                    if (text == null) {
                       _queue.quit();
                       break;
                    }
                    addTextInThread(text);
                }
            }
        }).start();
    }
    
    public void bind(JList list) {
        if (bind != null) {
            bind.setModel(new DefaultListModel());
        }
        bind = list;
        bind.setModel(this);
    }

    @Override
    public int getSize() {
        int count = indexTo - indexFrom;
        return (count >= 0) ? count : (count + listLogging.length);
    }

    @Override
    public Object getElementAt(int index) {
        int pos = indexFrom + index;
        while (pos >= listLogging.length) {
            pos -= listLogging.length;
        }
        return listLogging[pos];
    }
    
    static byte[] _hex = "0123456789ABCDEF".getBytes();

    public void addConsoleMessage(String msg)   {
        if (_pause) {
            return;
        }
        _queue.push(msg);
    }

    DateTimeFormatter simpleFormat = DateTimeFormatter.ofPattern("HH:mm:ss.SSS");
    //DateTimeFormatter simpleFormat = DateTimeFormatter.ofPattern("HH:mm");

    private void addTextInThread(final String msg) {
        if (SwingUtilities.isEventDispatchThread() == false) {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    addTextInThread(msg);
                }
            });
            return;
        }
        boolean eventRemoved = false;
        boolean eventAdded = false;
        
        String message = /* LocalDateTime.now().format(simpleFormat) + " > " + */msg;
        
        listLogging[indexTo++] = message;
        if (indexTo >= listLogging.length) {
            indexTo -= listLogging.length;
        }
        eventAdded = true;

        if (indexTo == indexFrom) {
            listLogging[indexFrom++] = null;
            if (indexFrom >= listLogging.length) {
                indexFrom -= listLogging.length;
            }
            eventRemoved = true;
        }

        synchronized(this) {
            if (eventRemoved) {
                final ListDataEvent e = new ListDataEvent(this, ListDataEvent.INTERVAL_REMOVED, 0, 0);
                for (ListDataListener listener : listListner) {
                    listener.intervalRemoved(e);
                }
            }
            if (eventAdded) {
                int size = MXConsoleModel.this.getSize();
                ListDataEvent e = new ListDataEvent(this, ListDataEvent.INTERVAL_ADDED, size - 1, size - 1);
                for (ListDataListener listener : listListner) {
                    listener.intervalAdded(e);
                }
            }

            if (bind != null && eventRemoved == false) {
                bind.ensureIndexIsVisible(bind.getModel().getSize() -1);
            }
        }
    }

    @Override
    public synchronized void addListDataListener(ListDataListener l) {
        listListner.add(l);
    }

    @Override
    public synchronized void removeListDataListener(ListDataListener l) {
        listListner.remove(l);
    }
}
