/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.libs.settings;

import java.awt.Desktop;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.TreeSet;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.MXLineReader;
import jp.synthtarou.midimixer.libs.MXLineWriter;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MXSetting {
    private static final MXDebugConsole _debug = new MXDebugConsole(MXSetting.class);

    public static void main(String[] args) {
        MXDebugConsole.globalSwitchOn();
        MXSetting root = new MXSetting("test");
         
        root.register("base.attribute");
        root.register("base.attribute[].text");
        root.register("base[b]");
        root.register("base[1]");
        root.register("base.[].position");
        
        root.setSetting("base", "123");
        root.setSetting("base.attribute[1].text", "a1");
        root.setSetting("base.attribute[2].text", "a2");
        root.setSetting("base.attribute[3].text", "a3");
        
        root.setSetting("base[12].position", "12");
        root.setSetting("base.[13].position", "13");
        root.setSetting("base[14]position", "14");
        
        try {
            root.dump(new OutputStreamWriter(System.out));
        }catch(IOException e) {
            e.printStackTrace();
        }
    }

    protected File _settingFile;
    static ArrayList<MXSetting> everySetting = new ArrayList();

    public File getFile() {
        return _settingFile;
    }
    
    public MXSetting(File file) {
        this(file, true);
    }

    public MXSetting(File file, boolean addToEvery) {
        _settingFile = file;
        _registered = new TreeSet(comparatorForRegister);
        if (addToEvery) {
            everySetting.add(this);
        }
    }

    public MXSetting(String name) {
        this(new File(MXSettingUtil.getSettingDirectory(), name +".ini"));
    }
    
    public static void saveEverySettingToFile() {
         for (MXSetting setting : everySetting) {
            try {
                
                setting.writeToFile();
            }catch(Exception e) {
                e.printStackTrace();
            }
        }
   }
   
    protected static class Detail {
        String key;
        String value;
    }
    
    protected static class DetailArray {
        ArrayList<Detail> list = new ArrayList();
    }
    
    MXSettingTarget _target = null;
    
    public void setTarget(MXSettingTarget target) {
        _target = target;
        _target.prepareSettingFields(this);
    }
    
    MXSettingNode _root = new MXSettingNode(this, null, null);
    
    public boolean readFile() {
        if (_registered.size() == 0) {
            return false;
        }
        InputStream fin = null;
        try {
            _root.clearValues();
            fin = new FileInputStream(_settingFile);
            _debug.println("reading " + _settingFile);
            MXLineReader reader = new MXLineReader(fin, "utf-8");
            while(true) {
                String line = reader.readLine();
                if (line == null) {
                    break;
                }
                if (line.length() == 0) {
                    continue;
                }
                char first = line.charAt(0);
                if (first == '#' || first == '/') {
                    continue;
                }
                int index = line.indexOf('=');
                if (index < 0) {
                    continue;
                }
                String key = line.substring(0, index);
                String value = line.substring(index + 1);
                _root.setSetting(key, value);
            }
        }catch(FileNotFoundException e) {
            _debug.println("First Time for [" + _settingFile + "]");
        }catch(MXSettingException e) {
            _debug.printStackTrace(e);
        }catch(IOException e) {
            _debug.printStackTrace(e);
        }finally {
            if (fin != null) {
                try {
                    fin.close();
                }catch(IOException e) {
                    e.printStackTrace();
                }
            }
        }
        if (_target != null) {
            _target.afterReadSettingFile(this);
        }
        return true;
    }
    
    public boolean writeToFile() {
        if (_target != null) {
            _root.clearValues();
            _target.beforeWriteSettingFile(this);
        }
        File temporary = MXSettingUtil.createTemporaryFile(_settingFile);
        MXLineWriter writer = null;
        _debug.println("writing " + _settingFile);
        
        try {
            writer = new MXLineWriter(temporary, "utf-8");
            
            dump(writer.getWriter());

            writer.close();

            File backup = MXSettingUtil.autobackupFile(_settingFile);
            temporary.renameTo(_settingFile);
            
            if (backup != null) {
                if (MXSettingUtil.isSameFile(backup, _settingFile)) {
                    try {
                        Desktop.getDesktop().moveToTrash(backup);
                    }catch(Exception e) {
                        backup.delete();
                    }
                }
            }

        }catch(IOException ioe) {
            _debug.printStackTrace(ioe);
            if (writer != null) {
                try {
                    writer.close();
                }catch(Exception e) {
                    _debug.printStackTrace(e);
                }
                _settingFile.delete();
            }
            return false;
        }
        
        return true;
    }
    
    public void clearValue() {
        _root.clearValues();
    }

    public String getSetting(String name) {
        return _root.getSetting(name);
    }

    public int getSettingAsInt(String name, int defvalue) {
        return _root.getSettingAsInt(name, defvalue);
    }

    public boolean getSettingAsBoolean(String name, boolean defvalue) {
        return _root.getSettingAsBoolean(name, defvalue);
    }
    
    public boolean setSetting(String name, String value) {
        try {
           return _root.setSetting(name, value);
        }catch(MXSettingException e) {
            _debug.printStackTrace(e);
            return false;
        }
    }
    
    public boolean isEmpty() {
        return _root.isEmpty();
    }

    public boolean setSetting(String name, int value) {
        return setSetting(name, String.valueOf(value));
    }
    
    public boolean setSetting(String name, boolean value) {
        if (value) {
            return setSetting(name, "1");
        }else {            
            return setSetting(name, "0");
        }
    }
    
    public boolean havingName(String name) {
        return _root.childByKey(name) != null;
    }
    
    public void dump(Writer writer) throws IOException {
        _root.recuesiveDump(writer);
        writer.flush();
    }
    
    public void register(String name) {
        try {
            Path p = _root.getPath().clone();
            p.addAll(Path.parsePath(name));
            register(p);
        }catch(MXSettingException e) {
            System.out.println("Can't regist " + name +" for " + getFile());
        }
    }

    public void register(Path path) {
        _registered.add(path);
    }

    public boolean isRegistered(String name) {
        Path p = _root.getPath().clone();
        try {
            p.addAll(Path.parsePath(name));
            return isRegistered(p);
        }catch(MXSettingException e) {
            e.printStackTrace();
            return false;
        }
    }

    public boolean isRegistered(Path path) {
        return _registered.contains(path);
    }

    protected TreeSet<Path> _registered;

    protected Comparator comparatorForRegister = new Comparator() {
        public int compare(Object o1, Object o2) {
            Path p1 = (Path)o1;
            Path p2 = (Path)o2;
            int length = Math.min(p1.size(), p2.size());
            for (int x = 0; x < length; ++ x) {
                String str1 = p1.get(x);
                String str2 = p2.get(x);
                
                if (MXSettingUtil.isInteger(str1)) {
                    str1 = "0";
                }
                if (MXSettingUtil.isInteger(str2)) {
                    str2 = "0";
                }
                
                int d = str1.compareTo(str2);
                if (d != 0) {
                    return d;
                }
            }
            if (p1.size() < p2.size()) {
                return -1;
            }else if (p1.size() > p2.size()) {
                return 1;
            }
            
            return 0;
        }
    };
    
    public ArrayList<MXSettingNode> findByPath(String name) {
        Path path = null;
        try {
            path = Path.parsePath(name);
        }catch(MXSettingException e) {
            e.printStackTrace();
            return null;
        }
    
        //TODO 理解不能

        ArrayList<MXSettingNode> seeking = new ArrayList();
        seeking.add(_root);

        for (String text : path) {
            ArrayList<MXSettingNode> hit = new ArrayList();
            boolean isInteger = MXSettingUtil.isInteger(text);
            
            for (MXSettingNode parent : seeking) {
                int count = parent.size();
                for (int x = 0; x < count; ++ x) {
                    MXSettingNode child = parent.childByIndex(x);
                    if (isInteger) {
                        if (child.isInteger()) {
                            hit.add(child);
                        }
                    }else {
                        if (child.getName().equalsIgnoreCase(text)) {
                            hit.add(child);
                        }
                    }
                }
            }
            
            seeking = hit;
        }

        return seeking;
    }
}
