/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx10input;

import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMidi;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MX10Data {
    private static final MXDebugConsole _debug = new MXDebugConsole(MX10Data.class);

    public static final int TYPE_PITCH_BEND = 1; 
    public static final int TYPE_MOD_WHEEL = 2;
    public static final int TYPE_DAMPER_PEDAL = 3;
    public static final int TYPE_BANK_SELECT = 4;
    public static final int TYPE_PROGRAM_CHANGE  = 5;
    public static final int TYPE_DATA_ENTRY = 6;
    public static final int TYPE_ANOTHER_CC = 7;
    public static final int TYPE_SYSEX = 8;
    public static final int TYPE_ACTIVE_SENSING = 9;

    public static final int TYPE_COUNT = 9;
    
    public static final int TYPE_ZERO = 0;

    public static final String[] typeNames = {
        "PitchBend", "ModWheel", "DamperPedal", "BankChange",
        "ProgramChange", "DataEntry", "AnotherCC", "SysEX",
        "ActiveSensing"
    };
    
    long[] _whichToSkip;
    int _portCount;

    public MX10Data() {
        _portCount = MXStatic.TOTAL_PORT_COUNT;
        _whichToSkip = new long[MXStatic.TOTAL_PORT_COUNT];
        for (int i = 0; i < _whichToSkip.length; ++ i) {
            _whichToSkip[i] = 1L << TYPE_ACTIVE_SENSING;
        }
    }
    
    public boolean isMessageToSkip(MXMessage message) {
        int port = message.getPort();
        int command = message.getCommand();

        int type = TYPE_ZERO;

        if (command == MXMidi.STATUS_ACTIVESENSING) {
            type = TYPE_ACTIVE_SENSING;
        }
        else if (command == 0xf0 || command == 0xf7) {
            type = TYPE_SYSEX;
        }
        else if (command == MXMidi.COMMAND_CONTROLCHANGE) {
            int data1 = message.getCCCodeFromBytes();
            if (data1 == MXMidi.DATA1_CCBANKSELECT || data1 == 0x20) {
                type = TYPE_BANK_SELECT;
            }else if (data1 == MXMidi.DATA1_RPN_LSB || data1 == MXMidi.DATA1_RPN_MSB
                   || data1 == MXMidi.DATA1_NRPN_LSB || data1 == MXMidi.DATA1_NRPN_MSB
                   || data1 == MXMidi.DATA1_CC_DATAINC || data1 == MXMidi.DATA1_CC_DATADEC
                   || data1 == MXMidi.DATA1_CC_DATAENTRY) {
                type = TYPE_DATA_ENTRY;
            }else if (data1 == MXMidi.DATA1_CCDUMPERPEDAL) {
                type = TYPE_DAMPER_PEDAL;
            }else if (data1 == MXMidi.DATA1_CCMODULATION) {
                type = TYPE_MOD_WHEEL;
            }else {
                type = TYPE_ANOTHER_CC;
            }
        }else  if (command  == MXMidi.COMMAND_PROGRAMCHANGE) {
            type = TYPE_PROGRAM_CHANGE;
        }else if (command == MXMidi.COMMAND_PITCHWHEEL) {
            type = TYPE_PITCH_BEND;
        }
        if (type == TYPE_ZERO) {
            return false;
        }
        return isSkip(port, type);
    }
    
    public void setSkip(int port, int type, boolean skipFlag) {
        long bit = 1L << type;
        if ((_whichToSkip[port] & bit) != 0) {
            if (skipFlag) {
                return;
            }else {
                _whichToSkip[port] -= bit;
            }
        }else {
            if (skipFlag) {
                _whichToSkip[port] |= bit;
            }else {
                return;
            }
        }
    }
    
    public boolean isSkip(int port, int type) {
        long bit = 1L << type;
        if ((_whichToSkip[port] & bit) != 0) {
            return true;
        }
        return false;
    }
    
    public static int countOfTypes() {
        return TYPE_COUNT;
    }

    public static String nameOfType(int x) {
        return typeNames[x];
    }
}
