/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx40layer;

import java.util.ArrayList;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.midi.MXNoteOffWatcher;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.MXMidiUI;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MX40Group {
    private static final MXDebugConsole _debug = new MXDebugConsole(MX40Group.class);

    public ArrayList<MX40Layer> _listLayer = new ArrayList();
    
    public String _title = "New";
    public boolean _isWatchPort = false;
    public int _watchingPort = 0;
    public boolean _isWatchChannel = false;
    public int _watchingChannel = 0;
    public boolean _isWatchBank = false;
    public int _watchingBankMSB = 0;
    public int _watchingBankLSB = 0;
    public boolean _isWatchProgram = false;
    public int _watchingProgram = 0;
    public boolean _rotateLayer = false;
    public int _rotatePoly = 16;

    public int[] _rotateCount = new int[16*9];

    private int _lastLayerPos = -1;
    
    private final MX40Process _process;
    
    MXNoteOffWatcher _noteOff = new MXNoteOffWatcher();
    
    public MX40Group(MX40Process process) {
        _process = process;
    }
    
    public String toString() {
        StringBuffer str = new StringBuffer();
        str.append("By ");
        str.append("Port=").append(_isWatchPort).append("[").append(MXMidiUI.nameOfPort(_watchingPort)).append("]");
        str.append("Channel=").append(_isWatchChannel).append("[").append(_watchingChannel+1).append("]");
        str.append("Program=").append(_isWatchProgram).append("[").append(_watchingProgram).append("]");
        str.append("Bank=").append(_watchingBankMSB).append(":").append(_watchingBankLSB).append("]");
        return str.toString();
    }
    
    public boolean isAssigned(MXChannelInfoReceiver info, int port, int channel) {
        MXChannelInfo e = info.getChannelInfo(port, channel);

        final int bankMSB = e._havingBank ? e._bankMSB : -1;
        final int bankLSB = e._havingBank ? e._bankLSB : -1;
        final int program = e._havingProgram ? e._program : -1;

        boolean assigned = false;
    
        if (_listLayer.size()== 0) {
            return false;
        }
        if (_isWatchPort || _isWatchChannel || _isWatchProgram || _isWatchBank) {
            assigned = true;
            if (_isWatchPort) {
                if (_watchingPort != port) {
                    assigned = false;
                }
            }
            if (_isWatchChannel) {
                if (_watchingChannel != channel) {
                    assigned = false;
                }
            }
            if (_isWatchProgram) {
                if (_watchingProgram != program) {
                    assigned = false;
                }
            }
            if (_isWatchBank) {
                if (_watchingBankMSB != bankMSB ||  _watchingBankLSB != bankLSB) {
                    assigned = false;
                }
            }
        }else {
            if(e._havingBank || e._havingProgram) {
                assigned = true;
            }else {
                assigned = false;
            }
        }
        
        return assigned;
    }
    
    public boolean processByGroup(MXMessage message) {
        if (message.isMessageTypeChannel() == false) {
            return false;
        }

        final int port = message.getPort();
        final int command = message.getCommand();
        final int channel = message.getChannel();

        if (command == MXMidi.COMMAND_NOTEOFF) {
            _noteOff.notifyNoteOffEvent(port, channel, message.getNoteNumberFromBytes(), "@1");
            return true;
        }

        if (command == MXMidi.COMMAND_CONTROLCHANGE) {
            int cc = message.getCCCodeFromBytes();
            if (cc == MXMidi.DATA1_CCBANKSELECT) {
                return true;
            }
            if (cc == MXMidi.DATA1_CCBANKSELECT + 32) {
                return true;
            }
        }

        //記録したプログラム番号が処理対象か判断する
        boolean assigned = isAssigned(_process._inputInfo, port, channel);
        
        if (command == MXMidi.COMMAND_PROGRAMCHANGE) {
            if (assigned) {
                for (MX40Layer layer: _listLayer) {
                    layer.processProgramChange(message);
                }
                return true;
            }
            return false;
        }

        if (!assigned) { 
            return false;
        }
        boolean proced = false;

        if (_listLayer.size() == 0) {
            return false;
        }
        if (_rotateLayer && command == MXMidi.COMMAND_NOTEON) {
            int startLayer = _lastLayerPos;
            do {
                int nextLayer = _lastLayerPos + 1;
                if (nextLayer < 0 || nextLayer >= _listLayer.size()) {
                    nextLayer = 0;
                }

                MX40Layer layer = _listLayer.get(nextLayer);
                proced = layer.processByLayer(message);
                MXMessage msg2 = MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_NOTEOFF + channel, message.getNoteNumberFromBytes(), 0);
                _noteOff.addListener(message, msg2,  new MXNoteOffWatcher.Handler() {
                    public void onNoteOffEvent(MXMessage target) {
                        layer.processByLayer(target);
                    }
                }, "@1");
                _lastLayerPos = nextLayer;
            }while(!proced && startLayer != _lastLayerPos);
        }else if (_rotatePoly == 1 && command == MXMidi.COMMAND_NOTEON) {
            int startLayer = _lastLayerPos;
            int found = -1;
            int foundPoly = 100;

            //一番発音の少ないところ
            for (int i = 0; i < _listLayer.size(); ++ i) {
                MX40Layer layer = _listLayer.get(i);
                if (_rotateCount[i] < foundPoly) {
                    found = i;
                    foundPoly = _rotateCount[i];
                }
               }
            if (_rotateCount[found] >= 1) {
                System.out.println("発音数オーバー");
                proced = true;
            }
            else {
                _rotateCount[found] ++;

                final int finalFound = found;

                MX40Layer layer = _listLayer.get(found);
                proced = layer.processByLayer(message);

                MXMessage msg2 = MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_NOTEOFF + channel, message.getNoteNumberFromBytes(), 0);
                _noteOff.addListener(message, msg2,  new MXNoteOffWatcher.Handler() {
                    public void onNoteOffEvent(MXMessage target) {
                        layer.processByLayer(target);
                    }
                }, "@1");
            }
        }else {
            for (MX40Layer layer: _listLayer) {
                layer.processByLayer(message);
                if (command == MXMidi.COMMAND_NOTEON) {
                    MXMessage msg2 = MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_NOTEOFF + channel, message.getNoteNumberFromBytes(), 0);
                    _noteOff.addListener(message, msg2,  new MXNoteOffWatcher.Handler() {
                    public void onNoteOffEvent(MXMessage target) {
                        layer.processByLayer(target);
                    }
                }, "@1");
                }
                proced = true;
            }
        }

        return proced;
    }

    
    public boolean equals(Object o) {
        MX40Group target = (MX40Group)o;
        if (this._title.equals(target._title)
        && this._isWatchPort == target._isWatchPort
        && this._watchingPort == target._watchingPort
        && this._isWatchChannel == target._isWatchChannel
        && this._watchingChannel == target._watchingChannel
        && this._isWatchBank == target._isWatchBank
        && this._watchingBankMSB == target._watchingBankMSB
        && this._watchingBankLSB == target._watchingBankLSB
        && this._isWatchProgram == target._isWatchProgram
        && this._watchingProgram == target._watchingProgram
        && this._rotateLayer == target._rotateLayer
        && this._rotatePoly == target._rotatePoly) {
            return true;
        }
        return false;
    }
}

