/*
 * Copyright (c) 2003, jMonkeyEngine - Mojo Monkey Coding
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this 
 * list of conditions and the following disclaimer. 
 * 
 * Redistributions in binary form must reproduce the above copyright notice, 
 * this list of conditions and the following disclaimer in the documentation 
 * and/or other materials provided with the distribution. 
 * 
 * Neither the name of the Mojo Monkey Coding, jME, jMonkey Engine, nor the 
 * names of its contributors may be used to endorse or promote products derived 
 * from this software without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */
package jme.geometry.hud;

/**
 * <code>Component</code> defines the base level object for the heads up display
 * and/or graphical user interface elements. The component is intended to be
 * heirarchical and allow for components to be added to the parent component.
 * 
 * @author Mark Powell
 */
public interface Component {
    /**
     * <code>add</code> adds a specified component as a child of this component.
     * The child components will always be rendered after the parent. The 
     * children will be rendered on top of the parent. 
     * @param subComponent the child component.
     */
    public void add(Component subComponent);
    
    /**
     * <code>remove</code> subtracts the specified component from the list of
     * children. This will call dispose on the child, and subsequently all its
     * children. 
     * @param subComponent the compoent to delete.
     */
    public void remove(Component subComponent);
    
    /**
     * <code>contains</code> returns true if the given point is within the
     * constraints of the component. Useful for things such as mouse placement.
     * @param x the x coordinate of the tested point.
     * @param y the y coordinate of the tested point.
     * @return true if the point is contained within the component, false 
     *      otherwise.
     */
    public boolean contains(int x, int y);
    
    /**
     * <code>setColor</code> sets the color of the component. The values of
     * the red, green and blue components are expected to be between 0 and 1.
     * @param r the red component of the color.
     * @param g the green component of the color.
     * @param b the blue component of the color.
     */
    public void setColor(float r, float g, float b);
    
    /**
     * 
     * <code>setTransparency</code> sets the transparency of the component 
     * where 0 is completely transparent and 1 is completely opaque.
     * @param a the alpha value of the component.
     */
    public void setTransparency(float a);
    
    /**
     * 
     * <code>setTexture</code> sets the texture if any of the component.
     * @param texture the texture image that defines the texture of the 
     *      component.
     */
    public void setTexture(String texture);
    
    /**
     * 
     * <code>setSize</code> sets the size of the component.
     * @param width the width of the component.
     * @param height the height of the component.
     */
    public void setSize(float width, float height);
    
    /**
     * 
     * <code>setLocation</code> sets the lower left point of the component.
     * @param x the x coordinate of the lower left point.
     * @param y the y coordinate of the lower left point.
     */
    public void setLocation(float x, float y);
    
    /**
     * 
     * <code>isVisible</code> determines if the component and it's children are
     * rendered. If the component is not visible it's children are also not
     * visible.
     * @param value if true the component and it's children are rendered, 
     *      otherwise not.
     */
    public void isVisible(boolean value);
    
    /**
     * 
     * <code>getWidth</code> returns the current width of the component.
     * @return the current width of the component.
     */
    public int getWidth();
    
    /**
     * 
     * <code>getHeight</code> returns the current height of the component.
     * @return the current height of the component.
     */
    public int getHeight();
    
    /**
     * 
     * <code>getX</code> returns the current x location of the component. 
     * Where the location is defined by the lower left point.
     * @return the x coordinate of the lower left point.
     */
    public int getX();
    
    
    /**
     * 
     * <code>getY</code> returns the current y location of the component.
     * Where the location is defined by the lower left point.
     * @return the y coordinate of the lower left point.
     */
    public int getY();
    
    /**
     * 
     * <code>dispose</code> destroys the component and all it's children.
     *
     */
    public void dispose();
    
    /**
     * <code>render</code> takes care of displaying the component to the 
     * screen.
     *
     */
    public void render();
}
