/*
 * Copyright (c) 2003, jMonkeyEngine - Mojo Monkey Coding
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this 
 * list of conditions and the following disclaimer. 
 * 
 * Redistributions in binary form must reproduce the above copyright notice, 
 * this list of conditions and the following disclaimer in the documentation 
 * and/or other materials provided with the distribution. 
 * 
 * Neither the name of the Mojo Monkey Coding, jME, jMonkey Engine, nor the 
 * names of its contributors may be used to endorse or promote products derived 
 * from this software without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package jme.controller;

/**
 * <code>MouseController</code> defines an interface for accessing current
 * mouse interaction. 
 * 
 * A <code>MouseController</code> implementation should keep track of the
 * absolute mouse coordinate, i.e. the coordinate of the cursor anywhere on
 * the screen, the regular mouse coordinate, i.e. the coordinate within the
 * application window (if the cursor moves out of the window, the regular
 * coordinate maintains the point where it exited). Third is the 
 * relative mouse coordinate with keeps track of the mouse change from one
 * frame to the next.
 * 
 * Using <code>setMousePosition</code> sets the position of the regular 
 * mouse coordinate.
 * 
 * @author Mark Powell
 * @version 1
 */
public interface MouseController {
    
    /**
     * <code>getMousePositionX</code> returns the x coordinate of the mouse
     * within the application window. If the mouse moves out of the window,
     * the position will set the correct coordinate to the window's extreme.
     * 
     * @return the regular x coordinate of the mouse.
     */
    public int getMousePositionX();
    
     /**
     * <code>getMousePositionY</code> returns the y coordinate of the mouse
     * within the application window. If the mouse moves out of the window,
     * the position will set the correct coordinate to the window's extreme.
     * 
     * @return the regular y coordinate of the mouse.
     */
    public int getMousePositionY();
    
    /**
     * <code>getAbsoluteMousePositionX</code> retrieves the actual x position
     * of the mouse cursor. 
     * 
     * @return the actual x position of the mouse cursor.
     */
    public int getAbsoluteMousePositionX();
    
    /**
     * <code>getAbsoluteMousePositionY</code> retrieves the actual y position
     * of the mouse cursor. 
     * 
     * @return the actual y position of the mouse cursor.
     */
    public int getAbsoluteMousePositionY();
    
    /**
     * <code>getMouseDeltaX</code> returns the change in the X coordinate of 
     * the mouse since the previous frame.
     * 
     * @return the change in X since the previous frame.
     */
    public int getMouseDeltaX();
    
    /**
     * <code>getMouseDeltaY</code> returns the change in the Y coordinate of 
     * the mouse since the previous frame.
     * 
     * @return the change in Y since the previous frame.
     */
    public int getMouseDeltaY();
    
    /**
     * <code>setMousePosition</code> sets the x and y coordinate of the mouse
     * within the cooridnates of the application window.
     * 
     * @param x the new x coordinate.
     * @param y the new y coordinate.
     */
    public void setMousePosition(int x, int y);
    
    /**
     * <code>pollMouse</code> obtains the latest deltas for the mouse from
     * the <code>Mouse</code> context.
     */
    public void pollMouse();
    
    /**
     * <code>isButtonDown</code> returns true if a specified button is currently
     * pressed on the mouse, false if it is not.
     * 
     * @param mouseButton the button id to check.
     * @return true if the button is pressed, false if it is not.
     */    
    public boolean isButtonDown(int mouseButton);
    
    /**
     * <code>getNumberOfButtons</code> returns the number of buttons
     * assigned to the mouse.
     * 
     * @return the number of buttons the mouse has.
     */
    public int getNumberOfButtons();
}
