﻿/**
 *@file xml.h
 *
 *@English
 *@brief Functions to handle XML data
 *@EndEnglish
 *
 *@Japanese
 *@brief XMLデータを扱う関数群
 *@EndJapanese
 *
 *
 */
#ifndef _NLC_XML_H
#define _NLC_XML_H
#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libxml/xpath.h>
#include <NicoLib/types.h>

#ifdef __cplusplus
extern "C" {
#endif


/** 
 *@English
 *@defgroup NLCXMLParser
 *@EndEnglish
 *
 *@Japanese
 *@defgroup NLCXMLParser
 *@EndJapanese
 *
 */
/*@{*/ /* vNLCXMLParserv */


/**
 *@English
 *@brief Structure for nlc_xmlparser functions
 *
 *@warning DO NOT TOUCH THIS MANUALLY.@n
 *         Use nlc_xmlparser_new() and nlc_xmlparser_delete() instead.
 *
 *@EndEnglish
 *
 *@Japanese
 *@brief nlc_xmlparser系列の関数で使用する構造体
 *
 *@warning 直接操作してはいけない。@n
 *         メモリの確保・解放には専用の関数を使用する。
 *
 *@sa nlc_xmlparser_new() @n
 *    nlc_xmlparser_delete()
 *@EndJapanese
 */
typedef struct
{
	xmlDoc *doc;			/**< 渡されたバッファに関連付けられるlibxmlのオブジェクト */
	xmlXPathContext *ctx;	/**< XPathでパースする時に使用されるコンテキストハンドル */
}NLCXMLParser;


/** 
 *@English
 *@defgroup Memory related functions
 *@EndEnglish
 *
 *@Japanese
 *@defgroup メモリ確保・解放系関数
 *@EndJapanese
 *
 */
/*@{*/ /* vMemory related functionsv */


/**
 *@English
 *@brief 
 *
 *
 *
 *@EndEnglish
 *
 *@Japanese
 *@brief nlc_xmlparser系列の関数で使用する NLCXMLParser 構造体を確保・初期化し、そのポインタを返す
 *
 *@param xml メモリ上のXMLドキュメントへのポインタ
 *@param xmllen \c xml のサイズ(NULL文字を含まない)
 *@param encoding XMLドキュメントのエンコーディング@n
 *                NULLが渡された場合は"utf-8"が内部で使用される
 *
 *@return 確保された NLCXMLParser 構造体へのポインタ
 *@retval NULL 確保失敗
 *
 *@sa nlc_xmlparser_delete()
 *@EndJapanese
 */
NLCXMLParser *nlc_xmlparser_new(
		const char *xml, unsigned int xmllen, 
		const char *encoding
		);

/**
 *@English
 *@brief 
 *
 *
 *
 *@EndEnglish
 *
 *@Japanese
 *@brief nlc_xmlparser系列の関数で使用する NLCXMLParser 構造体を解放する
 *
 *@param nlcxml 確保済みの NLCXMLParser 構造体へのポインタ
 *
 *@return 常に NLCResult::NLCE_OK
 *
 *@sa nlc_xmlparser_new()
 *@EndJapanese
 */
NLCResult nlc_xmlparser_delete(
		NLCXMLParser *nlcxml
		);


/*@}*/ /* ^Memory related functions^ */


/** 
 *@English
 *@defgroup XPath parsers
 *@EndEnglish
 *
 *@Japanese
 *@defgroup XPathパーサー
 *@EndJapanese
 *
 */
/*@{*/ /* vXPath parsersv */


/**
 *@English
 *@brief 
 *
 *
 *
 *@EndEnglish
 *
 *@Japanese
 *@brief 渡されたXPathに適合する最初のノードを文字列として変数に格納する
 *
 *@param nlcxml 確保済みの NLCXMLParser 構造体へのポインタ
 *@param buf 空のchar *型の変数のアドレス
 *@param xpath 評価するXPath
 *
 *@return NLCResult の汎用エラーまたは NLCResult::NLCE_XMLPARSER_ERR
 *
 *@attention bufには内部で確保されたメモリ領域へのポインタが代入される。
 *
 *@sa nlc_xmlparser_new() @n
 *    nlc_xmlparser_delete()
 *@EndJapanese
 */
NLCResult nlc_xmlparser_parse(
		NLCXMLParser *nlcxml, 
		char **buf, const char *xpath
		);

/**
 *@English
 *@brief 
 *
 *
 *
 *@EndEnglish
 *
 *@Japanese
 *@brief 可変引数によって渡された各XPathに適合する最初のノードを文字列として配列に格納する
 *
 *@param nlcxml 確保済みの NLCXMLParser 構造体へのポインタ
 *@param ptable 空のchar *型の配列のアドレス
 *@param num 評価するXPathの個数
 *@param ... 評価するXPath(複数可)
 *
 *@return NLCResult の汎用エラーまたは NLCResult::NLCE_XMLPARSER_ERR
 *
 *@attention この関数は nlc_xmlparser_parse() のラッパー関数である。
 *
 *@sa nlc_xmlparser_new() @n
 *    nlc_xmlparser_delete() @n
 *    nlc_xmlparser_parse()
 *@EndJapanese
 */
NLCResult nlc_xmlparser_parse_va(
		NLCXMLParser *nlcxml,
		char ***ptable, unsigned int num, ...
		);

/**
 *@English
 *@brief 
 *
 *
 *
 *@EndEnglish
 *
 *@Japanese
 *@brief char *型の配列によって渡された各XPathに適合する最初のノードを文字列として配列に格納する
 *
 *@param nlcxml 確保済みの NLCXMLParser 構造体へのポインタ
 *@param ptable 空のchar *型の配列のアドレス
 *@param num 評価するXPathの個数
 *@param xpaths 評価するXPathを格納したchar *型の配列
 *
 *@return NLCResult の汎用エラーまたは NLCResult::NLCE_XMLPARSER_ERR
 *
 *@attention この関数は nlc_xmlparser_parse() のラッパー関数である。
 *
 *@sa nlc_xmlparser_new() @n
 *    nlc_xmlparser_delete() @n
 *    nlc_xmlparser_parse() @n
 *    nlc_xmlparser_parse_va()
 *@EndJapanese
 */
NLCResult nlc_xmlparser_parse_array(
		NLCXMLParser *nlcxml,
		char ***ptable, unsigned int num, char **xpaths
		);


/*@}*/ /* ^XPath parsers^ */


/*@}*/ /* ^NLCXMLParser^ */


#ifdef __cplusplus
}
#endif
#endif


