﻿#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libxml/xpath.h>
#include <NicoLib/NicoLib.h>
#include <NicoLib/http.h>
#include <NicoLib/xmlparser.h>



NLCXMLParser *nlc_xmlparser_new(const char *xml, unsigned int xmllen, const char *encoding)
{
	NLCXMLParser *nlcxml = NULL;
	
	nlcxml = (NLCXMLParser *)malloc(sizeof(NLCXMLParser));
	nlcxml->doc = xmlReadMemory(xml, xmllen, "", encoding?encoding:"utf-8", 0);
	if(!nlcxml->doc)
	{
		free(nlcxml);
		return NULL;
	}

	nlcxml->ctx = xmlXPathNewContext(nlcxml->doc);
	if(!nlcxml->ctx)
	{
		xmlFreeDoc(nlcxml->doc);
		free(nlcxml);
		return NULL;
	}
	
	return nlcxml;
}


NLCResult nlc_xmlparser_delete(NLCXMLParser *nlcxml)
{
	if(!nlcxml)
	{
		return NLCE_OK;
	}

	if(nlcxml->doc)		xmlFreeDoc(nlcxml->doc);
	if(nlcxml->ctx)		xmlXPathFreeContext(nlcxml->ctx);
	free(nlcxml);

	return NLCE_OK;
}


NLCResult nlc_xmlparser_parse(NLCXMLParser *nlcxml, char **buf, const char *xpath)
{
	unsigned int i = 0;
	xmlXPathObject *xobj;
	xmlNodeSet *nodeset = NULL;
	xmlNode *cur = NULL;

	if(!nlcxml || !xpath || !buf)
	{
		return NLCE_BAD_FUNCTION_ARGUMENT;
	}

	if(!nlcxml->doc || !nlcxml->ctx)
	{
		#ifndef NLC_SILENT
		fprintf(stderr, "nlc: NLCXMLParser is not inited\n");
		#endif
		return NLCE_XMLPARSER_ERR;
	}


	#ifdef NLC_DEBUG
	printf("nlc: Parsing XML by XPath \"%s\"...\n", xpath);
	#endif

	// new xpath object
	xobj = xmlXPathEval((xmlChar *)xpath, nlcxml->ctx);
	if(!xobj)
	{
		fprintf(stderr, "nlc: Parsing XML by XPath \"%s\" failed\n", xpath);
		*buf = NULL;
		return NLCE_BAD_FUNCTION_ARGUMENT;
	}

	nodeset = xobj->nodesetval; // just for accessibility
	if(xmlXPathNodeSetIsEmpty(nodeset))
	{
		#ifdef NLC_DEBUG
		printf("nlc: Specified node [%s] not found\n", xpath);
		#endif
		*buf = NULL;
		return NLCE_LIBXML_ERR;
	}

	cur = nodeset->nodeTab[0];	// the first node found by given XPath
	if(cur->children->content == NULL) // TODO: <node />の時にエラー吐かないか
	{
		// maybe this might be somethig like <node></node>
		*buf = NULL;
		return NLCE_LIBXML_ERR;
	}


	// copy the content into char* array
	*buf = (char *)malloc(strlen((char *)cur->children->content)+1);
	if(!*buf)
	{
		return NLCE_NO_MEMORY;
	}
	strcpy(*buf, (char *)cur->children->content);
	
	
	// free xpath object
	xmlXPathFreeObject(xobj);

	return NLCE_OK;
}

NLCResult nlc_xmlparser_parse_va(NLCXMLParser *nlcxml, char ***ptable, unsigned int num, ...)
{
	char *xpath = NULL;
	unsigned int i = 0;
	va_list va = NULL;
	NLCResult res = NLCE_OK;

	if(!nlcxml || !ptable || num == 0)
	{
		return NLCE_BAD_FUNCTION_ARGUMENT;
	}

	if(!nlcxml->doc || !nlcxml->ctx)
	{
		#ifndef NLC_SILENT
		fprintf(stderr, "nlc: NLCXMLParser is not inited\n");
		#endif
		return NLCE_XMLPARSER_ERR;
	}

	// allocate memory for result
	*ptable = (char **)malloc(sizeof(char *)*num);
	if(!*ptable)
	{
		return NLCE_NO_MEMORY;
	}
	
	for(i=0; i<num; ++i)
	{
		(*ptable)[i] = NULL; // clear the char* array by NULL first
	}

	// use va-arg to receive XPaths
	va_start(va, num);
	for(i=0; i<num; ++i)
	{
		xpath = va_arg(va, char *);
		if(!xpath)
		{
			#ifndef NLC_SILENT
			fprintf(stderr, "nlc: Bad argument(XPath)\n");
			#endif
			continue;
		}

		// call parser function for each xpath
		res = nlc_xmlparser_parse(nlcxml, &((*ptable)[i]), xpath);
		if(res != NLCE_OK)
		{
			return res;
		}
	}
	va_end(va);

	return NLCE_OK;
}

NLCResult nlc_xmlparser_parse_array(NLCXMLParser *nlcxml, char ***ptable, unsigned int num, char **xpaths)
{
	char *xpath = NULL;
	unsigned int i = 0;
	va_list va = NULL;
	NLCResult res = NLCE_OK;

	if(!nlcxml || !ptable || num == 0 || !xpaths)
	{
		return NLCE_BAD_FUNCTION_ARGUMENT;
	}

	if(!nlcxml->doc || !nlcxml->ctx)
	{
		#ifndef NLC_SILENT
		fprintf(stderr, "nlc: NLCXMLParser is not inited\n");
		#endif
		return NLCE_XMLPARSER_ERR;
	}


	// allocate memory for result
	*ptable = (char **)malloc(sizeof(char *)*num);
	if(!*ptable)
	{
		return NLCE_NO_MEMORY;
	}
	
	for(i=0; i<num; ++i)
	{
		(*ptable)[i] = NULL; // clear the char* array by NULL first
	}

	// use char* table to receive XPaths
	for(i=0; i<num; ++i)
	{
		xpath = xpaths[i]; // just for accessibility
		if(!xpath)
		{
			#ifndef NLC_SILENT
			fprintf(stderr, "nlc: Bad argument(XPath)\n");
			#endif
			continue;
		}

		// call parser function for each xpath
		res = nlc_xmlparser_parse(nlcxml, &((*ptable)[i]), xpath);
		if(res != NLCE_OK)
		{
			return res;
		}
	}

	return NLCE_OK;
}



/*
NLCResult nlc_xml_get_and_parse(const char *url, char ***ptable, unsigned int num, ...)
{
	char *buf = NULL;
	char *param = NULL;
	char **xpaths = NULL;
	unsigned int i = 0;
	va_list va = NULL;
	NLCResult res = NLCE_OK;


	res = nlc_curl_http_get(&buf, url, NULL);
	if(res != NLCE_OK)
	{
		fprintf(stderr, "nlc: %s\n", nlc_err_tostr(res));
		return res;
	}
	if(!buf)
	{
		fprintf(stderr, "nlc: Bad buffer\n");
		return NLCE_UNK_ERR;
	}

	#ifdef NLC_DEBUG
	printf("buf = \n[%s]\n", buf);
	#endif

	xpaths = (char **)malloc(sizeof(char *)*num);
	if(!xpaths)
	{
		return NLCE_NO_MEMORY;
	}

	va_start(va, num);
	for(i=0; i<num; ++i)
	{
		param = va_arg(va, char *);
		if(!param)
		{
			xpaths[num] = NULL;
			continue;
		}

		xpaths[num] = param;				
	}
	va_end(va);

	res = nlc_xml_xpath_parse_array(buf, strlen(buf), NULL, ptable, 3, xpaths);
	free(buf);
	if(res != NLCE_OK)
	{
		fprintf(stderr, "nlc: %s\n", nlc_err_tostr(res));
		return res;
	}

	if(ptable == NULL)
	{
		fprintf(stderr, "nlc: XML parse failed!\n");
		return NLCE_LIBXML_ERR;
	}

	return NLCE_OK;
}
*/
