﻿#include <stdio.h>
#include <string.h>

#include <curl/curl.h>
#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libxml/xpath.h>

#include <NicoLib/NicoLib.h>
#include <NicoLib/http.h>
#include <NicoLib/xmlparser.h>
#include <NicoLib/api.h>


NLCResult nlc_live_antenna(NLCAlertData *adata, const char *mail, const char *password, NLCResult *api_err)
{
	char *query = NULL, *buf = NULL;
	char *statbuf = NULL, *codebuf = NULL;
	char **table = NULL;
	int i = 0, errcode = 0;
	NLCXMLParser *nlcxml = NULL;
	NLCResult res = NLCE_OK;


	// create POST parameter
	query = (char *)malloc(strlen("mail=&password=")+strlen(mail)+strlen(password)+1);
	if(!query)
	{
		return NLCE_NO_MEMORY;
	}
	sprintf(query, "mail=%s&password=%s", mail, password);

	// HTTP POST
	res = nlc_curl_http_post(&buf, "https://secure.nicovideo.jp/secure/login?site=nicolive_antenna", query);
	free(query);
	if(res != NLCE_OK)
	{
		fprintf(stderr, "nlc: %s\n", nlc_err_tostr(res));
		return res;
	}
	if(!buf)
	{
		fprintf(stderr, "nlc: Bad buffer\n");
		return NLCE_UNK_ERR;
	}

	#ifdef NLC_DEBUG
	// print raw XML data
	printf("nicolive_antenna = \n[%s]\n", buf);
	#endif


	// begin parsing XML
	nlcxml = nlc_xmlparser_new(buf, strlen(buf), "utf-8");
	if(!nlcxml)
	{
		free(buf);
		return NLCE_XMLPARSER_ERR;
	}

	// check for maintenance
	if(nlc_api_ismaintenance(nlcxml))
	{
		*api_err = NLCE_API_MAINTENANCE;

		free(buf);
		nlc_xmlparser_delete(nlcxml);
		return NLCE_API_ERR;
	}


	// check 'status' attribute
	res = nlc_xmlparser_parse(nlcxml, &statbuf, "/nicovideo_user_response/@status");
	if(!statbuf)
	{
		#ifndef NLC_SILENT
		fprintf(stderr, "nlc: failed to check 'status' attribute\n");
		#endif

		free(buf);
		nlc_xmlparser_delete(nlcxml);
		return NLCE_UNK_ERR;
	}
	if(res != NLCE_OK)
	{
		free(buf);
		free(statbuf);
		nlc_xmlparser_delete(nlcxml);
		return res;
	}

	#ifdef NLC_DEBUG
	printf("nlc: api response is '%s'\n", statbuf);
	#endif

	if(!stricmp(statbuf, "ok"))
	{
		// 'status' was "ok"
	}
	else if(!stricmp(statbuf, "fail"))
	{
		// 'status' was "fail"
		// parse full error info
		nlc_xmlparser_parse(nlcxml, &codebuf, "/nicovideo_user_response/error/code");
		errcode = atoi(codebuf);
		if(errcode == 0)
		{
			*api_err = NLCE_API_UNK_ERR;

			free(buf);
			free(statbuf);
			free(codebuf);
			nlc_xmlparser_delete(nlcxml);
			return NLCE_API_ERR;
		}

		// switch by <code>[errcode]</code>
		switch(errcode)
		{
			case 2:
				*api_err = NLCE_API_ACCOUNT_DATA_INCORRECT;
				break;

			case 3:
				*api_err = NLCE_API_ACCOUNT_LOCKED;
				break;

			default:
				*api_err = NLCE_API_UNK_ERR;
				break;
		}

		free(buf);
		free(statbuf);
		free(codebuf);
		nlc_xmlparser_delete(nlcxml);
		return NLCE_API_ERR;
	}
	else
	{
		// attribute 'status' was unknown value
		*api_err = NLCE_API_UNK_ERR;

		free(buf);
		free(statbuf);
		nlc_xmlparser_delete(nlcxml);
		return NLCE_API_ERR;
	}
	free(statbuf);

	// no error
	// parse the node 'ticket'
	res = nlc_xmlparser_parse(nlcxml, &adata->ticket, "/nicovideo_user_response/ticket");
	if(res != NLCE_OK)
	{
		#ifndef NLC_SILENT
		fprintf(stderr, "nlc: failed to parse 'ticket'\n");
		#endif
		
		free(buf);
		nlc_xmlparser_delete(nlcxml);
		return res;
	}
	free(buf);

	#ifdef NLC_DEBUG
	printf("nlc: 'ticket' is '%s'\n", adata->ticket);
	#endif

	nlc_xmlparser_delete(nlcxml);
	return NLCE_OK;
}



NLCResult nlc_live_getalertinfo(NLCAlertData *adata, NLCResult *api_err)
{
	char *buf = NULL;
	char **table;
	int i = 0;
	NLCResult res = NLCE_OK;
	NLCXMLParser *nlcxml = NULL;

	res = nlc_curl_http_get(&buf, "http://live.nicovideo.jp/api/getalertinfo", NULL);
	if(res != NLCE_OK)
	{
		#ifndef NLC_SILENT
		fprintf(stderr, "nlc: %s\n", nlc_err_tostr(res));
		#endif
		return res;
	}
	if(!buf)
	{
		#ifndef NLC_SILENT
		fprintf(stderr, "nlc: Bad buffer\n");
		#endif
		return NLCE_UNK_ERR;
	}

	#ifdef NLC_DEBUG
	printf("getalertinfo = \n[%s]\n", buf);
	#endif
	

	nlcxml = nlc_xmlparser_new(buf, strlen(buf), NULL);
	if(!nlcxml)
	{
		free(buf);
		return NLCE_XMLPARSER_ERR;
	}

	res = nlc_xmlparser_parse_va(nlcxml, &table, 3, "/getalertstatus/ms/addr", "/getalertstatus/ms/port", "/getalertstatus/ms/thread");
	if(res != NLCE_OK)
	{
		#ifndef NLC_SILENT
		fprintf(stderr, "nlc: failed to parse getalertstatus\n");
		#endif
		
		free(buf);
		nlc_xmlparser_delete(nlcxml);
		return res;
	}

	// Note: this is not an NLCE_NO_MEMORY error,
	// since nlc_xmlparser will return NLCE_MO_MEMORY if so.
	if(!table)
	{
		// unknown error. cool.

		#ifndef NLC_SILENT
		fprintf(stderr, "nlc: unknown error occured during XML parse!\n");
		#endif

		free(buf);
		nlc_xmlparser_delete(nlcxml);
		return NLCE_UNK_ERR;
	}

	// free the XML data
	free(buf);

	// copy the data into char* array
	if(table[0]) strcpy(adata->addr, table[0]);
	if(table[1]) adata->port = atoi(table[1]);
	if(table[2]) adata->thread = atoi(table[2]);


	// free the parser object
	nlc_xmlparser_delete(nlcxml);
	return NLCE_OK;
}
 

NLCResult nlc_live_getalertstatus(NLCAlertData *adata, NLCResult *api_err)
{
	char *query = NULL, *buf = NULL;
	char **table;
	int i = 0;
	NLCXMLParser *nlcxml = NULL;
	NLCResult res = NLCE_OK;

	if(!adata)
	{
		return NLCE_BAD_FUNCTION_ARGUMENT;
	}

	if(!adata->ticket)
	{
		fprintf(stderr, "nlc: obtain ticket first!\n");
		return  NLCE_BAD_FUNCTION_ARGUMENT;
	}


	query = (char *)malloc(strlen("ticket=")+strlen(adata->ticket)+1);
	sprintf(query, "ticket=%s", adata->ticket);

	res = nlc_curl_http_post(&buf, "http://live.nicovideo.jp/api/getalertstatus", query);
	free(query);

	if(res != NLCE_OK)
	{
		fprintf(stderr, "nlc: %s\n", nlc_err_tostr(res));
		return res;
	}
	if(!buf)
	{
		fprintf(stderr, "nlc: Bad buffer\n");
		return NLCE_UNK_ERR;
	}

	#ifdef NLC_DEBUG
	printf("getalertstatus = \n[%s]\n", buf);
	#endif


	// new NLCXMLParser object
	nlcxml = nlc_xmlparser_new(buf, strlen(buf), NULL);
	if(!nlcxml)
	{
		free(buf);
		return NLCE_XMLPARSER_ERR;
	}

	// parse Alert Server informations
	res = nlc_xmlparser_parse_va(nlcxml, &table, 3, "/getalertstatus/ms/addr", "/getalertstatus/ms/port", "/getalertstatus/ms/thread");
	if(res != NLCE_OK)
	{
		#ifndef NLC_SILENT
		fprintf(stderr, "nlc: failed to parse getalertstatus\n");
		#endif
		
		free(buf);
		nlc_xmlparser_delete(nlcxml);
		return res;
	}

	// Note: this is not an NLCE_NO_MEMORY error,
	// since nlc_xmlparser will return NLCE_MO_MEMORY if so.
	if(!table)
	{
		// unknown error. cool.

		#ifndef NLC_SILENT
		fprintf(stderr, "nlc: unknown error occured during XML parse!\n");
		#endif

		free(buf);
		nlc_xmlparser_delete(nlcxml);
		return NLCE_UNK_ERR;
	}

	// free the XML data
	free(buf);

	// copy the data into char* array
	if(table[0]) strcpy(adata->addr, table[0]);
	if(table[1]) adata->port = atoi(table[1]);
	if(table[2]) adata->thread = atoi(table[2]);


	// free the parser object
	nlc_xmlparser_delete(nlcxml);
	return NLCE_OK;
}


// ====================================================
// = Live stream API
// ====================================================





