﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Diagnostics;
using System.Net;
using CommonSystem;
namespace Nicolib.Cookie.BrowserCookie {
	abstract class CookieBase : IDisposable{
		/// <summary>
		/// プロセス名とCookieの場所の対応表
		/// </summary>
		protected List<ProcessInfo> ProcessList = new List<ProcessInfo>();

		/// <summary>
		/// データベースの一時コピーファイルパス
		/// </summary>
		protected string DbTempFile {
			get {
				return Path.Combine( Path.GetTempPath() , "NicoLibCookie.db" );
			}
		}
		/// <summary>
		/// ブラウザからCookieを取得します。
		/// </summary>
		/// <returns></returns>
		protected abstract string CookieGet(string path);
		/// <summary>
		/// ブラウザからニコニコのセッションを取得してPITAに関連付けします。
		/// </summary>
		/// <returns>セッション文字列</returns>
		public bool SessionGet() {
			string session = null;
			List<string> cookiePathList = CheckProcess();
			foreach(string path in cookiePathList) {
				if( File.Exists( path ) ) {
					try {
						File.Copy( path , DbTempFile , true );
						if(File.Exists( DbTempFile ) == false) {
							continue;
						}
					} catch(Exception) {
						session = null;
					}
				} else if( Directory.Exists( path ) == false ) {
					continue;
				}
				try {
					session = CookieGet( path );
					if(session != null && Session.SessionSet( session )) {
						break;	// 正しいセッションだった。
					} else {
						session = null;
					}
				} catch(Exception) {
					session = null;
				}

			}
			if(File.Exists( DbTempFile )) {
				File.Delete( DbTempFile );
			}
			return ( session != null );
		}

		/// <summary>
		/// 起動中のプロセスを判断して有効なCookieの場所候補を返します。
		/// </summary>
		/// <returns></returns>
		List<string> CheckProcess() {
			List<string> cookiePathList = new List<string>();
			foreach(ProcessInfo Info in ProcessList) {
				if(Info.CheckBoot == false) {
					cookiePathList.AddEx( Info.CookieLocation );
				} else {
					Process[] Proc = Process.GetProcessesByName( Info.Name );
					for(int i = 0 ; Proc.Length > i ; i++) {
						if( Info.CookieLocation != null ){
							if(Path.IsPathRooted( Info.CookieLocation ) == false) {
								DirectoryInfo DirInfo = Directory.GetParent( Proc[i].MainModule.FileName );
								if( DirInfo.FullName != null ){
									cookiePathList.AddEx( Path.Combine( DirInfo.FullName , Info.CookieLocation ) );
								}
							} else {
								cookiePathList.AddEx( Info.CookieLocation );
							}
						}
					}
				}
			}
			return ( cookiePathList );
		}

		/// <summary>
		/// プロセスとCookieの場所情報クラス
		/// </summary>
		protected class ProcessInfo {

			/// <summary>
			/// プロセス名とCookieの場所を登録します。
			/// </summary>
			/// <param field="Name">プロセス名</param>
			/// <param field="CheckBoot">true = プロセスが起動しているかチェックする</param>
			/// <param field="CookieLocation">Cookieの場所。相対パスならプロセスから見た場所になります。</param>
			public ProcessInfo(string Name , bool CheckBoot , string CookieLocation) {
				this.Name = Name;
				this.CheckBoot = CheckBoot;
				this.CookieLocation = CookieLocation;
			}

			/// <summary>
			/// プロセス名
			/// </summary>
			public string Name {
				get;
				private set;
			}
			/// <summary>
			/// プロセスの起動状態を検査するか
			/// </summary>
			public bool CheckBoot {
				get;
				private set;
			}
			/// <summary>
			/// Cookieの場所
			/// </summary>
			public string CookieLocation {
				get;
				private set;
			}
		}

		#region IDisposable メンバ
		
		/// <summary>
		/// 破棄します。
		/// </summary>
		public void Dispose() {
			GC.SuppressFinalize(this);
		}
		
		/// <summary>
		/// デストラクタ
		/// </summary>
		~CookieBase(){
			Dispose();
		}

		#endregion
	}
}
