package nicolib.api;

import java.util.Date;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import nicolib.NicoText;
import nicolib.util.Logger;

/**
 * APIの基底クラス
 * APIの取得は具象クラスのクラスメソッドgetおよびgetAsyncにより行う
 * 値の取り出しはselect系メソッドか具象クラスのget系メソッドにより行う
 * @author hal
 *
 */
public abstract class ApiBase {
	
	/**
	 * APIの取得試行回数
	 */
	protected static final int retryCount = 3;
	
	/**
	 * API非同期取得用のスレッドプール
	 */
	private static ExecutorService executor;
	protected static synchronized ExecutorService getExecutor(){
		if(executor == null) executor = Executors.newCachedThreadPool();
		return executor;
	}
	
	/**
	 * 指定したURLのAPIへアクセスする
	 * 失敗時には既定では３回まで接続を試みる
	 * @param url
	 * @throws NicoApiException
	 */
	protected abstract void access(String url) throws NicoApiException;
	
	/**
	 * 指定したURLのAPIへPOSTでアクセスする
	 * 失敗時には既定では３回まで接続を試みる
	 * @param url
	 * @param param
	 * @param referer
	 * @throws NicoApiException
	 */
	protected abstract void access(String url, String param, String referer) throws NicoApiException;
	
	/**
	 * 指定したURLのAPIへ非同期でアクセスする
	 * @param url
	 * @param handler
	 */
	protected void accessAsync(final String url, final AsyncLoadHandler handler){
		
		final ApiBase me = this;
		
		getExecutor().submit(new Runnable() {
			@Override
			public void run() {
				try{
					access(url);
					handler.onLoaded(me);
				}catch(NicoApiException e){
					handler.onError(e);
				}
			}
		});
	}
	
	/**
	 * 指定したURLのAPIへPOSTで非同期アクセスする
	 * @param url
	 * @param param
	 * @param referer
	 * @param handler
	 * @throws NicoApiException
	 */
	protected void accessAsync(final String url, final String param, final String referer, final AsyncLoadHandler handler){
		final ApiBase me = this;
		
		getExecutor().submit(new Runnable() {
			@Override
			public void run() {
				try{
					access(url, param, referer);
					handler.onLoaded(me);
				}catch(NicoApiException e){
					handler.onError(e);
				}
			}
		});
	}
	
	
	/**
	 * 指定された位置にある文字列データを取得する
	 * @param location
	 * @return
	 */
	public abstract String select(String location);
	
	/**
	 * 指定された位置にある整数データを取得する
	 * @param location
	 * @param defaultValue 
	 * @return 数値データが見つからなかった時はdefaultValueで指定した値
	 */
	public int selectInt(String location, int defaultValue) {
		String x = select(location);
		try{
			if(x != null){
				return Integer.parseInt(x);
			}
		}catch(NumberFormatException e){
			Logger.writeException(e);
		}
		
		return defaultValue;
	}
	
	/**
	 * 指定された位置にある長整数データを取得する
	 * @param location
	 * @param defaultValue 
	 * @return 数値データが見つからなかった時はdefaultValueで指定した値
	 */
	public long selectLong(String location, long defaultValue) {
		String x = select(location);
		try{
			if(x != null){
				return Long.parseLong(x);
			}
		}catch(NumberFormatException e){
			Logger.writeException(e);
		}
		
		return defaultValue;
	}
	
	/**
	 * 指定された位置にある日付データを取得する
	 * @param location
	 * @return データが見つからなかった場合は1970年くらいの日付が変える
	 */
	public Date selectDate(String location) {
		long x = selectInt(location, 0);
		return new Date(x * 1000);
	}
	
	/**
	 * Url形式やコメントなどからLiveIdを抽出する
	 * @param id
	 * @return
	 * @throws UncompatibleIdException IDが見つからなかった場合に発生します
	 */
	protected static String checkAndGetLiveId(String id) throws UncompatibleIdException
	{
		String lid = NicoText.getLiveId(id);
		
		if(lid == null){
			throw new UncompatibleIdException("指定されたIDは認識できませんでした");
		}
		
		return lid;
	}
	
}
