package jp.sourceforge.nicoro;

import org.json.JSONException;
import org.json.JSONObject;

import android.os.Parcel;
import android.os.Parcelable;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static jp.sourceforge.nicoro.Log.LOG_TAG;

/**
 * 動画ページ(watch)に含まれるJavaScriptのVideo変数
 */
public class WatchVideo implements Parcelable {
    private static final boolean DEBUG_LOGD = Release.IS_DEBUG & true;

    private HashMap<String, String> mVideo;
    private String mUrl;

    private String v = null;
    private String id = null;
    private String title = null;
    private String description = null;
    private String thumbnail = null;
    private Date postedAt = null;
    private int length = -1;
    private int viewCount = -1;
    private int mylistCount = -1;

    private DateFormat mDateFormat = new SimpleDateFormat(
            "yyyy/MM/dd HH:mm:ss");

    /**
     * Videoを読み込んだMapから生成
     * @param video
     */
    public WatchVideo(Map<String, String> video) {
        mVideo = new HashMap<String, String>(video);
    }

    /**
     * ThumbInfoから擬似的に生成
     * @param thumbInfo
     */
    public WatchVideo(ThumbInfo thumbInfo) {
        mVideo = new HashMap<String, String>();
        mVideo.put("v", thumbInfo.getVideoNumber());
        mVideo.put("id", thumbInfo.getVideoNumber());
        mVideo.put("title", thumbInfo.getTitle());
        mVideo.put("description", thumbInfo.getDescription());
        mVideo.put("thumbnail", thumbInfo.getThumbnailUrl());
        mVideo.put("postedAt", mDateFormat.format(thumbInfo.getFirstRetrieveAsDate()));
        mVideo.put("length", thumbInfo.getLength());
        mVideo.put("viewCount", Integer.toString(thumbInfo.getViewCounter()));
        mVideo.put("mylistCount", Integer.toString(thumbInfo.getMylistCounter()));
    }

    public String v() {
        if (v == null) {
            v = v(mVideo);
        }
        return v;
    }
    public static String v(Map<String, String> video) {
        return video.get("v");
    }

    public String id() {
        if (id == null) {
            id = id(mVideo);
        }
        return id;
    }
    public static String id(Map<String, String> video) {
        return video.get("id");
    }

    public String title() {
        if (title == null) {
            title = title(mVideo);
        }
        return title;
    }
    public static String title(Map<String, String> video) {
        return video.get("title");
    }

    public String description() {
        if (description == null) {
            description = description(mVideo);
        }
        return description;
    }
    public static String description(Map<String, String> video) {
        return video.get("description");
    }

    public String thumbnail() {
        if (thumbnail == null) {
            thumbnail = thumbnail(mVideo);
        }
        return thumbnail;
    }
    public static String thumbnail(Map<String, String> video) {
        return video.get("thumbnail");
    }

    public Date postedAt() {
        if (postedAt == null) {
            try {
                postedAt = mDateFormat.parse(postedAt(mVideo));
            } catch (ParseException e) {
                Log.w(LOG_TAG, e.toString(), e);
            }
        }
        return postedAt;
    }
    public static String postedAt(Map<String, String> video) {
        return video.get("postedAt");
    }

    public int length() {
        if (length < 0) {
            length = length(mVideo);
        }
        return length;
    }
    public static int length(Map<String, String> video) {
        return Util.parseInt(video.get("length"), -1);
    }

    public int viewCount() {
        if (viewCount < 0) {
            viewCount = viewCount(mVideo);
        }
        return viewCount;
    }
    public static int viewCount(Map<String, String> video) {
        return Util.parseInt(video.get("viewCount"), -1);
    }

    public int mylistCount() {
        if (mylistCount < 0) {
            mylistCount = mylistCount(mVideo);
        }
        return mylistCount;
    }
    public static int mylistCount(Map<String, String> video) {
        return Util.parseInt(video.get("mylistCount"), -1);
    }

    /**
     * vまたはidと一致するか判定
     * @param videoNumber
     * @return
     */
    public boolean hasVideoNumber(String videoNumber) {
        if (videoNumber == null) {
            return false;
        }
        return videoNumber.equals(v()) || videoNumber.equals(id());
    }

    /**
     * 最低限の正しい値を持っているか確認
     * @return
     */
    public boolean isValid() {
        return (v() != null && id() != null);
    }

    @Override
    public String toString() {
        return "WatchVideo: " + mVideo.toString();
    }

    public void setUrl(String url) {
        mUrl = url;
    }
    public String getUrl() {
        return mUrl;
    }

    public static WatchVideo createFromWatchHtml(String html) {
        try {
            Matcher matcher = Pattern.compile("var\\s+Video\\s*=\\s*(\\{.*\\})",
                    Pattern.DOTALL | Pattern.MULTILINE).matcher(html);
            if (matcher.find()) {
                JSONObject json = new JSONObject(matcher.group(1));
                HashMap<String, String> map = new HashMap<String, String>(json.length());
                Iterator<?> ite = json.keys();
                while (ite.hasNext()) {
                    String key = (String) ite.next();
                    String value = json.optString(key, null);
                    map.put(key, value);
                    if (DEBUG_LOGD) {
                        Log.d(LOG_TAG, Log.buf().append("Video.").append(key)
                                .append('=').append(value).toString());
                    }
                }
                return new WatchVideo(map);
            } else {
                if (DEBUG_LOGD) {
                    Log.d(LOG_TAG, "var Video is not found");
                }
            }
        } catch (JSONException e) {
            Log.w(LOG_TAG, e.toString(), e);
        }
        if (DEBUG_LOGD) {
            Log.dLong(LOG_TAG, html);
        }
        return null;
    }

    /**
     * 動画番号だけから仮生成
     * @param videoNumber
     * @return
     */
    public static WatchVideo createTemp(String videoNumber) {
        HashMap<String, String> map = new HashMap<String, String>();
        map.put("v", videoNumber);
        map.put("id", videoNumber);
        return new WatchVideo(map);
    }


    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeSerializable(mVideo);
        dest.writeString(mUrl);
    }

    public static final Parcelable.Creator<WatchVideo> CREATOR = new Creator<WatchVideo>() {
        @Override
        public WatchVideo createFromParcel(Parcel source) {
            return new WatchVideo(source);
        }

        @Override
        public WatchVideo[] newArray(int size) {
            return new WatchVideo[size];
        }
    };

    @SuppressWarnings("unchecked")
    WatchVideo(Parcel p) {
        mVideo = (HashMap<String, String>) p.readSerializable();
        mUrl = p.readString();
    }
}
