/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import java.util.Locale;
import java.io.Serializable;

import jp.ossc.nimbus.service.message.MessageRecord;
import jp.ossc.nimbus.service.message.MessageRecordFactory;

/**
 * {@link MessageRecordFactory}bpB<p>
 *
 * @author M.Takata
 */
class MessageRecordFactoryWrapper
 implements MessageRecordFactory, ServiceStateListener, Serializable{
    
    private static final long serialVersionUID = 8434021957805850115L;
    
    private MessageRecordFactory defaultMessage;
    private MessageRecordFactory currentMessage;
    
    public MessageRecordFactoryWrapper(MessageRecordFactory defaultMessage){
        this(null, null, defaultMessage);
    }
    
    public MessageRecordFactoryWrapper(
        MessageRecordFactory message,
        Service messageService,
        MessageRecordFactory defaultMessage
    ){
        setDefaultMessageRecordFactory(defaultMessage);
        setMessageRecordFactory(message, messageService);
    }
    
    public void setDefaultMessageRecordFactory(MessageRecordFactory message){
        if(message == this){
            return;
        }
        this.defaultMessage = message;
    }
    
    public MessageRecordFactory getDefaultMessageRecordFactory(){
        return defaultMessage;
    }
    
    public void setMessageRecordFactory(MessageRecordFactory message){
        setMessageRecordFactory(message, null);
    }
    
    public void setMessageRecordFactory(
        MessageRecordFactory message,
        Service messageService
    ){
        if(message == this){
            return;
        }
        if(messageService != null){
            if(messageService.getState() == Service.State.STARTED){
                currentMessage = message;
            }else{
                currentMessage = defaultMessage;
            }
            try{
                final ServiceStateListenable listenable
                    = ServiceManagerFactory.getServiceStateListenable(
                        messageService.getServiceManagerName(),
                        messageService.getServiceName()
                    );
                if(listenable != null){
                    listenable.addServiceStateListener(this);
                }
            }catch(ServiceNotFoundException e){
            }
        }else{
            if(message != null){
                currentMessage = message;
            }else{
                currentMessage = defaultMessage;
            }
        }
    }
    
    public MessageRecordFactory getMessageRecordFactory(){
        return currentMessage;
    }
    
    @Override
    public void stateChanged(ServiceStateChangeEvent e) throws Exception{
        final Service service = e.getService();
        final Service.State state = service.getState();
        final String managerName = service.getServiceManagerName();
        final String serviceName = service.getServiceName();
        switch(state){
        case STARTED:
            currentMessage = (MessageRecordFactory)ServiceManagerFactory
            .getServiceObject(
                managerName,
                serviceName
            );
            break;
        case STOPPED:
            currentMessage = defaultMessage;
            break;
        default:
        }
        
    }
    
    @Override
    public boolean isEnabledState(Service.State state){
        switch(state){
        case STARTED:
        case STOPPED:
            return true;
        default:
            return false;
        }
    }
    
    // MessageRecordFactoryJavaDoc
    @Override
    public MessageRecord findMessageRecord(String key){
        if(currentMessage != null){
            return currentMessage.findMessageRecord(key);
        }
        return null;
    }
    
    // MessageRecordFactoryJavaDoc
    @Override
    public String findMessageTemplete(String key){
        if(currentMessage != null){
            return currentMessage.findMessageTemplete(key);
        }
        return null;
    }
    
    // MessageRecordFactoryJavaDoc
    @Override
    public String findMessageTemplete(String key, Locale lo){
        if(currentMessage != null){
            return currentMessage.findMessageTemplete(key, lo);
        }
        return null;
    }
    
    // MessageRecordFactoryJavaDoc
    @Override
    public String findMessage(String key, Object... embeds){
        if(currentMessage != null){
            return currentMessage.findMessage(key, embeds);
        }
        return null;
    }
    
    // MessageRecordFactoryJavaDoc
    @Override
    public String findMessage(String key, Locale lo, Object... embeds){
        if(currentMessage != null){
            return currentMessage.findMessage(key, lo, embeds);
        }
        return null;
    }
    
    // MessageRecordFactoryJavaDoc
    @Override
    public void findLocale(Locale lo){
        if(currentMessage != null){
            currentMessage.findLocale(lo);
        }
    }
}
