/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import java.util.*;
import java.net.*;
import java.io.*;
import java.security.*;
import java.security.cert.Certificate;

/**
 * NimbusNX[_B<p>
 * Nimbus̃T[rXNX̃[hsNX[_łB<br>
 * ̃NX[_̃CX^X́A{@link NimbusClassLoader#getInstance()}Ŏ擾邱ƂłB̃CX^X̓XbhReLXgNX[_PʂɐBꂽCX^X́AXbhReLXgNX[_L[ɎQƂŕێ邽߁AXbhReLXgNX[_jƎIɔjB<br>
 * <p>
 * ܂ÃNX[_ɂ́A{@link AspectTranslator}o^鎖\łBAspectTranslatoro^ꂽԂŁANX̃[h˗ƁAAspectTranslatorɂNXt@C̕ϊsāANX[hB<br>
 * AANX̃[hŝ́ANXt@C̕ϊΏۂɂȂĂNXŁA[hꂽNXŎQƂʂ̃NX́ANXt@C̕ϊΏۂłȂ΁ÃNX[_Ń[hsȂB<br>
 * ́AAvP[VT[oȂǂ̕GȃNX[_\Reiɑ΂złBX^hA[̃AvP[Vł́A[@link #setLoadNotTransformClass(boolean) setLoadNotTransformClass(true)}ĂяoŁAϊΏۊÕNX̃[hÃNX[_ōs悤ɂȂB<br>
 * 
 * @author M.Takata
 * @see AspectTranslator
 */
public class NimbusClassLoader extends ClassLoader{
    
    /**
     * XbhReLXgNX[_Ɋ֘AtāÃNX[_̃CX^XێQƃ}bvB<p>
     */
    protected static final Map<ClassLoader, NimbusClassLoader> classLoader = new WeakHashMap<ClassLoader, NimbusClassLoader>();
    
    /**
     * VMxœo^ꂽ{@link AspectTranslator}̃XgB<p>
     */
    protected static final Map<String, List<AspectTranslator>> vmTranslators = new HashMap<String, List<AspectTranslator>>();
    
    /**
     * ThreadContextxœo^ꂽ{@link AspectTranslator}̃XgB<p>
     */
    protected final Map<String, List<AspectTranslator>> translators = new HashMap<String, List<AspectTranslator>>();
    
    private static final String CLASS_EXTEND = ".class";
    private boolean isLoadNotTransformClass = false;
    
    /**
     * w肳ꂽNX[_eɎCX^X𐶐B<p>
     *
     * @param parent eNX[_
     */
    protected NimbusClassLoader(ClassLoader parent){
        super(parent);
    }
    
    /**
     * NX[_̃CX^X擾B<p>
     * ̃\bhŎ擾NX[_́AXbhReLXgNX[_eɎB<br>
     * 
     * @return XbhReLXgNX[_Ɋ֘AtꂽNimbusNX[_̃CX^X
     */
    public static synchronized NimbusClassLoader getInstance(){
        final ClassLoader contextLoader
             = Thread.currentThread().getContextClassLoader();
        if(contextLoader instanceof NimbusClassLoader){
            return (NimbusClassLoader)contextLoader;
        }
        NimbusClassLoader loader
             = (NimbusClassLoader)classLoader.get(contextLoader);
        if(loader == null){
            loader = new NimbusClassLoader(contextLoader);
            classLoader.put(contextLoader, loader);
        }
        return loader;
    }
    
    /**
     * VMxŃNX[hɃNXt@CϊsAspectTranslatoro^B<p>
     *
     * @param translator o^AspectTranslator
     */
    public static void addVMAspectTranslator(AspectTranslator translator){
        synchronized(vmTranslators){
            List<AspectTranslator> list = null;
            if(vmTranslators.containsKey(translator.getAspectKey())){
                list = vmTranslators.get(translator.getAspectKey());
            }else{
                list = new ArrayList<AspectTranslator>();
                vmTranslators.put(translator.getAspectKey(), list);
            }
            if(!list.contains(translator)){
                list.add(translator);
            }
        }
    }
    
    /**
     * VMxŃNX[hɃNXt@CϊsAspectTranslatoro^B<p>
     *
     * @param translator o^AspectTranslator
     */
    public static void removeVMAspectTranslator(AspectTranslator translator){
        synchronized(vmTranslators){
            if(vmTranslators.containsKey(translator.getAspectKey())){
                final List<AspectTranslator> list = vmTranslators
                    .get(translator.getAspectKey());
                list.remove(translator);
                if(list.size() == 0){
                    vmTranslators.remove(translator.getAspectKey());
                }
            }
        }
    }
    
    /**
     * VMxœo^ĂAspectTranslator擾B<p>
     *
     * @return AspectTranslator̔z
     */
    @SuppressWarnings("unchecked")
    public static AspectTranslator[] getVMAspectTranslators(){
        synchronized(vmTranslators){
            final AspectTranslator[] result
                 = new AspectTranslator[vmTranslators.size()];
            final List<AspectTranslator>[] lists = (List<AspectTranslator>[])vmTranslators.values()
                .toArray(new List[vmTranslators.size()]);
            for(int i = 0; i < lists.length; i++){
                result[i] = (AspectTranslator)lists[i].get(0);
            }
            return result;
        }
    }
    
    /**
     * ThreadContextxŃNX[hɃNXt@CϊsAspectTranslatoro^B<p>
     *
     * @param translator o^AspectTranslator
     */
    public void addAspectTranslator(AspectTranslator translator){
        synchronized(translators){
            List<AspectTranslator> list = null;
            if(translators.containsKey(translator.getAspectKey())){
                list = translators.get(translator.getAspectKey());
            }else{
                list = new ArrayList<AspectTranslator>();
                translators.put(translator.getAspectKey(), list);
            }
            if(!list.contains(translator)){
                list.add(translator);
            }
        }
    }
    
    /**
     * ThreadContextxŃNX[hɃNXt@CϊsAspectTranslatoro^B<p>
     *
     * @param translator o^AspectTranslator
     */
    public void removeAspectTranslator(AspectTranslator translator){
        synchronized(translators){
            if(translators.containsKey(translator.getAspectKey())){
                final List<AspectTranslator> list = translators
                    .get(translator.getAspectKey());
                list.remove(translator);
                if(list.size() == 0){
                    translators.remove(translator.getAspectKey());
                }
            }
        }
    }
    
    /**
     * ThreadContextxœo^ĂAspectTranslator擾B<p>
     *
     * @return AspectTranslator̔z
     */
    @SuppressWarnings("unchecked")
    public AspectTranslator[] getAspectTranslators(){
        synchronized(translators){
            final AspectTranslator[] result
                 = new AspectTranslator[translators.size()];
            final List<AspectTranslator>[] lists = (List<AspectTranslator>[])translators.values()
                .toArray(new List[translators.size()]);
            for(int i = 0; i < lists.length; i++){
                result[i] = (AspectTranslator)lists[i].get(0);
            }
            return result;
        }
    }
    
    /**
     * {@link AspectTranslator}̕ϊΏۂłȂNX[h邩ǂݒ肷B<p>
     *
     * @param isLoad {@link AspectTranslator}̕ϊΏۂłȂNX[hꍇtrueBftHgfalse
     */
    public void setLoadNotTransformClass(boolean isLoad){
        isLoadNotTransformClass = isLoad;
    }
    
    /**
     * {@link AspectTranslator}̕ϊΏۂłȂNX[h邩ǂ𔻒肷B<p>
     *
     * @return truȅꍇ{@link AspectTranslator}̕ϊΏۂłȂNX[h
     */
    public boolean isLoadNotTransformClass(){
        return isLoadNotTransformClass;
    }
    
    /**
     * w肳ꂽNX̃NX[_Ń[hB<p>
     * {@link #isLoadNotTransformClass()}̒lɊւ炸AϊΏۂłȂNX̃NX[_ŖIɃ[hB<br>
     * 
     * @param name NX
     * @return [hNXIuWFNg
     * @exception ClassNotFoundException w肳ꂽNXȂꍇ
     */
    public synchronized Class<?> loadClassLocally(String name)
     throws ClassNotFoundException{
        return loadClass(name, false);
    }
    
    /**
     * w肳ꂽNX̃NX[_Ń[hB<p>
     * {@link #isLoadNotTransformClass()}̒lɊւ炸AϊΏۂłȂNX̃NX[_ŖIɃ[hB<br>
     * 
     * @param name NX
     * @param resolve true ̏ꍇ́ANXߏ
     * @return [hNXIuWFNg
     * @exception ClassNotFoundException w肳ꂽNXȂꍇ
     */
    public synchronized Class<?> loadClassLocally(String name, boolean resolve)
     throws ClassNotFoundException{
        return loadClass(name, resolve, true);
    }
    
    /**
     * w肳ꂽNX[hB<p>
     * o^ꂽ{@link AspectTranslator}̕ϊΏۂƂȂĂNX́ÃNX[_Ń[hBłȂNX́AeNX[_łXbhReLXg[_ɈϏBAA{@link #isLoadNotTransformClass()}̒ltruȅꍇ́AϊΏۂłȂNX̃NX[_Ń[hB<br>
     * 
     * @param name NX
     * @param resolve true ̏ꍇ́ANXߏ
     * @return [hNXIuWFNg
     * @exception ClassNotFoundException w肳ꂽNXȂꍇ
     */
    @Override
    protected synchronized Class<?> loadClass(String name, boolean resolve)
     throws ClassNotFoundException{
        return loadClass(name, resolve, false);
    }
    
    /**
     * w肳ꂽNX[hB<p>
     * 
     * @param name NX
     * @param resolve true ̏ꍇ́ANXߏ
     * @param isLocally truȅꍇ́A{@link #isLoadNotTransformClass()}̒lɊւ炸AϊΏۂłȂNX̃NX[_ŖIɃ[h
     * @return [hNXIuWFNg
     * @exception ClassNotFoundException w肳ꂽNXȂꍇ
     */
    protected synchronized Class<?> loadClass(
        String name,
        boolean resolve,
        boolean isLocally
    ) throws ClassNotFoundException{
        if(vmTranslators.size() == 0 && translators.size() == 0
             && !isLoadNotTransformClass){
            return super.loadClass(name, resolve);
        }
        if(isNonLoadableClassName(name)){
            return super.loadClass(name, resolve);
        }
        final boolean isNonTranslatableClass = isNonTranslatableClassName(name);
        if(isNonTranslatableClass && !isLoadNotTransformClass){
            return super.loadClass(name, resolve);
        }
        final Class<?> loadedClass = findLoadedClass(name);
        if(loadedClass != null){
            return loadedClass;
        }
        
        final URL classUrl = getClassURL(name);
        if(classUrl == null){
            return super.loadClass(name, resolve);
        }
        final byte[] bytecode = loadByteCode(classUrl);
        if(bytecode == null){
            return super.loadClass(name, resolve);
        }
        final URL codeSourceUrl = getCodeSourceURL(name, classUrl);
        if(codeSourceUrl == null){
            return super.loadClass(name, resolve);
        }
        final ProtectionDomain domain = getProtectionDomain(codeSourceUrl);
        boolean isTransform = false;
        byte[] transformedBytes = bytecode;
        if(!isNonTranslatableClass){
            synchronized(vmTranslators){
                final Object[] keys = vmTranslators.keySet().toArray();
                for(Object key : keys){
                    final AspectTranslator translator
                        = vmTranslators.get(key).get(0);
                    final byte[] tmpBytes = translator.transform(
                        this,
                        name,
                        domain,
                        transformedBytes
                    );
                    if(tmpBytes != null){
                        isTransform = true;
                        transformedBytes = tmpBytes;
                    }
                }
            }
            synchronized(translators){
                final Object[] keys = translators.keySet().toArray();
                for(Object key : keys){
                    final AspectTranslator translator
                        = translators.get(key).get(0);
                    final byte[] tmpBytes = translator.transform(
                        this,
                        name,
                        domain,
                        transformedBytes
                    );
                    if(tmpBytes != null){
                        isTransform = true;
                        transformedBytes = tmpBytes;
                    }
                }
            }
        }
        if(isTransform || isLocally || isLoadNotTransformClass){
            definePackage(name);
            final Class<?> clazz = defineClass(
                name,
                transformedBytes,
                0,
                transformedBytes.length,
                domain
            );
            if(resolve){
                resolveClass(clazz);
            }
            return clazz;
        }else{
            int innerClassIndex = name.lastIndexOf('$');
            if(innerClassIndex != -1
                && name.length() - 1 != innerClassIndex
                && findLoadedClass(name.substring(0, innerClassIndex)) != null
            ){
                return loadClass(name, resolve, true);
            }
            return super.loadClass(name, resolve);
        }
    }
    
    /**
     * [hΏۂƂȂȂNX𔻒肷B<p>
     * ȉ̃NX́A@ȂꍇϊΏۂƂȂȂB<br>
     * <ul>
     *   <li>"org.omg."n܂NX</li>
     *   <li>"org.w3c."n܂NX</li>
     *   <li>"org.xml.sax."n܂NX</li>
     *   <li>"sunw."n܂NX</li>
     *   <li>"sun."n܂NX</li>
     *   <li>"java."n܂NX</li>
     *   <li>"javax."n܂NX</li>
     *   <li>"com.sun."n܂NX</li>
     *   <li>"javassist."n܂NX</li>
     * </ul>
     * 
     * @param classname NX
     * @return [hΏۂƂȂȂNX̏ꍇAtrue
     */
    public static boolean isNonLoadableClassName(String classname){
      return classname.startsWith("org.omg.")
              || classname.startsWith("org.w3c.")
              || classname.startsWith("org.xml.sax.")
              || classname.startsWith("sunw.")
              || classname.startsWith("sun.")
              || classname.startsWith("java.")
              || classname.startsWith("javax.")
              || classname.startsWith("com.sun.")
              || classname.equals("jp.ossc.nimbus.core.NimbusClassLoader")
              || classname.equals("jp.ossc.nimbus.core.AspectTranslator");
    }
    
    /**
     * ϊΏۂƂȂȂNX𔻒肷B<p>
     * ȉ̃NX́A@ȂꍇϊΏۂƂȂȂB<br>
     * <ul>
     *   <li>"jp.ossc.nimbus.service.aop."n܂NX</li>
     * </ul>
     * 
     * @param classname NX
     * @return ϊΏۂƂȂȂNX̏ꍇAtrue
     */
    public static boolean isNonTranslatableClassName(String classname){
      return classname.startsWith("jp.ossc.nimbus.service.aop.");
    }
    
    /**
     * w肳ꂽNX̃pbP[W`B<p>
     * pbP[Wɑ݂ꍇɂ́AȂB
     *
     * @param className NX
     */
    protected void definePackage(String className){
        int index = className.lastIndexOf('.');
        if(index == -1){
            return;
        }
        try{
            definePackage(
                className.substring(0, index),
                null,
                null,
                null,
                null,
                null,
                null,
                null
            );
        }catch(IllegalArgumentException alreadyDone){
        }
    }
    
    /**
     * w肳ꂽNX̃NXt@CURL擾B<p>
     *
     * @param classname NX
     * @return NXt@CURL
     */
    protected URL getClassURL(String classname){
        final String classRsrcName = classname.replace('.', '/') + CLASS_EXTEND;
        return getResource(classRsrcName);
    }
    
    /**
     * w肳ꂽURL̃NXt@C̃oCgR[h擾B<p>
     *
     * @param classURL NXt@CURL
     * @return NXt@C̃oCgR[hBǂݍ߂ȂꍇnullԂB
     */
    protected byte[] loadByteCode(URL classURL){
        byte[] bytecode = null;
        InputStream is = null;
        try{
            is = classURL.openStream();
            final ByteArrayOutputStream baos = new ByteArrayOutputStream();
            byte[] tmp = new byte[1024];
            int read = 0;
            while((read = is.read(tmp)) > 0){
                baos.write(tmp, 0, read);
            }
            bytecode = baos.toByteArray();
        }catch(IOException e){
            return null;
        }finally{
            if(is != null){
                try{
                    is.close();
                }catch(IOException e){
                }
            }
        }
        return bytecode;
    }
    
    /**
     * w肳ꂽNX̃NXt@C{@link CodeSource}URL擾B<p>
     * 
     * @param classname NX
     * @param classURL NXt@CURL
     * @return CodeSourcëʒu߂URL
     */
    protected URL getCodeSourceURL(String classname, URL classURL){
        final String classRsrcName = classname.replace('.', '/') + CLASS_EXTEND;
        String urlAsString = classURL.toString();
        final int index = urlAsString.indexOf(classRsrcName);
        if(index == -1){
            return classURL;
        }
        urlAsString = urlAsString.substring(0, index);
        try{
            return new URL(urlAsString);
        }catch(MalformedURLException e){
            return null;
        }
    }
    
    /**
     * w肳ꂽURL{@link CodeSource}ɑΉ{@link PermissionCollection}{@link ProtectionDomain}擾B<p>
     *
     * @param codesourceUrl CodeSourcëʒu߂URL
     * @return w肳ꂽURLCodeSourceɑΉPermissionCollection
     */
    protected ProtectionDomain getProtectionDomain(URL codesourceUrl){
    	Certificate[] certificates = null;
        final CodeSource cs = new CodeSource(codesourceUrl, certificates);
        final PermissionCollection permissions
             = Policy.getPolicy().getPermissions(cs);
        return new ProtectionDomain(cs, permissions);
    }
}